package api

import (
	"encoding/json"
	"net/http"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/service-common"
	"goji.io"
	"goji.io/pat"
)

// HTTPConfig configures a HTTPServer
type HTTPConfig struct {
	service_common.BaseHTTPServerConfig
}

// Load config from distconf
func (s *HTTPConfig) Load(dconf *distconf.Distconf) error {
	return s.Verify(dconf, "feed-settings")
}

// HTTPServer handles responding to API requests via HTTP
type HTTPServer struct {
	service_common.BaseHTTPServer
	DB *DB
}

func (s *HTTPServer) createHandler(name string, callback func(req *http.Request) (interface{}, error)) *service_common.JSONHandler {
	return &service_common.JSONHandler{
		Log:          s.Log,
		Stats:        s.Stats.NewSubStatSender(name),
		ItemCallback: callback,
	}
}

// SetupRoutes configures feed-settings goji routes
func (s *HTTPServer) SetupRoutes(mux *goji.Mux) {
	mux.Handle(pat.Get("/debug/health"), s.createHandler("health_check", s.healthCheck))

	// CRUD
	mux.Handle(pat.Get("/v1/settings/:entity"), s.createHandler("get_settings", s.getSettings))
	mux.Handle(pat.Post("/v1/settings/:entity"), s.createHandler("update_settings", s.updateSettings))
}

func (s *HTTPServer) healthCheck(r *http.Request) (interface{}, error) {
	if err := s.DB.CheckAWS(); err != nil {
		return nil, err
	}
	return "OK", nil
}

func (s *HTTPServer) getSettings(req *http.Request) (interface{}, error) {
	entity := pat.Param(req, "entity")
	settings, err := s.DB.GetSettings(req.Context(), entity)
	if err != nil {
		return nil, err
	}
	if settings == nil {
		settings = (&Settings{Entity: entity}).Defaults()
	}
	return settings, nil
}

func (s *HTTPServer) updateSettings(req *http.Request) (interface{}, error) {
	entity := pat.Param(req, "entity")
	var updates SettingsUpdate
	if err := json.NewDecoder(req.Body).Decode(&updates); err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "cannot parse request body"),
		}
	}
	return s.DB.UpdateSettings(req.Context(), entity, &updates)
}
