package api

import (
	"strconv"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"golang.org/x/net/context"
)

// DBConfig will configure the database
type DBConfig struct {
	SettingsTableName *distconf.Str
}

// Load the config from a distconf source
func (c *DBConfig) Load(dconf *distconf.Distconf) error {
	c.SettingsTableName = dconf.Str("feed-settings.settings_table", "")
	if c.SettingsTableName.Get() == "" {
		return errors.New("unable to find feed-settings settings table")
	}

	return nil
}

// DB holds the dynamo DB connection
type DB struct {
	Dynamo                 *dynamodb.DynamoDB
	Config                 *DBConfig
	Log                    *log.ElevatedLog
	RequireConsistentReads bool
}

// CheckAWS returns nil if a call to DescribeTable succeeds
func (d *DB) CheckAWS() error {
	_, err := d.Dynamo.DescribeTable(&dynamodb.DescribeTableInput{TableName: aws.String(d.Config.SettingsTableName.Get())})
	return err
}

// GetSettings gets settings for an entity
func (d *DB) GetSettings(ctx context.Context, entity string) (*Settings, error) {
	req, out := d.Dynamo.GetItemRequest(d.withConsistentReads(ctx, &dynamodb.GetItemInput{
		Key:       entityToKey(entity),
		TableName: aws.String(d.Config.SettingsTableName.Get()),
	}))
	if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
		return nil, err
	} else if out.Item == nil {
		return nil, nil
	}

	return attrsToSettings(out.Item)
}

// UpdateSettings updates non-nil values from the settings struct to the DB
func (d *DB) UpdateSettings(ctx context.Context, entity string, params *SettingsUpdate) (*Settings, error) {
	updateExpression := `set
		created_at = if_not_exists(created_at, :now),
		updated_at = :now`
	updateAttrs := map[string]*dynamodb.AttributeValue{":now": timeToAttr(time.Now())}

	if params != nil {
		if params.SubsCanComment != nil {
			updateExpression += `, subs_can_comment = :subs_can_comment`
			updateAttrs[":subs_can_comment"] = &dynamodb.AttributeValue{BOOL: params.SubsCanComment}
		}
		if params.FriendsCanComment != nil {
			updateExpression += `, friends_can_comment = :friends_can_comment`
			updateAttrs[":friends_can_comment"] = &dynamodb.AttributeValue{BOOL: params.FriendsCanComment}
		}
		if params.FollowersCanComment != nil {
			updateExpression += `, followers_can_comment = :followers_can_comment`
			updateAttrs[":followers_can_comment"] = &dynamodb.AttributeValue{BOOL: params.FollowersCanComment}
		}
		if params.UserDisabledComments != nil {
			updateExpression += `, user_disabled_comments = :user_disabled_comments`
			updateAttrs[":user_disabled_comments"] = &dynamodb.AttributeValue{BOOL: params.UserDisabledComments}
		}
		if params.AdminDisabledComments != nil {
			updateExpression += `, admin_disabled_comments = :admin_disabled_comments`
			updateAttrs[":admin_disabled_comments"] = &dynamodb.AttributeValue{BOOL: params.AdminDisabledComments}
		}
		if params.ChannelFeedEnabled != nil {
			updateExpression += `, channel_feed_enabled = :channel_feed_enabled`
			updateAttrs[":channel_feed_enabled"] = &dynamodb.AttributeValue{BOOL: params.ChannelFeedEnabled}
		}
	}

	req, out := d.Dynamo.UpdateItemRequest(&dynamodb.UpdateItemInput{
		TableName:                 aws.String(d.Config.SettingsTableName.Get()),
		Key:                       entityToKey(entity),
		UpdateExpression:          &updateExpression,
		ExpressionAttributeValues: updateAttrs,
		ReturnValues:              aws.String(dynamodb.ReturnValueAllNew),
	})
	if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
		return nil, err
	}

	return attrsToSettings(out.Attributes)
}

func (d *DB) withConsistentReads(ctx context.Context, in *dynamodb.GetItemInput) *dynamodb.GetItemInput {
	if d.RequireConsistentReads {
		in.ConsistentRead = aws.Bool(true)
	}
	return in
}

func entityToKey(entity string) map[string]*dynamodb.AttributeValue {
	return map[string]*dynamodb.AttributeValue{"entity": {S: aws.String(entity)}}
}

func attrsToSettings(attrs map[string]*dynamodb.AttributeValue) (*Settings, error) {
	settings := (&Settings{}).Defaults()
	if entity, ok := attrs["entity"]; ok {
		settings.Entity = *entity.S
	}
	if attr, ok := attrs["created_at"]; ok {
		createdAt, err := attrToTime(attr)
		if err != nil {
			return nil, err
		}
		settings.CreatedAt = *createdAt
	}
	if attr, ok := attrs["updated_at"]; ok {
		updatedAt, err := attrToTime(attr)
		if err != nil {
			return nil, err
		}
		settings.UpdatedAt = *updatedAt
	}

	setCommentSettings(attrs, settings)
	setChannelFeedSettings(attrs, settings)

	return settings, nil
}

func setCommentSettings(attrs map[string]*dynamodb.AttributeValue, settings *Settings) {
	if subsCanComment, ok := attrs["subs_can_comment"]; ok {
		settings.SubsCanComment = *subsCanComment.BOOL
	}
	if friendsCanComment, ok := attrs["friends_can_comment"]; ok {
		settings.FriendsCanComment = *friendsCanComment.BOOL
	}
	if followersCanComment, ok := attrs["followers_can_comment"]; ok {
		settings.FollowersCanComment = *followersCanComment.BOOL
	}
	if userDisabledComments, ok := attrs["user_disabled_comments"]; ok {
		settings.UserDisabledComments = *userDisabledComments.BOOL
	}
	if adminDisabledComments, ok := attrs["admin_disabled_comments"]; ok {
		settings.AdminDisabledComments = *adminDisabledComments.BOOL
	}
}

func setChannelFeedSettings(attrs map[string]*dynamodb.AttributeValue, settings *Settings) {
	if channelFeedEnabled, ok := attrs["channel_feed_enabled"]; ok {
		settings.ChannelFeedEnabled = *channelFeedEnabled.BOOL
	}
}

func timeToAttr(t time.Time) *dynamodb.AttributeValue {
	nano := strconv.FormatInt(t.UTC().UnixNano(), 10)
	return &dynamodb.AttributeValue{N: aws.String(nano)}
}

func attrToTime(val *dynamodb.AttributeValue) (*time.Time, error) {
	nano, err := strconv.ParseInt(*val.N, 10, 64)
	if err != nil {
		return nil, err
	}
	t := time.Unix(0, nano).UTC()
	return &t, nil
}
