package api

import (
	"time"

	"github.com/aws/aws-sdk-go/aws"
)

const (
	// DefaultSubsCanComment stores the default value for Settings.SubsCanComment
	DefaultSubsCanComment = true
	// DefaultFriendsCanComment stores the default value for Settings.FriendsCanComment
	DefaultFriendsCanComment = true
	// DefaultFollowersCanComment stores the default value for Settings.FollowersCanComment
	DefaultFollowersCanComment = true
	// DefaultUserDisabledComments stores the default value for Settings.UserDisabledComments
	DefaultUserDisabledComments = false
	// DefaultAdminDisabledComments stores the default value for Settings.AdminDisabledComments
	DefaultAdminDisabledComments = false
	// DefaultChannelFeedEnabled stores the default value for Settings.ChannelFeedEnabled
	DefaultChannelFeedEnabled = true
)

// Settings contains user settings
// see also https://git-aws.internal.justin.tv/web/audrey/blob/ecaf75e25387df5e2ea57b00cefd58027652a2ac/cmd/audrey/internal/db/models.go#L77-L85
type Settings struct {
	Entity                string    `json:"entity"` // User ID or feed ID
	CreatedAt             time.Time `json:"created_at"`
	UpdatedAt             time.Time `json:"updated_at"`
	SubsCanComment        bool      `json:"subs_can_comment"`
	FriendsCanComment     bool      `json:"friends_can_comment"`
	FollowersCanComment   bool      `json:"followers_can_comment"`
	UserDisabledComments  bool      `json:"user_disabled_comments"`
	AdminDisabledComments bool      `json:"admin_disabled_comments"`
	ChannelFeedEnabled    bool      `json:"channel_feed_enabled"`
}

// Defaults sets defaults and returns the receiver
func (s *Settings) Defaults() *Settings {
	return s.ApplyUpdates(SettingsUpdate{
		SubsCanComment:        aws.Bool(DefaultSubsCanComment),
		FriendsCanComment:     aws.Bool(DefaultFriendsCanComment),
		FollowersCanComment:   aws.Bool(DefaultFollowersCanComment),
		UserDisabledComments:  aws.Bool(DefaultUserDisabledComments),
		AdminDisabledComments: aws.Bool(DefaultAdminDisabledComments),
		ChannelFeedEnabled:    aws.Bool(DefaultChannelFeedEnabled),
	})
}

// ApplyUpdates and return the receiver. Does not persist changes to DB
func (s *Settings) ApplyUpdates(updates SettingsUpdate) *Settings {
	if updates.SubsCanComment != nil {
		s.SubsCanComment = *updates.SubsCanComment
	}
	if updates.FriendsCanComment != nil {
		s.FriendsCanComment = *updates.FriendsCanComment
	}
	if updates.FollowersCanComment != nil {
		s.FollowersCanComment = *updates.FollowersCanComment
	}
	if updates.UserDisabledComments != nil {
		s.UserDisabledComments = *updates.UserDisabledComments
	}
	if updates.AdminDisabledComments != nil {
		s.AdminDisabledComments = *updates.AdminDisabledComments
	}
	if updates.ChannelFeedEnabled != nil {
		s.ChannelFeedEnabled = *updates.ChannelFeedEnabled
	}
	return s
}

// SettingsUpdate is accepted by the
type SettingsUpdate struct {
	SubsCanComment        *bool `json:"subs_can_comment,omitempty"`
	FriendsCanComment     *bool `json:"friends_can_comment,omitempty"`
	FollowersCanComment   *bool `json:"followers_can_comment,omitempty"`
	UserDisabledComments  *bool `json:"user_disabled_comments,omitempty"`
	AdminDisabledComments *bool `json:"admin_disabled_comments,omitempty"`
	ChannelFeedEnabled    *bool `json:"channel_feed_enabled,omitempty"`
}
