package main

import (
	"net"
	"os"

	"os/signal"

	"code.justin.tv/common/config"
	"code.justin.tv/feeds/feed-settings/cmd/feed-settings/internal/api"
	"code.justin.tv/feeds/service-common"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

const (
	teamName    = "feeds"
	serviceName = "feed-settings"
)

// CodeVersion is set by build script
var CodeVersion string

var instance = service{
	osExit: os.Exit,
	serviceCommon: service_common.ServiceCommon{
		ConfigCommon: service_common.ConfigCommon{
			Team:       teamName,
			Service:    serviceName,
			OsGetenv:   os.Getenv,
			OsHostname: os.Hostname,
		},
		CodeVersion: CodeVersion,
	},
}

type service struct {
	osExit        func(code int)
	onListen      func(net.Addr)
	sigChan       chan os.Signal
	serviceCommon service_common.ServiceCommon
	runner        service_common.ServiceRunner

	server api.HTTPServer

	configs struct {
		HTTPConfig api.HTTPConfig
		DBConfig   api.DBConfig
	}
	skipGlobalConfigParse bool
}

func (f *service) setup() error {
	if err := f.serviceCommon.Setup(); err != nil {
		return err
	}
	if err := service_common.LoadConfigs(f.serviceCommon.Config, &f.configs.HTTPConfig, &f.configs.DBConfig); err != nil {
		return err
	}

	if !f.skipGlobalConfigParse {
		if err := config.Parse(); err != nil {
			return err
		}
	}

	return nil
}

func (f *service) inject() {
	session, awsConf := service_common.CreateAWSSession(f.serviceCommon.Config)
	dynamoClient := dynamodb.New(session, awsConf...)
	f.serviceCommon.XRay.AWS(dynamoClient.Client)

	f.server = api.HTTPServer{
		BaseHTTPServer: service_common.BaseHTTPServer{
			Config: &f.configs.HTTPConfig.BaseHTTPServerConfig,
			Stats: &service_common.StatSender{
				SubStatter:   f.serviceCommon.Statsd.NewSubStatter("http"),
				ErrorTracker: &f.serviceCommon.ErrorTracker,
			},
			Dims:        &f.serviceCommon.CtxDimensions,
			Log:         f.serviceCommon.Log,
			ElevateKey:  f.serviceCommon.ElevateLogKey,
			Ctxlog:      &f.serviceCommon.Ctxlog,
			OnListen:    f.onListen,
			PanicLogger: f.serviceCommon.PanicLogger,
			XRay:        f.serviceCommon.XRay,
		},
		DB: &api.DB{
			Dynamo: dynamoClient,
			Config: &f.configs.DBConfig,
			Log:    f.serviceCommon.Log,
			RequireConsistentReads: f.serviceCommon.Config.Bool("dynamo.consistent_reads", false).Get(),
		},
	}
	f.server.BaseHTTPServer.SetupRoutes = f.server.SetupRoutes

	f.runner = service_common.ServiceRunner{
		Log: f.serviceCommon.Log,
		Services: []service_common.Service{
			&f.server, &f.serviceCommon,
		},
		SigChan:      f.sigChan,
		SignalNotify: signal.Notify,
	}

	res := (&service_common.NilCheck{
		IgnoredPackages: []string{"aws-sdk-go", "net/http"},
	}).Check(f, f.server, f.configs, f.runner)
	res.MustBeEmpty(os.Stderr)
}

func (f *service) main() {
	if err := f.setup(); err != nil {
		if f.serviceCommon.Log != nil {
			f.serviceCommon.Log.Log("err", err, "Unable to load initial config")
		}
		service_common.SetupLogger.Log("err", err, "Unable to load initial config")
		f.osExit(1)
		return
	}
	f.inject()
	if err := f.runner.Execute(); err != nil {
		f.serviceCommon.Log.Log("err", err, "wait to end finished with an error")
		f.osExit(1)
		return
	}
	f.serviceCommon.Log.Log("Finished main")
}

func main() {
	instance.main()
}
