// +build integration

package main

import (
	"fmt"
	"net"
	"net/http"
	"os"
	"syscall"
	"testing"
	"time"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feed-settings/cmd/feed-settings/internal/api"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"github.com/aws/aws-sdk-go/aws"
	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
)

func getSettings(ts *testSetup, entity string) (*api.Settings, error) {
	var ret *api.Settings
	if err := request(ts, "GET", "/v1/settings/"+entity, nil, &ret); err != nil {
		return nil, err
	}
	return ret, nil
}

func updateSettings(ts *testSetup, entity string, params *api.SettingsUpdate) (*api.Settings, error) {
	var ret *api.Settings
	if err := request(ts, "POST", "/v1/settings/"+entity, params, &ret); err != nil {
		return nil, err
	}
	return ret, nil
}

func TestIntegration(t *testing.T) {
	host, cleanShutdown := startServer(t)
	if host == "" {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+host, t, func() {
		ts := &testSetup{host: host}
		So(ts.Setup(), ShouldBeNil)
		Reset(ts.cancelFunc)

		Convey("Getting settings", func() {
			Convey("Should get settings for a new entity", func() {
				entity := fmt.Sprintf("test:%v", time.Now().UnixNano())
				settings, err := getSettings(ts, entity)
				So(err, ShouldBeNil)
				So(settings.Entity, ShouldEqual, entity)
				So(settings.CreatedAt.Equal(settings.UpdatedAt), ShouldBeTrue)
				So(settings, ShouldResemble, &api.Settings{
					Entity:                entity,
					CreatedAt:             settings.CreatedAt,
					UpdatedAt:             settings.UpdatedAt,
					SubsCanComment:        true,
					FriendsCanComment:     true,
					FollowersCanComment:   true,
					UserDisabledComments:  false,
					AdminDisabledComments: false,
					ChannelFeedEnabled:    true,
				})
			})

			Convey("Should return 4xx for empty entity", func() {
				settings, err := getSettings(ts, "")
				So(HTTPCode(err), ShouldBeBetweenOrEqual, 400, 499)
				So(settings, ShouldBeNil)
			})

			Convey("Should be nilpotent (safe)", func() {
				// RFC2616 "Safe Methods" https://www.w3.org/Protocols/rfc2616/rfc2616-sec9.html#sec9.1.1
				entity := fmt.Sprintf("test:%v", time.Now().UnixNano())
				settings1, err1 := getSettings(ts, entity)
				settings2, err2 := getSettings(ts, entity)
				So(err1, ShouldBeNil)
				So(err2, ShouldBeNil)
				So(settings1, ShouldResemble, settings2)
			})
		})

		Convey("Updating settings", func() {
			Convey("Should update settings", func() {
				entity := fmt.Sprintf("test:%v", time.Now().UnixNano())
				settings, err := getSettings(ts, entity)
				So(err, ShouldBeNil)
				So(settings.AdminDisabledComments, ShouldBeFalse)

				updates := api.SettingsUpdate{
					AdminDisabledComments: aws.Bool(true),
				}
				updated, err := updateSettings(ts, entity, &updates)
				So(err, ShouldBeNil)
				// we no longer create on GET so updated.CreatedAt != settings.CreatedAt
				So(updated.UpdatedAt.Equal(updated.CreatedAt), ShouldBeTrue)
				So(updated.UpdatedAt.Equal(settings.UpdatedAt), ShouldBeFalse)
				So(WithoutTimestamps(updated), ShouldResemble, WithoutTimestamps((*settings).ApplyUpdates(updates)))
			})

			Convey("Should update channel feed settings", func() {
				entity := fmt.Sprintf("test:%v", time.Now().UnixNano())
				settings, err := getSettings(ts, entity)
				So(err, ShouldBeNil)
				So(settings.ChannelFeedEnabled, ShouldBeTrue)

				updates := api.SettingsUpdate{
					ChannelFeedEnabled: aws.Bool(false),
				}
				updated, err := updateSettings(ts, entity, &updates)
				So(err, ShouldBeNil)
				So(updated.ChannelFeedEnabled, ShouldBeFalse)
				So(WithoutTimestamps(updated), ShouldResemble, WithoutTimestamps((*settings).ApplyUpdates(updates)))
				get, err := getSettings(ts, entity)
				So(err, ShouldBeNil)
				So(get.ChannelFeedEnabled, ShouldBeFalse)
			})

			Convey("Should not update other settings", func() {
				entity := fmt.Sprintf("test:%v", time.Now().UnixNano())
				update, err := updateSettings(ts, entity, &api.SettingsUpdate{
					SubsCanComment: aws.Bool(false),
				})
				So(err, ShouldBeNil)
				So(update.SubsCanComment, ShouldBeFalse)
				So(update.FriendsCanComment, ShouldEqual, api.DefaultFriendsCanComment)
				So(update.FollowersCanComment, ShouldEqual, api.DefaultFollowersCanComment)
				So(update.UserDisabledComments, ShouldEqual, api.DefaultUserDisabledComments)
				So(update.AdminDisabledComments, ShouldEqual, api.DefaultAdminDisabledComments)
				So(update.ChannelFeedEnabled, ShouldEqual, api.DefaultChannelFeedEnabled)

				get, err := getSettings(ts, entity)
				So(err, ShouldBeNil)
				So(update, ShouldResemble, get)
			})

			Convey("Should be idempotent", func() {
				// RFC2616 "Idempotent Methods" https://www.w3.org/Protocols/rfc2616/rfc2616-sec9.html#sec9.1.2
				entity := fmt.Sprintf("test:%v", time.Now().UnixNano())
				updates := api.SettingsUpdate{
					AdminDisabledComments: aws.Bool(true),
				}

				get1, err := getSettings(ts, entity)
				So(err, ShouldBeNil)
				So(get1.AdminDisabledComments, ShouldBeFalse)

				update1, err := updateSettings(ts, entity, &updates)
				So(err, ShouldBeNil)
				get2, err := getSettings(ts, entity)
				So(err, ShouldBeNil)
				update2, err := updateSettings(ts, entity, &updates)
				So(err, ShouldBeNil)
				get3, err := getSettings(ts, entity)
				So(err, ShouldBeNil)

				// idempotency
				So(WithoutUpdatedAt(update1), ShouldNotResemble, WithoutUpdatedAt(get1))
				So(WithoutUpdatedAt(update1), ShouldResemble, WithoutUpdatedAt(get2))

				So(WithoutUpdatedAt(update2), ShouldResemble, WithoutUpdatedAt(get2))
				So(WithoutUpdatedAt(update2), ShouldResemble, WithoutUpdatedAt(get3))
			})
		})
	})

	cleanShutdown(time.Second * 3)
}

type HTTPError interface {
	error
	HTTPCode() int
}

func HTTPCode(err error) int {
	if errCause, ok := errors.Cause(err).(HTTPError); ok {
		return errCause.HTTPCode()
	}
	return -1
}

var dummyTime = time.Now()

func WithoutUpdatedAt(s *api.Settings) *api.Settings {
	tmp := *s
	tmp.UpdatedAt = dummyTime
	return &tmp
}

func WithoutTimestamps(s *api.Settings) *api.Settings {
	s = WithoutUpdatedAt(s)
	s.CreatedAt = dummyTime
	return s
}

type testSetup struct {
	ctx        context.Context
	cancelFunc func()
	client     *http.Client
	host       string
}

func (t *testSetup) Setup() error {
	t.ctx, t.cancelFunc = context.WithTimeout(context.Background(), time.Second*3)
	t.client = &http.Client{}
	return nil
}

func request(ts *testSetup, method, url string, body interface{}, into interface{}) error {
	return clients.DoHTTP(ts.ctx, ts.client, method, ts.host+url, nil, body, into, nil)
}

func addMapValues(m *distconf.InMemory, vals map[string][]byte) error {
	for k, v := range vals {
		if err := m.Write(k, v); err != nil {
			return err
		}
	}
	return nil
}

type panicPanic struct{}

func (p panicPanic) OnPanic(pnc interface{}) {
	panic(pnc)
}

func startServer(t *testing.T) (string, func(time.Duration)) {
	localConf := &distconf.InMemory{}
	err := addMapValues(localConf, map[string][]byte{
		"feed-settings.listen_addr": []byte(":0"),
		"rollbar.access_token":      []byte(""),
		"statsd.hostport":           []byte(""),
		"debug.addr":                []byte(":0"),
		"logging.to_stdout":         []byte("false"),
		"logging.to_stderr":         []byte("true"),
		"dynamo.consistent_reads":   []byte("true"),
	})
	if err != nil {
		t.Error(err)
		return "", nil
	}

	started := make(chan string)
	finished := make(chan struct{})
	signalToClose := make(chan os.Signal)
	exitCalled := make(chan struct{})
	elevateKey := "hi"
	thisInstance := service{
		osExit: func(i int) {
			if i != 0 {
				t.Error("Invalid osExit status code", i)
			}
			close(exitCalled)
		},
		serviceCommon: service_common.ServiceCommon{
			ConfigCommon: service_common.ConfigCommon{
				Team:          teamName,
				Service:       serviceName,
				CustomReaders: []distconf.Reader{localConf},
				BaseDirectory: "../../",
				ElevateLogKey: elevateKey,
				OsGetenv:      os.Getenv,
				OsHostname:    os.Hostname,
			},
			CodeVersion: CodeVersion,
			Log: &log.ElevatedLog{
				ElevateKey: elevateKey,
				NormalLog: log.ContextLogger{
					Logger: t,
				},
				DebugLog: log.ContextLogger{
					Logger: log.Discard,
				},
				LogToDebug: func(_ ...interface{}) bool {
					return false
				},
			},
			PanicLogger: panicPanic{},
		},
		sigChan: signalToClose,
		onListen: func(listeningAddr net.Addr) {
			started <- fmt.Sprintf("http://localhost:%d", listeningAddr.(*net.TCPAddr).Port)
		},
	}
	thisInstance.serviceCommon.Log.NormalLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	thisInstance.serviceCommon.Log.DebugLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	go func() {
		thisInstance.main()
		close(finished)
	}()

	var addressForIntegrationTests string
	select {
	case <-exitCalled:
		return "", nil
	case addressForIntegrationTests = <-started:
	case <-time.After(time.Second * 10):
		t.Error("Took to long to start service")
		return "", nil
	}

	onFinish := func(timeToWait time.Duration) {
		signalToClose <- syscall.SIGTERM
		select {
		case <-finished:
			return
		case <-time.After(timeToWait):
			t.Error("Timed out waiting for server to end")
		}
	}
	return addressForIntegrationTests, onFinish
}

func TestMainFunc(t *testing.T) {
	ret := 0
	a := instance.osExit
	b := instance.serviceCommon.BaseDirectory
	c := instance.serviceCommon.Log
	defer func() {
		instance.osExit = a
		instance.serviceCommon.BaseDirectory = b
		instance.serviceCommon.Log = c
	}()
	instance.osExit = func(i int) { ret = i }
	instance.serviceCommon.BaseDirectory = "asdfjadsfjsdkfjsdkjghkdls"
	instance.serviceCommon.Log = nil
	main()
	if ret == 0 {
		t.Error("expected instance to fail")
	}
}
