locals {
  owner   = "jlindamo@twitch.tv"
  team    = "feeds"
  region  = "us-west-2"
  service = "feed-settings"

  account_ids = {
    integration = "724951484461"
    staging     = "724951484461"
    production  = "914569885343"
  }

  profiles = {
    integration = "twitch-feed-dev"
    staging     = "twitch-feed-dev"
    production  = "twitch-feed-aws"
  }

  settings_read_capacities = {
    integration = 200
    staging     = 200
    production  = 2500
  }

  settings_read_capacity = "${lookup(local.settings_read_capacities, terraform.workspace)}"

  settings_write_capacities = {
    integration = 100
    staging     = 100
    production  = 5000
  }

  settings_write_capacity = "${lookup(local.settings_write_capacities, terraform.workspace)}"

  alarm_threshold_percents = {
    integration = 70
    staging     = 70
    production  = 70
  }

  alarm_threshold_percent = "${lookup(local.alarm_threshold_percents, terraform.workspace)}"

  alarm_periods = {
    integration = 60
    staging     = 60
    production  = 60
  }

  alarm_period = "${lookup(local.alarm_periods, terraform.workspace)}"

  profile = "${lookup(local.profiles, terraform.workspace)}"
}

provider "aws" {
  profile             = "${local.profile}"
  allowed_account_ids = ["${lookup(local.account_ids, terraform.workspace)}"]
  region              = "${local.region}"
}

terraform {
  backend "s3" {
    bucket  = "twitch-feed-aws"
    key     = "feeds/feed-settings-dynamodb-us-west-2.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "account_global" {
  backend     = "s3"
  environment = "${local.profile}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/core-global.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "account_core" {
  backend     = "s3"
  environment = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/core-account.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "region_core" {
  backend     = "s3"
  environment = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/core-region-${local.region}.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

resource "aws_dynamodb_table" "settings_dynamodb" {
  name           = "${local.service}_${terraform.workspace}_settings"
  read_capacity  = "${local.settings_read_capacity}"
  write_capacity = "${local.settings_write_capacity}"
  hash_key       = "entity"

  attribute {
    name = "entity"
    type = "S"
  }

  lifecycle {
    prevent_destroy = true
  }

  tags {
    Environment = "${terraform.workspace}"
    Service     = "${local.service}"
    Team        = "${local.team}"
  }
}

module settings_dynamodb_backup {
  source              = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//dynamodb/datapipeline_backup?ref=v0.7.9"
  region              = "${local.region}"
  profile             = "${local.profile}"
  dynamodb_table_name = "${aws_dynamodb_table.settings_dynamodb.name}"
  subnet              = "${data.terraform_remote_state.account_global.emr_subnet}"
  s3_log_bucket       = "${data.terraform_remote_state.account_core.s3_log_bucket}"
  s3_backup_bucket    = "${data.terraform_remote_state.account_core.s3_backup_bucket}"
}

module settings_dynamodb_cloudwatch {
  source         = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//dynamodb/table_cloudwatch?ref=v0.7.9"
  table_name     = "${aws_dynamodb_table.settings_dynamodb.name}"
  alerts_sns_arn = "${data.terraform_remote_state.region_core.pagerduty_alarm_arn}"
  read_capacity  = "${aws_dynamodb_table.settings_dynamodb.read_capacity}"
  write_capacity = "${aws_dynamodb_table.settings_dynamodb.write_capacity}"
}

output settings_dynamodb_arn {
  value = "${aws_dynamodb_table.settings_dynamodb.arn}"
}
