package rollbar

import (
	"errors"
	"net/http"
	"testing"

	"code.justin.tv/common/golibs/errorlogger/rollbar/mocks"
	"github.com/heroku/rollbar"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type RollbarSuite struct {
	suite.Suite
	rollbar *mocks.Client
	logger  errorLoggerImpl
}

func (s *RollbarSuite) SetupTest() {
	rollbar := &mocks.Client{}
	rollbar.On("Message", mock.Anything, mock.Anything).Return()
	rollbar.On("ErrorWithStackSkip", mock.Anything, mock.Anything, mock.Anything).Return()
	rollbar.On("RequestErrorWithStackSkip", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return()
	s.rollbar = rollbar
	logger := errorLoggerImpl{rollbar}
	s.logger = logger
}

func (s *RollbarSuite) TestNew() {
	logger := NewRollbarLogger("token", "development")
	s.NotNil(logger)
	l, ok := logger.(*errorLoggerImpl)
	s.True(ok)
	s.Equal("token", l.rollbar.GetToken())
	s.Equal("development", l.rollbar.GetEnvironment())
}

func (s *RollbarSuite) TestInfo() {
	s.logger.Info("hello")
	s.rollbar.AssertCalled(s.T(), "Message", rollbar.INFO, "hello")
}

func (s *RollbarSuite) TestError() {
	err := errors.New("uhoh")
	s.logger.Error(err)
	s.rollbar.AssertCalled(s.T(), "ErrorWithStackSkip", rollbar.ERR, err, entireStack)
}

func (s *RollbarSuite) TestRequestError() {
	err := errors.New("uhoh")
	request := &http.Request{}
	s.logger.RequestError(request, err)
	s.rollbar.AssertCalled(s.T(), "RequestErrorWithStackSkip", rollbar.ERR, request, err, entireStack)
}

func (s *RollbarSuite) TestRequestPanic() {
	request := &http.Request{}
	expectedErr := errors.New("panic: do not panic")
	s.logger.RequestPanic(request, "do not panic")
	s.rollbar.AssertCalled(s.T(), "RequestErrorWithStackSkip", rollbar.ERR, request, expectedErr, truncatedStack)
}

func TestRollbarSuite(t *testing.T) {
	suite.Run(t, new(RollbarSuite))
}
