package awsexpvar

import (
	"encoding/json"
	"errors"
	"expvar"
	"io/ioutil"
	"net/http"
	"os"
	"strings"
)

const metadataURL = "http://169.254.169.254/latest/meta-data/"
const taskRoleURL = "http://169.254.170.2"
const instanceIdentURL = "http://169.254.169.254/latest/dynamic/instance-identity/document"
const userDataURL = "http://169.254.169.254/latest/user-data"

func localIP() string {
	localIPResp, err := http.Get("http://169.254.169.254/latest/meta-data/local-ipv4/")
	if err != nil {
		return ""
	}
	localIP, err := ioutil.ReadAll(localIPResp.Body)
	if err != nil {
		return ""
	}
	return string(localIP)
}

func ecsURL() string {
	ip := localIP()
	if ip == "" {
		return ""
	}
	return "http://" + ip + ":51678"
}

// Expvar allows exposing ECS and EC2 metadata on expvar
type Expvar struct {
	Client *http.Client
}

type availableCommandResponse struct {
	AvailableCommands []string `json:"AvailableCommands"`
}

func (e *Expvar) Var() expvar.Var {
	return expvar.Func(func() interface{} {
		ret := make(map[string]interface{}, 5)
		ret["meta-data"] = e.metaData()
		ret["ecs-metadata"] = e.ecs()
		ret["instance-identity"] = e.instanceIdentity()
		ret["user-data"] = e.userData()
		return ret
	})
}

func (e *Expvar) userData() interface{} {
	val, err := e.single(userDataURL)
	if err != nil {
		return err
	}
	return val
}

func (e *Expvar) instanceIdentity() interface{} {
	val, err := e.single(instanceIdentURL)
	if err != nil {
		return err
	}
	return val
}

func (e *Expvar) metaData() interface{} {
	val, err := e.recurse(metadataURL)
	if err != nil {
		return err
	}
	return val
}

func (e *Expvar) ecs() interface{} {
	ecsURL := ecsURL()
	if ecsURL == "" {
		return "-not-on-ecs-"
	}
	val, err := e.recurse(ecsURL)
	if err != nil {
		return err
	}
	if asMap, ok := val.(map[string]interface{}); ok {
		taskRole, err := e.taskRole()
		if err != nil {
			asMap["RoleArn"] = err.Error()
		} else {
			asMap["RoleArn"] = taskRole
		}
	}
	return val
}

type metadataTask struct {
	Arn           string
	DesiredStatus string
	KnownStatus   string
	Family        string
	Version       string
	Containers    []metadataContainer
}

type metadataContainer struct {
	DockerId   string
	DockerName string
	Name       string
}

type tasksEndpoint struct {
	Tasks []metadataTask
}

func (e *Expvar) taskRole() (string, error) {
	credURL := os.Getenv("AWS_CONTAINER_CREDENTIALS_RELATIVE_URI")
	if credURL == "" {
		return "(no-relative-url-for-task-information)", nil
	}
	singleVal, err := e.single(taskRoleURL + credURL)
	if err != nil {
		return err.Error(), nil
	}
	if asMap, ok := singleVal.(map[string]string); ok {
		return asMap["RoleArn"], nil
	}
	return "<invalid_single_value>", nil
}

func (e *Expvar) single(base string) (interface{}, error) {
	req, err := http.NewRequest("GET", base, nil)
	if err != nil {
		return nil, err
	}
	resp, err := e.Client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if resp.StatusCode == http.StatusNotFound {
		return nil, errors.New("not found")
	}
	respBody := ""
	if b, err := ioutil.ReadAll(resp.Body); err != nil {
		return nil, err
	} else {
		respBody = string(b)
	}
	m := map[string]string{}
	if err := json.Unmarshal([]byte(respBody), &m); err == nil {
		clearOut(m, "Token")
		clearOut(m, "AccessKeyId")
		clearOut(m, "SecretAccessKey")
		return m, nil
	}
	t := tasksEndpoint{}
	if err := json.Unmarshal([]byte(respBody), &t); err == nil && len(t.Tasks) > 0 {
		return t, nil
	}
	return respBody, nil
}

func clearOut(m map[string]string, key string) {
	if _, exists := m[key]; exists {
		m[key] = "(removed)"
	}
}

func (e *Expvar) recurse(base string) (interface{}, error) {
	ret := make(map[string]interface{})
	req, err := http.NewRequest("GET", base, nil)
	if err != nil {
		return nil, err
	}
	resp, err := e.Client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if resp.StatusCode == http.StatusNotFound {
		return nil, errors.New("not found")
	}
	respBody := ""
	if b, err := ioutil.ReadAll(resp.Body); err != nil {
		return nil, err
	} else {
		respBody = string(b)
	}
	// Try availableCommandResponse for sub commands
	var m availableCommandResponse
	if err := json.Unmarshal([]byte(respBody), &m); err == nil {
		if len(m.AvailableCommands) > 0 {
			for _, subCommand := range m.AvailableCommands {
				if subCommand == "/license" {
					continue
				}
				val, err := e.single(base + subCommand)
				if err != nil {
					ret[subCommand] = err
				} else {
					ret[subCommand] = val
				}
			}
			return ret, nil
		}
	}
	// Got an object back.  Is it a link to more sub directories, or is it the end.  We don't know.
	parts := strings.Split(respBody, "\n")
	for _, part := range parts {
		if part == "" {
			continue
		}
		if part == "security-credentials/" {
			continue
		}
		if !strings.HasSuffix(part, "/") {
			val, err := e.single(base + "/" + part)
			if err != nil {
				ret[part] = err
			} else {
				ret[part] = val
			}
			continue
		}
		val, err := e.recurse(base + "/" + part)
		if err != nil {
			ret[part] = err
		} else {
			ret[part] = val
		}
	}
	return ret, nil
}
