package manager

import (
	"testing"

	sandstormError "code.justin.tv/systems/sandstorm/errors"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

func TestManagerGet(t *testing.T) {
	assert := assert.New(t)
	// manager with mocked dynamo
	t.Run("manager.get unit tests", func(t *testing.T) {

		t.Run("Get", func(t *testing.T) {

			t.Run("should not error and return nil on non-existent key", func(t *testing.T) {
				t.Parallel()
				mockedManager, mockedDDB, _ := createMocks(t)
				mockedDDB.On("Query", mock.Anything).Return(&dynamodb.QueryOutput{
					Count:            aws.Int64(0),
					Items:            []map[string]*dynamodb.AttributeValue{},
					LastEvaluatedKey: map[string]*dynamodb.AttributeValue{},
				}, nil).Once()

				secret, err := mockedManager.Get("nonexistentSecret")
				assert.Nil(err)

				mockedDDB.AssertExpectations(t)
				assert.Nil(secret)
			})

			t.Run("should not error out for an existing key", func(t *testing.T) {
				t.Parallel()
				mockedManager, mockedDDB, mockedEnveloper := createMocks(t)
				testDynamoSecret := newTestDynamoSecret()
				secretMap, err := dynamodbattribute.MarshalMap(testDynamoSecret)
				assert.Nil(err)

				mockedDDB.On("Query", mock.Anything).Return(&dynamodb.QueryOutput{
					Count:            aws.Int64(1),
					Items:            []map[string]*dynamodb.AttributeValue{secretMap},
					LastEvaluatedKey: map[string]*dynamodb.AttributeValue{},
				}, nil).Once()
				mockedEnveloper.On("Open", mock.Anything, mock.Anything, mock.Anything).Return([]byte(testDynamoSecret.Value), nil).Once()

				actualSecretValue, err := mockedManager.Get(testDynamoSecret.Name)
				assert.Nil(err)
				if assert.NotNil(actualSecretValue) {
					assert.Equal(testDynamoSecret.Value, string(actualSecretValue.Plaintext))
				}

				mockedDDB.AssertExpectations(t)
			})

		})

		t.Run("GetVersionsEncrypted", func(t *testing.T) {

			t.Run("should not return any versions for a non-existent key", func(t *testing.T) {
				t.Parallel()
				mockedManager, mockedDDB, _ := createMocks(t)
				mockedDDB.On("Query", mock.Anything).Return(&dynamodb.QueryOutput{
					Count:            aws.Int64(0),
					Items:            []map[string]*dynamodb.AttributeValue{},
					LastEvaluatedKey: map[string]*dynamodb.AttributeValue{},
				}, nil).Once()

				versions, err := mockedManager.GetVersionsEncrypted("nonexistentSecret", 10, 0)
				assert.Nil(err, "err should be nil")

				mockedDDB.AssertExpectations(t)
				assert.Empty(versions.Secrets, "should not return any secrets")
			})

			t.Run("should return a single version for an existing key", func(t *testing.T) {
				t.Parallel()
				mockedManager, mockedDDB, _ := createMocks(t)
				testDynamoSecret := newTestDynamoSecret()
				secretMap, err := dynamodbattribute.MarshalMap(testDynamoSecret)
				assert.Nil(err, "error marshalling secret")

				mockedDDB.On("Query", mock.Anything).Return(&dynamodb.QueryOutput{
					Count:            aws.Int64(1),
					Items:            []map[string]*dynamodb.AttributeValue{secretMap},
					LastEvaluatedKey: map[string]*dynamodb.AttributeValue{},
				}, nil).Once()

				versions, err := mockedManager.GetVersionsEncrypted(testDynamoSecret.Name, 10, 0)
				assert.Nil(err, "error getting encrypted versions")

				mockedDDB.AssertExpectations(t)
				if assert.NotNil(versions) {
					assert.Len(versions.Secrets, 1)
				}
			})

			t.Run("should not error out on deleted secret", func(t *testing.T) {
				t.Parallel()
				mockedManager, mockedDDB, _ := createMocks(t)
				testTombstonedDynamoSecret := newTestDynamoSecret()
				testTombstonedDynamoSecret.Tombstone = true

				secretMap, err := dynamodbattribute.MarshalMap(testTombstonedDynamoSecret)
				assert.Nil(err, "error marshalling secret")

				mockedDDB.On("Query", mock.Anything).Return(&dynamodb.QueryOutput{
					Count:            aws.Int64(1),
					Items:            []map[string]*dynamodb.AttributeValue{secretMap},
					LastEvaluatedKey: map[string]*dynamodb.AttributeValue{},
				}, nil).Once()

				versions, err := mockedManager.GetVersionsEncrypted(testTombstonedDynamoSecret.Name, 10, 0)
				assert.Nil(err, "error marshalling secret")

				mockedDDB.AssertExpectations(t)

				if assert.NotNil(versions) && assert.NotNil(versions.Secrets) {
					assert.Empty(versions.Secrets)
				}
			})

			t.Run("negative version should error and not populate versions", func(t *testing.T) {
				t.Parallel()
				mockedManager, _, _ := createMocks(t)
				versions, err := mockedManager.GetVersionsEncrypted("testSecret", -1, 0)
				if assert.NotNil(err) {
					assert.Equal(sandstormError.ErrNegativeLimit, err)
				}
				assert.Nil(versions)
			})
		})

		t.Run("GetVersion", func(t *testing.T) {

			t.Run("should not error on nonexistent key", func(t *testing.T) {
				t.Parallel()
				mockedManager, mockedDDB, _ := createMocks(t)

				mockedDDB.On("Query", mock.Anything).Return(&dynamodb.QueryOutput{
					Count:            aws.Int64(0),
					Items:            []map[string]*dynamodb.AttributeValue{},
					LastEvaluatedKey: map[string]*dynamodb.AttributeValue{},
				}, nil).Once()

				actualSecretValue, err := mockedManager.GetVersion("nonexistentSecret", 420)
				assert.Nil(err)
				assert.Nil(actualSecretValue)
			})

			t.Run("should not error on non-existing version", func(t *testing.T) {
				t.Parallel()
				mockedManager, mockedDDB, _ := createMocks(t)

				mockedDDB.On("Query", mock.Anything).Return(&dynamodb.QueryOutput{
					Count:            aws.Int64(0),
					Items:            []map[string]*dynamodb.AttributeValue{},
					LastEvaluatedKey: map[string]*dynamodb.AttributeValue{},
				}, nil).Once()

				actualSecretValue, err := mockedManager.GetVersion("secretName", 431)
				assert.Nil(err)
				assert.Nil(actualSecretValue)
			})
		})
	})
}
