package manager

import (
	"errors"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

func TestPatchInput(t *testing.T) {
	t.Run("Validate", func(t *testing.T) {
		t.Run("should return error if both autogenerate and plaintext are sent", func(t *testing.T) {
			assert := assert.New(t)

			p := &PatchInput{
				Autogenerate: &FillPlaintextRequest{},
				Plaintext:    []byte("some plaintext"),
			}
			err := p.Validate()

			assert.Equal(ErrAutogenerateAndPlaintextBothSent, err)
		})

		t.Run("should validate autogenerate if sent", func(t *testing.T) {
			assert := assert.New(t)

			p := &PatchInput{
				Autogenerate: &FillPlaintextRequest{
					Length: -1,
				},
			}
			err := p.Validate()

			assert.Error(err)
		})

		t.Run("validate should only run once", func(t *testing.T) {
			assert := assert.New(t)

			p := &PatchInput{
				// bad case when both autogenerate and plaintext are sent
				Autogenerate: &FillPlaintextRequest{},
				Plaintext:    []byte("some plaintext"),
				validated:    true,
			}
			err := p.Validate()

			assert.NoError(err)
		})
	})
}

func TestSealPatchInputPlaintext(t *testing.T) {
	t.Run("should seal plaintext values", func(t *testing.T) {
		const (
			testsecretName = "mySecretName"
			testCipherText = "some ciphertext"
			testKey        = "some key"
			testKeyArn     = "some key arn"
			testPlaintext  = "some plaintext"
			testUpdatedAt  = 17483
		)

		assert := assert.New(t)

		mt := newManagerTest(t)
		defer mt.teardown(t)

		s := Secret{
			UpdatedAt:  testUpdatedAt,
			key:        []byte(testKey),
			ciphertext: []byte(testCipherText),
			KeyARN:     testKeyArn,
		}
		sealedSecret := s.asDynamoSecret()

		mt.mockEnveloper.On("Seal", mock.Anything, mock.Anything, []byte(testPlaintext)).Return(
			[]byte(testKey),
			[]byte(testCipherText),
			testKeyArn,
			nil,
		).Once()

		p := &PatchInput{
			Name:      testSecretName,
			Plaintext: []byte(testPlaintext),
		}

		err := mt.m.sealPatchInputPlaintext(p)
		assert.NoError(err)

		assert.Equal(&PatchInput{
			Name:      testSecretName,
			Plaintext: make([]byte, len(testPlaintext)),
			Key:       &sealedSecret.Key,
			Value:     &sealedSecret.Value,
			KeyARN:    &sealedSecret.KeyARN,

			// only know this at runtime
			UpdatedAt: p.UpdatedAt,
		}, p)
	})

	t.Run("should autogenerate if sent in request seal plaintext values", func(t *testing.T) {
		const (
			testsecretName = "mySecretName"
			testCipherText = "some ciphertext"
			testKey        = "some key"
			testKeyArn     = "some key arn"
			testPlaintext  = "some plaintext"
			testUpdatedAt  = 17483
		)

		assert := assert.New(t)

		mt := newManagerTest(t)
		defer mt.teardown(t)

		s := Secret{
			UpdatedAt:  testUpdatedAt,
			key:        []byte(testKey),
			ciphertext: []byte(testCipherText),
			KeyARN:     testKeyArn,
		}
		sealedSecret := s.asDynamoSecret()

		mt.mockEnveloper.On(
			"Seal", mock.Anything, mock.Anything, mock.Anything,
		).Run(func(args mock.Arguments) {
			plaintext := args.Get(2).([]byte)
			assert.Len(plaintext, 64)
		}).Return(
			[]byte(testKey),
			[]byte(testCipherText),
			testKeyArn,
			nil,
		).Once()

		p := &PatchInput{
			Name: testSecretName,
			Autogenerate: &FillPlaintextRequest{
				Length: 48,
			},
		}

		err := mt.m.sealPatchInputPlaintext(p)
		assert.NoError(err)

		assert.Equal(&PatchInput{
			Autogenerate: &FillPlaintextRequest{
				Length: 48,
			},
			Name:   testSecretName,
			Key:    &sealedSecret.Key,
			Value:  &sealedSecret.Value,
			KeyARN: &sealedSecret.KeyARN,

			// only know this at runtime
			UpdatedAt: p.UpdatedAt,
		}, p)
	})

	t.Run("should forward autogenerate errors", func(t *testing.T) {
		assert := assert.New(t)

		mt := newManagerTest(t)
		defer mt.teardown(t)

		err := mt.m.sealPatchInputPlaintext(&PatchInput{
			Name: "mySecretName",
			Autogenerate: &FillPlaintextRequest{
				Length: -1,
			},
		})
		assert.Error(err)
	})

	t.Run("should forward seal errors", func(t *testing.T) {
		const (
			testsecretName = "mySecretName"
			testCipherText = "some ciphertext"
			testKey        = "some key"
			testKeyArn     = "some key arn"
			testPlaintext  = "some plaintext"
			testUpdatedAt  = 17483
		)

		var (
			sealError = errors.New("mySealError")
		)

		assert := assert.New(t)

		mt := newManagerTest(t)
		defer mt.teardown(t)

		mt.mockEnveloper.On(
			"Seal", mock.Anything, mock.Anything, mock.Anything,
		).Return(nil, nil, "", sealError).Once()

		err := mt.m.sealPatchInputPlaintext(&PatchInput{
			Name: testSecretName,
			Autogenerate: &FillPlaintextRequest{
				Length: 48,
			},
		})
		assert.Equal(sealError, err)
	})
}
