package api

import (
	"encoding/json"
	"fmt"
	"net/http"
	"regexp"
	"strconv"
	"strings"
	"sync"
	"time"

	leviathan "code.justin.tv/chat/leviathan/client"
	"code.justin.tv/chat/zuma/client"
	"code.justin.tv/edge/client-incubator/nuclear_waste"
	"code.justin.tv/feeds/clients/duplo"
	"code.justin.tv/feeds/clients/feed-settings"
	"code.justin.tv/feeds/clients/masonry"
	"code.justin.tv/feeds/clients/shine"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-edge/cmd/feeds-edge/internal/api/v2"
	"code.justin.tv/feeds/feeds-edge/cmd/feeds-edge/internal/emotes"
	"code.justin.tv/feeds/feeds-edge/cmd/feeds-edge/internal/user"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/spade"
	"code.justin.tv/foundation/twitchclient"
	twitter "code.justin.tv/web/twitter/client"
	users "code.justin.tv/web/users-service/client"
	"goji.io"
	"goji.io/pat"
	"golang.org/x/net/context"
	"golang.org/x/sync/errgroup"
)

var (
	errPostNotFound = errors.New("post does not exist")
	legacyIDRegex   = regexp.MustCompile("^[0-9]+$")
)

const (
	// reportPostContent identifies the type of content that is reported when we report a post.
	// Its value was taken from https://git-aws.internal.justin.tv/chat/leviathan/blob/master/docs/api.md
	reportPostContentType = "channel_feed_post_report"

	// leviathanReportOrigin identifies the service that reported a post.
	leviathanReportOrigin = "feeds/feeds-edge"

	defaultPostLimit = v2.DefaultPostLimit
)

// HTTPConfig configures a HTTPServer
type HTTPConfig struct {
	service_common.BaseHTTPServerConfig
	feedMaxLimit              *distconf.Int
	embedTimeout              *distconf.Duration
	anonFeedCacheDuration     *distconf.Duration
	postUpdateChanSize        *distconf.Int
	duploUpdateTimeout        *distconf.Duration
	updateDuploEmotes         *distconf.Bool
	cooldownExemptTestUsers   *distconf.Str
	getReactionsEntityLimit   *distconf.Int
	enableAutomodFilter       *distconf.Bool
	automodFilterEnabledUsers *distconf.Str
	filterItems               *distconf.Bool
	backfillEmbedURLsPct      *distconf.Int
	rerouteToNuclearWaste     *distconf.Bool
}

// Load config from distconf
func (s *HTTPConfig) Load(dconf *distconf.Distconf) error {
	if err := s.BaseHTTPServerConfig.Verify(dconf, "feeds-edge"); err != nil {
		return err
	}
	s.feedMaxLimit = dconf.Int("feeds-edge.feed_max_limit", 20)
	s.embedTimeout = dconf.Duration("feeds-edge.embed_timeout", 100*time.Millisecond)
	s.duploUpdateTimeout = dconf.Duration("feeds-edge.duplo_update_post_timeout", time.Second*5)
	s.postUpdateChanSize = dconf.Int("feeds-edge.post_update_chan_size", 1024)
	s.updateDuploEmotes = dconf.Bool("feeds-edge.update_duplo_emotes", false)
	s.cooldownExemptTestUsers = dconf.Str("feeds-edge.cooldown_exempt_test_users", "")
	s.getReactionsEntityLimit = dconf.Int("feeds-edge.get_reactions_entity_limit", 100)
	s.anonFeedCacheDuration = dconf.Duration("feeds-edge.cache.anon_feed", time.Second*10)
	s.enableAutomodFilter = dconf.Bool("feeds-edge.enable_automod_filter", false)
	s.automodFilterEnabledUsers = dconf.Str("feeds-edge.automod_filter_enabled_users", "")
	s.filterItems = dconf.Bool("feeds-edge.enable_filter_items", true)
	s.backfillEmbedURLsPct = dconf.Int("feeds-edge.backfill_embedurls_pct", 0)
	s.rerouteToNuclearWaste = dconf.Bool("feeds-edge.reroute_to_nuclear_waste", false)
	return nil
}

type postEmotesToUpdate struct {
	PostID string
	Emotes []*Emote
}

// HTTPServer answers masonry HTTP requests
type HTTPServer struct {
	service_common.BaseHTTPServer
	Config      *HTTPConfig
	DuploCacher DuploCacher
	APIv2       *v2.API

	Masonry            *masonry.Client
	Duplo              *duplo.Client
	Shine              ShineClient
	Zuma               zuma.Client
	NuclearWaste       nuclear_waste.Client
	FeedSettingsClient *feedsettings.Client
	EmoteParser        *emotes.EmoteParser
	Filter             *Filter
	Authorization      *Authorization
	postUpdateChan     chan postEmotesToUpdate
	Cooldowns          v2.Cooldowns
	TwitterClient      *twitter.Client
	UsersClient        users.Client
	LeviathanClient    leviathan.Client
	ClueClient         ClueClient
	SpadeClient        v2.SpadeClient
	UserDeleter        *feedsedgeuser.Deleter
}

// ShineClient asserts what functions are needed from shine.Client to allow mocking
type ShineClient interface {
	GetEmbed(ctx context.Context, embedURL string, options *shine.GetEmbedOptions, reqOpts *twitchclient.ReqOpts) (*shine.Embed, error)
	GetEntitiesForURLs(ctx context.Context, urls []string, reqOpts *twitchclient.ReqOpts) (*shine.EntitiesForURLs, error)
	GetEmbedForEntity(ctx context.Context, ent entity.Entity, options *shine.GetEmbedOptions, reqOpts *twitchclient.ReqOpts) (*shine.Embed, error)
}

var _ ShineClient = shine.Client(nil)

type httpCallback func(req *http.Request) (interface{}, error)

func (s *HTTPServer) createHandler(name string, callback httpCallback) *service_common.JSONHandler {
	return &service_common.JSONHandler{
		Log:          s.Log,
		Stats:        s.Stats.NewSubStatSender(name),
		ItemCallback: s.filterItems(callback),
	}
}

type canFilter interface {
	filter(ctx context.Context, f *Filter, knownValidUsers stringSet) interface{}
}

func (s *HTTPServer) filterItems(callback httpCallback) httpCallback {
	return func(req *http.Request) (interface{}, error) {
		originalItem, err := callback(req)
		if !s.Config.filterItems.Get() {
			return originalItem, err
		}
		if err != nil {
			return nil, err
		}
		if originalItem == nil {
			return nil, nil
		}
		if filterable, ok := originalItem.(canFilter); ok {
			// We assume the logged in user is valid.  Strange things could happen if we filter your own
			// posts for some reason
			userID := req.URL.Query().Get("user_id")
			knownIDs := stringSet(make(map[string]struct{}, 1))
			if userID != "" {
				knownIDs.add(userID)
			}
			newItem := filterable.filter(req.Context(), s.Filter, knownIDs)
			if newItem == nil {
				return nil, &service_common.CodedError{
					Code: http.StatusNotFound,
					Err:  errors.New("no valid item found"),
				}
			}
			return newItem, nil
		}
		return originalItem, nil
	}
}

// SetupRoutes configures feeds-edge routes
func (s *HTTPServer) SetupRoutes(mux *goji.Mux) {
	s.postUpdateChan = make(chan postEmotesToUpdate, s.Config.postUpdateChanSize.Get())

	mux.Handle(pat.Post("/v1/posts"), s.createHandler("create_post", s.createPost))
	mux.Handle(pat.Post("/v1/posts/syndicate"), s.createHandler("create_post", s.createSyndicatedPost))
	mux.Handle(pat.Get("/v1/posts/:post_id"), s.createHandler("get_post", s.getPost))
	mux.Handle(pat.Delete("/v1/posts/:post_id"), s.createHandler("delete_post", s.deletePost))
	mux.Handle(pat.Post("/v1/posts/:post_id/report"), s.createHandler("report_post", s.reportPost))

	// Get the permissions for the feed for the given user id
	mux.Handle(pat.Get("/v1/permissions/:feed_id"), s.createHandler("get_feed_permissions", s.getFeedPermissions))

	mux.Handle(pat.Post("/v1/comments"), s.createHandler("create_comment", s.createComment))
	mux.Handle(pat.Get("/v2/comments/:comment_id"), s.createHandler("get_comment", s.getComment))
	mux.Handle(pat.Delete("/v1/comments/:comment_id"), s.createHandler("delete_comment", s.deleteComment))
	mux.Handle(pat.Post("/v1/comments/:comment_id/report"), s.createHandler("report_comment", s.reportComment))
	mux.Handle(pat.Put("/v1/comments/:comment_id/approve"), s.createHandler("approve_comment", s.approveComment))
	mux.Handle(pat.Put("/v1/comments/:comment_id/deny"), s.createHandler("deny_comment", s.denyComment))

	mux.Handle(pat.Get("/v1/posts/:post_id/comments"), s.createHandler("get_comments_for_post", s.getCommentsForPost))
	mux.Handle(pat.Delete("/v1/comments/by_parent_and_user/:parent_entity/:comment_user_id"), s.createHandler("delete_comments_by_parent_and_user", s.deleteCommentsByParentAndUser))

	mux.Handle(pat.Get("/v1/reactions"), s.createHandler("get_reactions_by_parents", s.getReactionsByParents))
	mux.Handle(pat.Put("/v1/reactions/:parent_entity/:emote_id"), s.createHandler("create_reaction", s.createReaction))
	mux.Handle(pat.Delete("/v1/reactions/:parent_entity/:emote_id"), s.createHandler("delete_reaction", s.deleteReaction))

	mux.Handle(pat.Get("/v1/feeds/:feed_id"), s.createHandler("get_feed", s.getFeed))

	mux.Handle(pat.Get("/v1/settings/:entity"), s.createHandler("get_settings", s.getSettings))
	mux.Handle(pat.Post("/v1/settings/:entity"), s.createHandler("update_settings", s.updateSettings))

	mux.Handle(pat.Get("/v1/embed"), s.createHandler("get_embed", s.getEmbed))

	// Share CRUD
	mux.Handle(pat.Post("/v1/shares"), s.createHandler("create_share", s.createShare))
	mux.Handle(pat.Delete("/v1/shares/:share_id"), s.createHandler("delete_share", s.deleteShare))
	mux.Handle(pat.Get("/v1/shares/:share_id"), s.createHandler("get_share", s.getShare))
	mux.Handle(pat.Get("/v1/shares"), s.createHandler("get_shares", s.getShares))

	// /v1/shares_summary is RPC
	mux.Handle(pat.Get("/v1/shares_summaries"), s.createHandler("get_shares_summaries", s.getSharesSummaries))

	// /v1/unshare_by_target is RPC
	mux.Handle(pat.Delete("/v1/unshare_by_target"), s.createHandler("unshare_by_target", s.unshareByTarget))

	mux.Handle(pat.Delete("/v1/hard_delete_by_user"), s.createHandler("hard_delete", s.hardDeleteUser))

	// version 2 handlers
	mux.Handle(pat.Get("/v2/get_feed"), s.createHandler("/v2/get_feed", s.APIv2.GetFeed))

	mux.Handle(pat.Post("/v2/create_post"), s.createHandler("/v2/create_post", s.APIv2.CreatePost))
	mux.Handle(pat.Delete("/v2/delete_post"), s.createHandler("/v2/delete_post", s.APIv2.DeletePost))
	mux.Handle(pat.Get("/v2/get_posts_by_ids"), s.createHandler("/v2/get_posts_by_ids", s.APIv2.GetPostsByIDs))
	mux.Handle(pat.Get("/v2/get_posts_permissions_by_ids"), s.createHandler("/v2/get_posts_permissions_by_ids", s.APIv2.GetPostsPermissionsByIDs))

	mux.Handle(pat.Post("/v2/create_share"), s.createHandler("/v2/create_share", s.APIv2.CreateShare))
	mux.Handle(pat.Delete("/v2/delete_share"), s.createHandler("/v2/delete_share", s.APIv2.DeleteShare))
	mux.Handle(pat.Get("/v2/get_shares_by_ids"), s.createHandler("/v2/get_shares_by_ids", s.APIv2.GetSharesByIDs))

	mux.Handle(pat.Put("/v2/create_reaction"), s.createHandler("/v2/create_reaction", s.APIv2.CreateReaction))
	mux.Handle(pat.Delete("/v2/delete_reaction"), s.createHandler("/v2/delete_reaction", s.APIv2.DeleteReaction))
	mux.Handle(pat.Get("/v2/get_reactions_by_entities"), s.createHandler("/v2/get_reactions_by_entities", s.APIv2.GetReactionsSummariesByEntities))
	mux.Handle(pat.Get("/v2/suggested_feeds"), s.createHandler("/v2/suggested_feeds", s.APIv2.GetSuggestedFeeds))
}

func parseInt(intParam string, defaultValue int) (int, error) {
	return v2.ParseInt(intParam, defaultValue)
}

func requireUserID(req *http.Request) (string, error) {
	return v2.RequireUserID(req)
}

func entitiesFromQueryParam(r *http.Request, paramName string) ([]entity.Entity, error) {
	values := itemsFromQueryParam(r, paramName)
	if len(values) == 0 {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("must provide at least one " + paramName),
		}
	}
	entities := make([]entity.Entity, len(values))
	for i, value := range values {
		ent, err := entity.Decode(value)
		if err != nil {
			return nil, &service_common.CodedError{
				Code: http.StatusBadRequest,
				Err:  errors.Wrap(err, "malformed "+paramName),
			}
		}
		entities[i] = ent
	}
	return entities, nil
}

func itemsFromQueryParam(r *http.Request, paramName string) []string {
	return v2.ItemsFromQueryParam(r, paramName)
}

func (s *HTTPServer) requireValidSettingsEntity(req *http.Request) (*entity.Entity, error) {
	settingsEntity, err := entity.Decode(pat.Param(req, "entity"))
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "malformed entity"),
		}
	}

	if settingsEntity.Namespace() != entity.NamespaceUser {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Errorf("expected user:<id>, found %s:<id>", settingsEntity.Namespace()),
		}
	}

	return &settingsEntity, nil
}

func (s *HTTPServer) loadAndBackfillDuploPost(ctx context.Context, postID string) (*duplo.Post, error) {
	return s.APIv2.LoadAndBackfillDuploPost(ctx, postID)
}

func (s *HTTPServer) requireDuploPost(ctx context.Context, postID string) (*duplo.Post, error) {
	return s.APIv2.RequireDuploPost(ctx, postID)
}

func (s *HTTPServer) getPostIDIfLegacy(ctx context.Context, postID string) (string, error) {
	if legacyIDRegex.MatchString(postID) {
		postIDContainer, err := s.Duplo.GetPostIDByLegacyID(ctx, postID)
		if err != nil {
			return "", err
		}
		postID = postIDContainer.ID
	}
	return postID, nil
}

func (s *HTTPServer) getFeedPermissions(req *http.Request) (interface{}, error) {
	feedID := pat.Param(req, "feed_id")
	userID := req.URL.Query().Get("user_id")
	postPermissions, err := s.Authorization.GetPostPermissionsForUserToAuthor(req.Context(), userID, feedID)
	if err != nil {
		return nil, err
	}
	return postPermissions, nil
}

func (s *HTTPServer) getPost(req *http.Request) (interface{}, error) {
	postID, err := s.getPostIDIfLegacy(req.Context(), pat.Param(req, "post_id"))
	if err != nil {
		return nil, err
	}
	userID := req.URL.Query().Get("user_id")
	return s.loadPost(req.Context(), postID, userID)
}

func (s *HTTPServer) populatedPostFromAnyID(ctx context.Context, postID string) (*Post, error) {
	contentEntity, err := entity.Decode(postID)
	if err == nil {
		var shinePost *Post
		shinePost, err = s.loadPostFromShineData(ctx, contentEntity)
		if err != nil {
			return nil, err
		}
		return shinePost, nil
	}
	duploPost, err := s.requireDuploPost(ctx, postID)
	if err != nil {
		return nil, err
	}
	post := createPostFromDuploPost(duploPost)
	return post, nil
}

func (s *HTTPServer) loadPost(ctx context.Context, postID string, userID string) (*Post, error) {
	post, err := s.populatedPostFromAnyID(ctx, postID)
	if err != nil {
		return nil, err
	}

	if err := s.populatePosts(ctx, []*Post{post}, userID); err != nil {
		return nil, err
	}

	if len(s.filterInvalidPosts(ctx, []*Post{post})) == 0 {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errPostNotFound,
		}
	}

	return post, nil
}

func (s *HTTPServer) loadPostFromShineData(ctx context.Context, ent entity.Entity) (*Post, error) {
	if ent.Namespace() == entity.NamespaceVod {
		return createPostFromVod(ent), nil
	}
	if ent.Namespace() == entity.NamespaceClip {
		return createPostFromClip(ent), nil
	}
	if ent.Namespace() == entity.NamespaceStream {
		postFromStream := createPostFromStream(ent)
		if postFromStream == nil {
			return nil, &service_common.CodedError{
				Code: http.StatusBadRequest,
				Err:  errors.Errorf("stream entity %s provided in an invalid format", ent.ID()),
			}
		}
		return postFromStream, nil
	}
	return nil, &service_common.CodedError{
		Code: http.StatusBadRequest,
		Err:  errors.Errorf("namespace %s not supported", ent.Namespace()),
	}
}

func (s *HTTPServer) deletePost(req *http.Request) (interface{}, error) {
	postID, err := s.getPostIDIfLegacy(req.Context(), pat.Param(req, "post_id"))
	if err != nil {
		return nil, err
	}

	duploPost, userID, err := s.APIv2.DeletePostShared(req, postID)
	if err != nil {
		return nil, err
	}

	post := createPostFromDuploPost(duploPost)

	if err := s.populatePosts(req.Context(), []*Post{post}, userID); err != nil {
		return nil, err
	}

	return post, nil
}

func (s *HTTPServer) hardDeleteUser(req *http.Request) (interface{}, error) {
	// The user is who is asking to remove the posts
	userID := req.URL.Query().Get("user_id")

	if userID == "" {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("missing parameter user_id"),
		}
	}

	// TODO (bern): We can change this to s.Log.LogCtx once inactive users are purged (https://twitchtv.atlassian.net/browse/USER-188, 6/8/17)
	s.Log.DebugCtx(req.Context(), "user_id", userID, "This is where we would remove a user ...")

	return s.UserDeleter.HardDeleteUser(userID)
}

// ReportParams represents the request body expected by the ReportComment and ReportPost endpoints.
type ReportParams struct {
	Reason string `json:"reason"`
}

func (s *HTTPServer) reportPost(req *http.Request) (interface{}, error) {
	reporterUserID, err := requireUserID(req)
	if err != nil {
		return nil, err
	}

	postID, err := s.getPostIDIfLegacy(req.Context(), pat.Param(req, "post_id"))
	if err != nil {
		return nil, err
	}
	contentID := postID

	post, err := s.loadPost(req.Context(), postID, reporterUserID)
	if err != nil {
		return nil, err
	}

	var postContent string
	var reportContentType string
	var ent entity.Entity
	if ent, err = entity.Decode(postID); err == nil {
		if post.EmbedURLs != nil {
			postContent = strings.Join(*post.EmbedURLs, ",")
		}
		reportContentType = generateReportContentType(ent.Namespace())
		// Leviathan does not handle the namespace prefix, so we remove it from the contentID.
		// See CF-884 for more details.
		contentID = ent.ID()
	} else {
		ent, _ = entity.Decode("post:" + postID)
		postContent = post.Body
		reportContentType = reportPostContentType
	}

	var params ReportParams
	if err = json.NewDecoder(req.Body).Decode(&params); err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "could not parse body"),
		}
	}

	reportParams := leviathan.CreateReportParams{
		FromUserID:   reporterUserID,
		TargetUserID: post.UserID,
		Reason:       params.Reason,
		Description:  postContent,
		Content:      reportContentType,
		Origin:       leviathanReportOrigin,
		ContentID:    contentID,
	}

	if err = s.LeviathanClient.CreateReport(req.Context(), reportParams, nil); err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusInternalServerError,
			Err:  errors.Wrap(err, "creating a report in Leviathan failed"),
		}
	}

	s.SpadeClient.QueueEvents(spade.Event{
		Name: "feed_server_report",
		Properties: v2.ServerReportTracking{
			UserID:       reporterUserID,
			TargetEntity: ent.Encode(),
			TargetType:   ent.Namespace(),
			TargetID:     ent.ID(),
			Reason:       params.Reason,
		},
	})

	return nil, nil
}

func generateReportContentType(entNamespace string) string {
	reportSuffix := "_report"
	switch entNamespace {
	case entity.NamespaceVod:
		return "vod" + reportSuffix
	case entity.NamespaceClip:
		return "clip" + reportSuffix
	}
	return "user" + reportSuffix
}

func (s *HTTPServer) createShare(req *http.Request) (interface{}, error) {
	return s.APIv2.CreateShare(req)
}

func entitiesFromURL(r *http.Request, paramName string) ([]entity.Entity, error) {
	targetEntityIDs := itemsFromQueryParam(r, paramName)
	targetEntities := make([]entity.Entity, len(targetEntityIDs))
	for idx, id := range targetEntityIDs {
		targetEntity, err := entity.Decode(id)
		if err != nil {
			return nil, &service_common.CodedError{
				Code: http.StatusBadRequest,
				Err:  errors.Wrapf(err, "malformed entity query parameter %s", paramName),
			}
		}
		targetEntities[idx] = targetEntity
	}
	return targetEntities, nil
}

func (s *HTTPServer) populateShareForAuthor(ctx context.Context, authorID string, parentEntities []entity.Entity, ret *SharesSummaries, itemMu *[]sync.Mutex) error {
	shares, err := s.Duplo.GetSharesByAuthor(ctx, authorID, parentEntities)
	if err != nil {
		return err
	}
	for _, share := range shares.Items {
		foundParentEntity := false
		for idx := range ret.Items {
			if parentEntities[idx] == share.TargetEntity {
				(*itemMu)[idx].Lock()
				ret.Items[idx].UserIDs = ensureContains(ret.Items[idx].UserIDs, authorID)
				(*itemMu)[idx].Unlock()
				foundParentEntity = true
				break
			}
		}
		if !foundParentEntity {
			s.Log.Log("target_entity", share.TargetEntity, "This is bad, could not find target entity in return from duplo")
		}
	}
	return nil
}

// precheckUnshareByTarget is used only by unshareByTarget.  It does a beginning check to make sure the request looks ok
// at the start
func (s *HTTPServer) precheckUnshareByTarget(req *http.Request) (string, []entity.Entity, error) {
	// The author is who's posts you want to remove
	authorID := req.URL.Query().Get("author_id")
	// The user is who is asking to remove the posts
	userID := req.URL.Query().Get("user_id")

	targetEntities, err := entitiesFromURL(req, "target_entities")
	if err != nil {
		return "", nil, err
	}
	if userID == "" {
		return "", nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("missing parameter user_id"),
		}
	}
	if authorID == "" {
		return "", nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("missing parameter author_id"),
		}
	}
	canRemove, err := s.Authorization.CanUserDeleteShareForAuthor(req.Context(), userID, authorID)
	if err != nil {
		return "", nil, err
	}
	if !canRemove {
		return "", nil, &service_common.CodedError{
			Code: http.StatusForbidden,
			Err:  errors.Errorf("user %s cannot delete posts for %s", userID, authorID),
		}
	}
	return authorID, targetEntities, nil
}

func (s *HTTPServer) unshareByTarget(req *http.Request) (interface{}, error) {
	authorID, targetEntities, err := s.precheckUnshareByTarget(req)
	if err != nil {
		return nil, err
	}

	shares, err := s.Duplo.GetSharesByAuthor(req.Context(), authorID, targetEntities)
	if err != nil {
		return nil, err
	}
	ret := Shares{
		Items: make([]*Share, 0, len(shares.Items)),
	}
	eg, egCtx := errgroup.WithContext(req.Context())
	var mu sync.Mutex
	for _, share := range shares.Items {
		eg.Go(func() error {
			res, err := s.Duplo.DeleteShare(egCtx, share.ID)
			if err != nil {
				return err
			}
			mu.Lock()
			if res != nil {
				ret.Items = append(ret.Items, createShareFromDuploShare(res))
			}
			mu.Unlock()
			return nil
		})
	}
	if err := eg.Wait(); err != nil {
		return nil, err
	}
	return ret, nil
}

func (s *HTTPServer) getSharesSummaries(r *http.Request) (interface{}, error) {
	parentEntities, err := entitiesFromURL(r, "parent_entities")
	if err != nil {
		return nil, err
	}

	authorIDs := itemsFromQueryParam(r, "author_ids")

	ret := SharesSummaries{
		Items: make([]*ShareSummary, len(parentEntities)),
	}
	for idx := range ret.Items {
		ret.Items[idx] = &ShareSummary{}
	}
	eg, egCtx := errgroup.WithContext(r.Context())
	eg.Go(func() error {
		summaries, err := s.Duplo.GetSharesSummaries(egCtx, parentEntities)
		if err != nil {
			return err
		}
		for idx, summary := range summaries.Items {
			ret.Items[idx].ShareCount = summary.Total
			if summary.ParentEntity != parentEntities[idx] {
				return errors.New("assumptions wrong: Ids did not match")
			}
		}
		return nil
	})
	itemMu := make([]sync.Mutex, len(parentEntities))
	for _, authorID := range authorIDs {
		eg.Go(func() error {
			// Note: itemMu must by be & because mutex doesn't work if it's copied
			return s.populateShareForAuthor(egCtx, authorID, parentEntities, &ret, &itemMu)
		})
	}

	if err := eg.Wait(); err != nil {
		return nil, err
	}

	return ret, nil
}

func (s *HTTPServer) getShares(req *http.Request) (interface{}, error) {
	return s.APIv2.GetSharesShared(req.Context(), itemsFromQueryParam(req, "ids"))
}

func (s *HTTPServer) getShare(req *http.Request) (interface{}, error) {
	shareID := pat.Param(req, "share_id")

	duploShare, err := s.Duplo.GetShare(req.Context(), shareID)
	if err != nil {
		return nil, err
	}
	if duploShare == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errPostNotFound,
		}
	}

	share := createShareFromDuploShare(duploShare)
	return share, nil
}

func (s *HTTPServer) deleteShare(req *http.Request) (interface{}, error) {
	return s.APIv2.DeleteShareShared(req, pat.Param(req, "share_id"))
}

func (s *HTTPServer) urlsToEntities(ctx context.Context, embedURLs *[]string) (*[]entity.Entity, *[]string, error) {
	return s.APIv2.UrlsToEntities(ctx, embedURLs)
}

func (s *HTTPServer) createPostV1(req *http.Request) (bool, string, *Post, error) {
	postToTwitter, userID, duploPost, err := s.APIv2.CreatePostShared(req, false)
	if err != nil {
		return false, "", nil, err
	}

	post := createPostFromDuploPost(duploPost)
	if err := s.populatePosts(req.Context(), []*Post{post}, userID); err != nil {
		return false, "", nil, err
	}

	return postToTwitter, userID, post, nil
}

func (s *HTTPServer) createPost(req *http.Request) (interface{}, error) {
	_, _, post, err := s.createPostV1(req)
	return post, err
}

func (s *HTTPServer) createSyndicatedPost(req *http.Request) (interface{}, error) {
	postToTwitter, userID, post, err := s.createPostV1(req)
	if err != nil {
		return nil, err
	}

	tweetStatus := 0
	tweetMessage := ""
	if postToTwitter {
		tweetStatus, tweetMessage = s.APIv2.CreateTweetForPost(req.Context(), post.ID, post.Body, userID)
	}
	return createSyndicatedPostResponse{post, tweetStatus, tweetMessage}, nil
}

type createSyndicatedPostResponse struct {
	Post        *Post  `json:"post"`
	TweetStatus int    `json:"tweet_status"`
	Tweet       string `json:"tweet"`
}

func (s *HTTPServer) populateEmbedURLs(body string) *[]string {
	return s.APIv2.PopulateEmbedURLs(body)
}

func (s *HTTPServer) getComment(req *http.Request) (interface{}, error) {
	return nil, &service_common.CodedError{
		Code: http.StatusNotFound,
		Err:  errors.New("The requested resource could not be found but may be available in the future."),
	}
}

func (s *HTTPServer) deleteComment(req *http.Request) (interface{}, error) {
	return nil, &service_common.CodedError{
		Code: http.StatusNotFound,
		Err:  errors.New("The requested resource could not be found but may be available in the future."),
	}
}

func (s *HTTPServer) reportComment(req *http.Request) (interface{}, error) {
	return nil, &service_common.CodedError{
		Code: http.StatusNotFound,
		Err:  errors.New("The requested resource could not be found but may be available in the future."),
	}
}

func (s *HTTPServer) approveComment(req *http.Request) (interface{}, error) {
	return nil, &service_common.CodedError{
		Code: http.StatusNotFound,
		Err:  errors.New("The requested resource could not be found but may be available in the future."),
	}
}

func (s *HTTPServer) denyComment(req *http.Request) (interface{}, error) {
	return nil, &service_common.CodedError{
		Code: http.StatusNotFound,
		Err:  errors.New("The requested resource could not be found but may be available in the future."),
	}
}

func (s *HTTPServer) createComment(req *http.Request) (interface{}, error) {
	return nil, &service_common.CodedError{
		Code: http.StatusNotFound,
		Err:  errors.New("The requested resource could not be found but may be available in the future."),
	}
}

func (s *HTTPServer) getParentUserID(ctx context.Context, parentEntity entity.Entity) (string, error) {
	if parentEntity.Namespace() != entity.NamespacePost {
		return "", &service_common.CodedError{
			Code: http.StatusNotImplemented,
			Err:  fmt.Errorf("handling %s parent entities has not been implemented", parentEntity.Encode()),
		}
	}

	duploPost, err := s.loadAndBackfillDuploPost(ctx, parentEntity.ID())
	if err != nil {
		return "", err
	} else if duploPost == nil {
		return "", &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errPostNotFound,
		}
	}
	return duploPost.UserID, nil
}

func (s *HTTPServer) getCommentsForPost(req *http.Request) (interface{}, error) {
	return nil, &service_common.CodedError{
		Code: http.StatusNotFound,
		Err:  errors.New("The requested resource could not be found but may be available in the future."),
	}
}

func (s *HTTPServer) deleteCommentsByParentAndUser(req *http.Request) (interface{}, error) {
	return nil, &service_common.CodedError{
		Code: http.StatusNotFound,
		Err:  errors.New("The requested resource could not be found but may be available in the future."),
	}
}

func (s *HTTPServer) getReactionsByParents(req *http.Request) (interface{}, error) {
	parentEntities, err := entitiesFromQueryParam(req, "parent_entity")
	if err != nil {
		return nil, err
	}

	if len(parentEntities) > int(s.Config.getReactionsEntityLimit.Get()) {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Errorf("cannot exceed %v entities requested", s.Config.getReactionsEntityLimit.Get()),
		}
	}

	userID := req.URL.Query().Get("user_id")

	reactions := make([]*Reactions, len(parentEntities))
	havers := make([]ReactionHaver, len(parentEntities))
	for i, ent := range parentEntities {
		reactions[i] = &Reactions{
			ParentEntity: ent,
		}
		havers[i] = reactions[i]
	}
	if err := s.populateReactions(req.Context(), havers, []string{userID}); err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusInternalServerError,
			Err:  errors.Wrap(err, "could not get reactions"),
		}
	}

	ret := BatchReactionsResponse{
		Reactions: reactions,
	}

	return ret, nil
}

func normalizeClipEntityForReactions(ent entity.Entity) entity.Entity {
	// lowercase clip entity because clips creates reactions this way
	// https://git-aws.internal.justin.tv/video/clips-upload/blob/078418e03ee29325fff30de764fb2b4de34c0022/clients/feeds/client.go#L42
	return entity.New(ent.Namespace(), strings.ToLower(ent.ID()))
}

func (s *HTTPServer) createReaction(req *http.Request) (interface{}, error) {
	parentEntity, err := entity.Decode(pat.Param(req, "parent_entity"))
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "malformed parent_entity"),
		}
	}
	emoteID := pat.Param(req, "emote_id")
	if emoteID == "" {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("malformed or missing emote_id"),
		}
	}
	userID, err := requireUserID(req)
	if err != nil {
		return nil, err
	}

	if parentEntity.Namespace() == entity.NamespaceClip {
		parentEntity = normalizeClipEntityForReactions(parentEntity)
	}

	rs, err := s.Duplo.GetReactionsSummary(req.Context(), parentEntity)
	if err != nil {
		return nil, err
	}
	_, isExistingReaction := rs.EmoteSummaries[emoteID]

	canAddReaction, err := s.Authorization.CanUserAddReaction(req.Context(), userID, emoteID, isExistingReaction)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusInternalServerError,
			Err:  errors.Wrap(err, "could not check if user can add reaction"),
		}
	}
	if !canAddReaction {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("cannot use reaction"),
		}
	}

	duploReaction, err := s.Duplo.CreateReaction(req.Context(), parentEntity, userID, emoteID)
	if err != nil {
		return nil, err
	}

	s.SpadeClient.QueueEvents(spade.Event{
		Name: "feed_server_reaction",
		Properties: v2.ServerReactionTracking{
			Action:       "create",
			UserID:       userID,
			ReactionID:   emoteID,
			TargetEntity: parentEntity.Encode(),
			TargetType:   parentEntity.Namespace(),
			TargetID:     parentEntity.ID(),
		},
	})

	return createUserReactionsFromDuploReactions(duploReaction), nil
}

func (s *HTTPServer) deleteReaction(req *http.Request) (interface{}, error) {
	parentEntity, err := entity.Decode(pat.Param(req, "parent_entity"))
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "malformed parent_entity"),
		}
	}
	emoteID := pat.Param(req, "emote_id")
	if emoteID == "" {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("malformed or missing emote_id"),
		}
	}
	userID, err := requireUserID(req)
	if err != nil {
		return nil, err
	}

	if parentEntity.Namespace() == entity.NamespaceClip {
		parentEntity = normalizeClipEntityForReactions(parentEntity)
	}

	reactions, err := s.Duplo.DeleteReaction(req.Context(), parentEntity, userID, emoteID)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusInternalServerError,
			Err:  errors.Wrap(err, "could not delete reaction"),
		}
	} else if reactions == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("could not find reaction"),
		}
	}

	s.SpadeClient.QueueEvents(spade.Event{
		Name: "feed_server_reaction",
		Properties: v2.ServerReactionTracking{
			Action:       "remove",
			UserID:       userID,
			ReactionID:   emoteID,
			TargetEntity: parentEntity.Encode(),
			TargetType:   parentEntity.Namespace(),
			TargetID:     parentEntity.ID(),
		},
	})

	return createUserReactionsFromDuploReactions(reactions), nil
}

// Deprecated.  Use graphql for now on
func (s *HTTPServer) getFeed(req *http.Request) (interface{}, error) {
	start := time.Now()
	clientID := req.Header.Get("Twitch-Client-Id")

	feedID := pat.Param(req, "feed_id")
	userID := req.URL.Query().Get("user_id")
	cursor := req.URL.Query().Get("cursor")
	deviceID := req.URL.Query().Get("device_id")
	language := req.URL.Query().Get("language")

	if permissionsCheck := s.Authorization.CanUserGetFeed(userID, feedID); !permissionsCheck {
		return nil, &service_common.CodedError{
			Code: http.StatusUnauthorized,
			Err:  errors.Errorf("User %s not allowed to load feed %s", userID, feedID),
		}
	}

	limit, err := parseInt(req.URL.Query().Get("limit"), defaultPostLimit)
	if err != nil {
		return nil, err
	}

	// When limit is -1, we are only checking whether the feed is disabled.
	// Clients are expected to make another call to actually load the feed.
	if limit == -1 {
		s.Stats.IncC("get_feed.permission_check", 1, 1.0)
		// Since all feeds have been enabled for all users, we don't need to do an explicit check, and can
		// return an empty ChannelFeed struct, which has Disabled set to false.
		return &ChannelFeed{}, nil
	}

	if limit > int(s.Config.feedMaxLimit.Get()) {
		limit = int(s.Config.feedMaxLimit.Get())
	}

	if deviceID == "" {
		s.Stats.IncC(fmt.Sprintf("emptyDeviceID.%s", v2.LookupClientPlatform(clientID)), 1, 1.0)
	}

	masonryFeed, err := s.getMasonryFeed(req, feedID, &masonry.GetFeedOptions{
		Cursor:   cursor,
		Limit:    limit,
		DeviceID: deviceID,
		Language: language,
	})
	if err != nil {
		return nil, err
	} else if masonryFeed == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("feed not found"),
		}
	}

	feed, err := s.toChannelFeed(req.Context(), masonryFeed, userID)
	if err != nil {
		return nil, err
	}

	s.recordFeedStats(feedID, feed, false, cursor, start)

	return feed, nil
}

func (s *HTTPServer) getMasonryFeed(req *http.Request, feedID string, opts *masonry.GetFeedOptions) (*masonry.Feed, error) {
	if mockFeed := req.URL.Query().Get("mockFeed"); mockFeed != "" {
		return createMockMasonryFeed(mockFeed, feedID)
	}
	// return 410 with service_common.CodedError
	return nil, &service_common.CodedError{
		Code: http.StatusGone,
		Err:  errors.New("masonry is no more"),
	}
}

func (s *HTTPServer) recordFeedStats(feedID string, feed *ChannelFeed, disable bool, cursor string, start time.Time) {
	prefix := "get_feed.stats"
	if strings.HasPrefix(feedID, "n:") {
		prefix += ".news"
	} else if strings.HasPrefix(feedID, "c:") {
		prefix += ".channel"
	} else if strings.HasPrefix(feedID, "r:") {
		prefix += ".recommendation"
	} else {
		return
	}

	if disable {
		prefix += ".disabled"
	} else {
		prefix += ".enabled"
	}

	if cursor == "" {
		if feed == nil {
			s.Stats.TimingC(prefix+".num_posts", 0, 1.0)
		} else {
			s.Stats.TimingC(prefix+".num_posts", int64(len(feed.Posts)), 1.0)
			if len(feed.Posts) > 0 {
				age := time.Now().Unix() - feed.Posts[0].CreatedAt.Unix()
				s.Stats.TimingC(prefix+".min_age", age, 1.0)
			}
		}
	} else {
		s.Stats.IncC(prefix+".load_more", 1, 1.0)
	}

	s.Stats.TimingDurationC(prefix+".duration", time.Since(start), 1.0)
}

func (s *HTTPServer) getSettings(req *http.Request) (interface{}, error) {
	settingsEntity, err := s.requireValidSettingsEntity(req)
	if err != nil {
		return nil, err
	}

	userID, err := requireUserID(req)
	if err != nil {
		return nil, err
	}

	if !s.Authorization.CanUserTouchSettings(req.Context(), userID, settingsEntity.ID()) {
		return nil, &service_common.CodedError{
			Code: http.StatusForbidden,
			Err:  errors.New("unable to get settings"),
		}
	}

	settings, err := s.FeedSettingsClient.GetSettings(req.Context(), settingsEntity.Encode())
	if err != nil {
		return nil, err
	}
	ss := createSettingsFromFeedSettingsSettings(settings)

	// NOTE: this branch is hit if there are issues with the wexit migration of channel_feed_enabled
	if s.Config.rerouteToNuclearWaste.Get() {
		s.Stats.IncC("feed_settings.get_channel_feed_enabled", 1, 1.0)
		res, err := s.NuclearWaste.GetChannelFeedEnabled(req.Context(), userID, nil)
		if err != nil {
			s.Stats.IncC("feed_settings.get_channel_feed_enabled.error", 1, 1.0)
			return nil, err
		}
		ss.ChannelFeedEnabled = res.ChannelFeedEnabled
	}

	return ss, nil
}

type updateSettingsParams struct {
	SubsCanComment        *bool `json:"subs_can_comment,omitempty"`
	FriendsCanComment     *bool `json:"friends_can_comment,omitempty"`
	FollowersCanComment   *bool `json:"followers_can_comment,omitempty"`
	UserDisabledComments  *bool `json:"user_disabled_comments,omitempty"`
	AdminDisabledComments *bool `json:"admin_disabled_comments,omitempty"`
	ChannelFeedEnabled    *bool `json:"channel_feed_enabled,omitempty"`
}

func (s *HTTPServer) updateSettings(req *http.Request) (interface{}, error) {
	settingsEntity, err := s.requireValidSettingsEntity(req)
	if err != nil {
		return nil, err
	}

	userID, err := requireUserID(req)
	if err != nil {
		return nil, err
	}

	if !s.Authorization.CanUserTouchSettings(req.Context(), userID, settingsEntity.ID()) {
		return nil, &service_common.CodedError{
			Code: http.StatusForbidden,
			Err:  errors.New("unable to update settings"),
		}
	}

	var opts updateSettingsParams
	if err = json.NewDecoder(req.Body).Decode(&opts); err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "could not parse body"),
		}
	}

	// only pass requests that mutate ChannelFeedEnabled to block guarded by distconf variable
	if opts.ChannelFeedEnabled != nil && s.Config.rerouteToNuclearWaste.Get() {
		s.Stats.IncC("feed_settings.set_channel_feed_enabled", 1, 1.0)
		err = s.NuclearWaste.SetChannelFeedEnabled(req.Context(), userID, *opts.ChannelFeedEnabled, nil)
		if err != nil {
			s.Stats.IncC("feed_settings.set_channel_feed_enabled.error", 1, 1.0)
			return nil, err
		}
		return &Settings{ChannelFeedEnabled: *opts.ChannelFeedEnabled}, nil
	}

	settings, err := s.FeedSettingsClient.UpdateSettings(req.Context(), settingsEntity.Encode(), (*feedsettings.UpdateSettingsOptions)(&opts))
	if err != nil {
		return nil, err
	}
	return createSettingsFromFeedSettingsSettings(settings), nil
}

func (s *HTTPServer) getEmbed(req *http.Request) (interface{}, error) {
	var autoplay *bool
	embedURL := req.URL.Query().Get("url")
	if str := req.URL.Query().Get("autoplay"); str != "" {
		value, err := strconv.ParseBool(str)
		if err == nil {
			autoplay = &value
		}
	}
	embed, err := s.requestEmbed(req.Context(), embedURL, autoplay)
	if err != nil {
		return nil, err
	}
	if embed == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusUnprocessableEntity,
			Err:  errors.New("request url is not supported"),
		}
	}
	return embed, nil
}

func createMockMasonryFeed(mockFeed string, feedID string) (*masonry.Feed, error) {
	var jsonMasonryFeed masonry.Feed
	if err := json.Unmarshal([]byte(mockFeed), &jsonMasonryFeed); err == nil {
		return &jsonMasonryFeed, nil
	}

	entities := strings.Split(mockFeed, ",")
	items := make([]masonry.Activity, 0, len(entities))
	for _, item := range entities {
		ent, err := entity.Decode(item)
		if err != nil {
			return nil, err
		}
		items = append(items, masonry.Activity{Entity: ent})
	}
	feed := &masonry.Feed{
		Items: items,
		ID:    feedID,
	}
	return feed, nil
}

var loadableEntitiesConst = map[string]struct{}{
	entity.NamespacePost:   {},
	entity.NamespaceShare:  {},
	entity.NamespaceVod:    {},
	entity.NamespaceClip:   {},
	entity.NamespaceStream: {},
}

func (s *HTTPServer) toChannelFeed(ctx context.Context, masonryFeed *masonry.Feed, userID string) (*ChannelFeed, error) {
	items := make([]masonry.Activity, 0, len(masonryFeed.Items))
	for _, item := range masonryFeed.Items {
		if _, exists := loadableEntitiesConst[item.Entity.Namespace()]; exists {
			items = append(items, item)
		}
	}
	feed := &ChannelFeed{
		Cursor: masonryFeed.Cursor,
	}
	if shouldCacheFeedResponse(userID, masonryFeed.ID) {
		feed.cacheTime = s.Config.anonFeedCacheDuration.Get()
	}
	if err := s.populateFeed(ctx, feed, items, userID, masonryFeed.ID, masonryFeed.Tracking); err != nil {
		return nil, err
	}

	return feed, nil
}

func shouldCacheFeedResponse(requestingUser string, feedID string) bool {
	return requestingUser == "" && strings.HasPrefix(feedID, "c:")
}

func (s *HTTPServer) statDuration(stat string, start time.Time) {
	s.Stats.TimingDurationC(stat, time.Since(start), 1.0)
}

func (s *HTTPServer) drainPostUpdateChan(rootContext context.Context, shouldStopDraining <-chan struct{}) {
	for {
		select {
		case item := <-s.postUpdateChan:
			ctx, cancel := context.WithTimeout(rootContext, s.Config.duploUpdateTimeout.Get())
			duploEmotes := make([]duplo.Emote, 0, len(item.Emotes))
			for _, itemEmote := range item.Emotes {
				duploEmotes = append(duploEmotes, duplo.Emote{
					ID:    itemEmote.ID,
					Start: itemEmote.Start,
					End:   itemEmote.End,
					Set:   itemEmote.Set,
				})
			}
			err := s.Duplo.UpdatePost(ctx, item.PostID, duplo.UpdatePostOptions{
				Emotes: &duploEmotes,
			})
			s.Stats.IncC("drainPostUpdateChan.total", 1, 1.0)
			if err != nil {
				s.Stats.IncC("drainPostUpdateChan.err", 1, 1.0)
			}
			cancel()
		case <-shouldStopDraining:
			return
		}
	}
}

// Start begins serving HTTP content
func (s *HTTPServer) Start() error {
	wg, errCtx := errgroup.WithContext(context.Background())
	closedOnStop := make(chan struct{})
	wg.Go(func() error {
		defer close(closedOnStop)
		return s.BaseHTTPServer.Start()
	})
	wg.Go(func() error {
		s.drainPostUpdateChan(errCtx, closedOnStop)
		return nil
	})
	err := wg.Wait()
	s.Log.Log("err", err, "server finished listening")
	return err
}
