package api

import (
	"net/http"
	"strconv"
	"strings"
	"time"

	friendship "code.justin.tv/chat/friendship/client"
	clue "code.justin.tv/chat/tmi/client"
	"code.justin.tv/chat/tmi/clue/api"
	zumaapi "code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/client"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/feeds/clients/feed-settings"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-edge/cmd/feeds-edge/internal/emotes"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/foundation/twitchclient"
	"golang.org/x/net/context"
	"golang.org/x/sync/errgroup"
)

var (
	moderationCriteria = []string{
		friendship.ModCriterion,
		friendship.GlobalModCriterion,
		friendship.AdminCriterion,
		friendship.StaffCriterion,
	}

	deletionCriteria = []string{
		friendship.GlobalModCriterion,
		friendship.AdminCriterion,
		friendship.StaffCriterion,
	}
)

// FriendshipClient asserts what functions are needed from friendship.Client to allow mocking
type FriendshipClient interface {
	IsFamiliar(ctx context.Context, userID, targetID string, criteria []string, reqOpts *twitchhttp.ReqOpts) (string, bool, error)
}

var _ FriendshipClient = friendship.Client(nil)

// ClueClient asserts what functions are needed from clue.Client to allow mocking
type ClueClient interface {
	GetBanStatus(ctx context.Context, userID, channelID string, reqOpts *twitchclient.ReqOpts) (clue.ChannelBannedUser, bool, error)
	IsSubscriber(ctx context.Context, userID, channelID string, reqOpts *twitchclient.ReqOpts) (bool, error)
	GetUserType(ctx context.Context, userID int64, reqOpts *twitchclient.ReqOpts) (*clue.UserTypeResponse, error)
	AutoModCheckMessage(ctx context.Context, params api.AutoModCheckMessage, reqOpts *twitchclient.ReqOpts) (api.AutoModCheckMessageResult, error)
}

var _ ClueClient = clue.Client(nil)

// SettingsClient asserts what functions are needed from feedsettings.Client to allow mocking
type SettingsClient interface {
	GetSettings(ctx context.Context, entity string) (*feedsettings.Settings, error)
}

var _ SettingsClient = &feedsettings.Client{}

// Authorization is the container for connections to friendships + siteDB + ban
type Authorization struct {
	Log              *log.ElevatedLog
	Zuma             zuma.Client
	Stats            *service_common.StatSender
	FriendshipClient FriendshipClient
	ClueClient       ClueClient
	SettingsClient   SettingsClient
	Config           *HTTPConfig
}

func (a *Authorization) recordTimeSince(label string, start time.Time) {
	a.Stats.TimingDurationC(label, time.Since(start), 1.0)
}

type hasStatusCode interface {
	StatusCode() int
}

func (a *Authorization) isStaff(ctx context.Context, userID string) (bool, error) {
	userIDInt64, err := strconv.ParseInt(userID, 10, 64)
	if err != nil {
		return false, errors.Wrapf(err, "cannot parse userID %s", userID)
	}
	userType, err := a.ClueClient.GetUserType(ctx, userIDInt64, nil)

	if err != nil {
		// Users that don't exist (I'm not sure how they don't exist, but whatever) are not admins
		if statusCode, ok := err.(hasStatusCode); ok && statusCode.StatusCode() == http.StatusNotFound {
			return false, nil
		}
		return false, errors.Wrapf(err, "cannot get user type from clue for user %d", userIDInt64)
	}
	if userType == nil {
		return false, nil
	}
	return userType.IsStaff, nil
}

func (a *Authorization) CanUserCreatePostForAuthor(ctx context.Context, userID, authorID string) (bool, error) {
	return authorID == userID, nil
}

// Check if user can create a share for the author
func (a *Authorization) CanUserCreateShareForAuthor(ctx context.Context, userID, authorID string) (bool, error) {
	if userID == "" {
		return false, nil
	}
	return authorID != userID, nil
}

// Check if an user can delete a share
func (a *Authorization) CanUserDeleteShareForAuthor(ctx context.Context, userID, authorID string) (bool, error) {
	return authorID == userID, nil
}

func (a *Authorization) CanUserDeletePostForAuthor(ctx context.Context, userID, authorID string) (bool, error) {
	defer a.recordTimeSince("can_delete", time.Now())
	if userID == "" || authorID == "" {
		return false, nil
	}

	if userID == authorID {
		return true, nil
	}

	_, canDelete, err := a.FriendshipClient.IsFamiliar(ctx, userID, authorID, deletionCriteria, nil)
	if err != nil {
		return false, errors.Wrap(err, "cannot check is familiar for post delete")
	}

	return canDelete, nil
}

func (a *Authorization) canUserModeratePostForAuthor(ctx context.Context, userID, authorID string) (bool, error) {
	defer a.recordTimeSince("can_moderate", time.Now())
	if userID == "" || authorID == "" {
		return false, nil
	}

	if userID == authorID {
		return true, nil
	}

	_, canModerate, err := a.FriendshipClient.IsFamiliar(ctx, userID, authorID, moderationCriteria, nil)
	if err != nil {
		return false, errors.Wrap(err, "cannot check IsFamiliar for can moderate")
	}

	return canModerate, nil
}

type relationsBag struct {
	IsSubscriber bool
	IsFamiliar   bool
	IsPartnered  bool
}

// CanUserGetFeed returns true if the requestingUser can load feed feedID
func (a *Authorization) CanUserGetFeed(requestingUser string, feedID string) bool {
	if strings.HasPrefix(feedID, "c:") {
		// Anyone can load any channel feed
		return true
	}
	if strings.HasPrefix(feedID, "r:") {
		// users can only load their own recommended feed
		return requestingUser == feedID[2:]
	}
	if strings.HasPrefix(feedID, "n:") {
		// users can only load their own news feed
		return requestingUser == feedID[2:]
	}
	return true
}

func (a *Authorization) addSubCheckFunction(ctx context.Context, userID string, authorID string, relations *relationsBag) func() error {
	return func() error {
		defer a.recordTimeSince("can_reply.checking_subscriber", time.Now())
		isSubscriber, err := a.ClueClient.IsSubscriber(ctx, userID, authorID, nil)
		if err != nil {
			a.Stats.IncC("is_subscriber.error", 1, 1.0)
			if strings.Contains(err.Error(), "EOF") {
				// ... this error is terrible and happens all the time.  Just assume they're
				// not subscribers if we get an EOF error
				relations.IsSubscriber = false
				return nil
			}
			return errors.Wrap(err, "error if clueclient can subscribe")
		}
		relations.IsSubscriber = isSubscriber

		return nil
	}
}

// CanUserUseEmote returns true if the given userID can use the given emoteID
func (a *Authorization) CanUserUseEmote(ctx context.Context, userID, emoteID string) (bool, error) {
	if emoteID == emotes.EndorseEmote {
		return true, nil
	}
	resp, err := a.Zuma.CheckEmoticonsAccess(ctx, zumaapi.CheckEmoticonsAccessRequest{
		UserID:      userID,
		EmoticonIDs: []string{emoteID},
	}, nil)

	if err != nil {
		if twitchHTTPError, ok := err.(*twitchhttp.Error); ok {
			if twitchHTTPError.StatusCode == http.StatusUnprocessableEntity {
				a.Log.LogCtx(ctx, "emoteID", emoteID, "invalid emote")
				return false, nil
			}
		}
		return false, errors.Wrap(err, "unable to check emoticon access")
	}

	// If the user doesn't have access, this will be non empty
	return len(resp.NoAccessIDs) == 0, nil
}

// CanUserAddReaction returns true if the given userID can use the given emoteID for the given reactions
func (a *Authorization) CanUserAddReaction(ctx context.Context, userID, emote string, isExistingReaction bool) (bool, error) {
	if isExistingReaction {
		// if someone has already used this reaction then anyone can also use the reaction
		return true, nil
	}
	return a.CanUserUseEmote(ctx, userID, emote)
}

// CanUserTouchSettings returns true if the given userID can get or update the given settingsUserID
func (a *Authorization) CanUserTouchSettings(ctx context.Context, userID, settingsUserID string) bool {
	return userID == settingsUserID
}

// GetPostPermissionsForUserToAuthor returns a PostPermissions struct for a user to a given author
func (a *Authorization) GetPostPermissionsForUserToAuthor(ctx context.Context, userID, authorID string) (*PostPermissions, error) {
	defer a.recordTimeSince("post_permissions", time.Now())
	g, ctx := errgroup.WithContext(ctx)
	permissions := &PostPermissions{}

	g.Go(func() error {
		canShare, err := a.CanUserCreateShareForAuthor(ctx, userID, authorID)
		if err != nil {
			return errors.Wrap(err, "cannot get post for author")
		}
		permissions.CanShare = canShare
		return nil
	})
	g.Go(func() error {
		canDelete, err := a.CanUserDeletePostForAuthor(ctx, userID, authorID)
		if err != nil {
			return errors.Wrap(err, "cannot get can user delete")
		}
		permissions.CanDelete = canDelete
		return nil
	})

	g.Go(func() error {
		canModerate, err := a.canUserModeratePostForAuthor(ctx, userID, authorID)
		if err != nil {
			return errors.Wrap(err, "cannot get can user moderate")
		}
		permissions.CanModerate = canModerate
		return nil
	})

	err := g.Wait()
	if err != nil {
		return nil, err
	}
	return permissions, nil
}
