package api

import (
	"fmt"
	"testing"
	"time"

	clue "code.justin.tv/chat/tmi/client"
	"code.justin.tv/feeds/feeds-edge/cmd/feeds-edge/internal/api/mocks"
	"code.justin.tv/feeds/service-common"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
	"golang.org/x/net/context"
)

const (
	testUser   = "123"
	testTarget = "456"
)

// IsBan IsSub IsFriend IsMod IsAdmin AdminDisabled UserDisabled | CanReply CanModerate CanDelete

var (
	defaultChannelBannedUser = clue.ChannelBannedUser{
		BannedUserID: 0,
		BannedAt:     time.Now(),
		ExpiresAt:    optTime(time.Now()),
	}
)

type mockSet struct {
	ClueClient       *mocks.ClueClient
	FriendshipClient *mocks.FriendshipClient
	SettingsClient   *mocks.SettingsClient
}

type conditionSet struct {
	IsSubscriber bool
	IsBanned     bool
	IsFriend     bool
	IsMod        bool
	IsAdmin      bool
	IsFollower   bool
	IsPartnered  bool
}

type resultSet struct {
	CanReply    bool
	CanModerate bool
	CanDelete   bool
	Error       error
}

func optTime(t time.Time) *time.Time {
	ptr := new(time.Time)

	bs, err := t.MarshalJSON()
	if err != nil {
		return nil
	}

	if err := ptr.UnmarshalJSON(bs); err != nil {
		return nil
	}
	return ptr
}

func setupModule(mockset mockSet, conditions conditionSet) *Authorization {
	return &Authorization{
		Stats: service_common.NopStatSender(),

		FriendshipClient: mockset.FriendshipClient,
		ClueClient:       mockset.ClueClient,
		SettingsClient:   mockset.SettingsClient,
	}
}

func setMocksForConditions(mockset mockSet, conditions conditionSet) {
	mockset.ClueClient.On("GetBanStatus",
		mock.Anything,
		testUser,
		testTarget,
		mock.Anything,
	).Return(defaultChannelBannedUser, conditions.IsBanned, nil)

	mockset.ClueClient.On("IsSubscriber",
		mock.Anything,
		testUser,
		testTarget,
		mock.Anything,
	).Return(conditions.IsSubscriber, nil)

	mockset.FriendshipClient.On("IsFamiliar",
		mock.Anything,
		testUser,
		testTarget,
		moderationCriteria,
		mock.Anything,
	).Return("", conditions.IsMod || conditions.IsAdmin, nil)

	mockset.FriendshipClient.On("IsFamiliar",
		mock.Anything,
		testUser,
		testTarget,
		deletionCriteria,
		mock.Anything,
	).Return("", conditions.IsAdmin, nil)
}

func testConditionSet(title string, mockset mockSet, conditions conditionSet, results resultSet) {
	Convey(fmt.Sprintf("%s | %+v => %+v", title, conditions, results), func() {
		setMocksForConditions(mockset, conditions)

		auth := setupModule(mockset, conditions)
		ctx := context.Background()

		permissions, err := auth.GetPostPermissionsForUserToAuthor(ctx, testUser, testTarget)

		So(permissions.CanModerate, ShouldEqual, results.CanModerate)
		So(permissions.CanDelete, ShouldEqual, results.CanDelete)
		So(err, ShouldEqual, results.Error)
	})
}

func TestAllConditionSets(t *testing.T) {

	Convey("Testing Post Permissions", t, func() {
		mockset := mockSet{
			SettingsClient:   new(mocks.SettingsClient),
			ClueClient:       new(mocks.ClueClient),
			FriendshipClient: new(mocks.FriendshipClient),
		}

		testConditionSet("No relations makes you unable to reply", mockset,
			conditionSet{
				IsFriend:     false,
				IsMod:        false,
				IsAdmin:      false,
				IsSubscriber: false,
				IsBanned:     false,
			},
			resultSet{
				CanReply:    false,
				CanModerate: false,
				CanDelete:   false,
			},
		)

		testConditionSet("Moderators can reply and moderate posts", mockset,
			conditionSet{
				IsFriend:     false,
				IsMod:        true,
				IsAdmin:      false,
				IsSubscriber: false,
				IsBanned:     false,
			},
			resultSet{
				CanReply:    true,
				CanModerate: true,
				CanDelete:   false,
			},
		)

		testConditionSet("Admins can delete posts always", mockset,
			conditionSet{
				IsFriend:     false,
				IsMod:        false,
				IsAdmin:      true,
				IsSubscriber: false,
				IsBanned:     true,
			},
			resultSet{
				CanReply:    false,
				CanModerate: true,
				CanDelete:   true,
			},
		)
	})
}
