package api

import (
	"code.justin.tv/feeds/clients/duplo"
	"code.justin.tv/feeds/clients/masonry"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/feeds-common/entity"
	"golang.org/x/net/context"
)

// A DuploCacher is something that can populate entities from duplo and return an easy to use duploCache struct for
// looking up information you may want.
type DuploCacher struct {
	Duplo             *duplo.Client
	DuploCacherConfig *DuploCacherConfig
}

// DuploCacherConfig configures DuploCacher
type DuploCacherConfig struct {
	// Can turn this off to stop loading share info for normal posts.  It will assume shares never happened.  It
	// can be used to turn off load on duplo for an emergency.
	LoadShareInfo *distconf.Bool
}

// Load config information from distconf
func (d *DuploCacherConfig) Load(dconf *distconf.Distconf) error {
	d.LoadShareInfo = dconf.Bool("feeds-edge.sharecount_in_feed", true)
	return nil
}

type duploCache struct {
	posts  map[entity.Entity]*duplo.Post
	shares map[entity.Entity]*duplo.Share
}

func (d *DuploCacher) populateDuploCache(ctx context.Context, items []masonry.Activity, sharedByUsers []string) (duploCache, error) {
	if len(items) == 0 {
		return duploCache{}, nil
	}
	// There are two steps
	// Step (1): Load shares and what they point to
	shareToEntity, err := d.loadEntityShares(ctx, items)
	if err != nil {
		return duploCache{}, err
	}
	// Step (2): Load all the posts (and pointed to posts by a share)
	posts, err := d.loadEntityPosts(ctx, items, shareToEntity, sharedByUsers)
	if err != nil {
		return duploCache{}, err
	}
	return duploCache{
		posts:  posts,
		shares: shareToEntity,
	}, nil
}

func findPostIDs(items []masonry.Activity, shares map[entity.Entity]*duplo.Share) []string {
	allPostIDs := make([]string, 0, len(items))
	for _, item := range items {
		if item.Entity.Namespace() == entity.NamespacePost {
			allPostIDs = append(allPostIDs, item.Entity.ID())
			continue
		}
		if item.Entity.Namespace() == entity.NamespaceShare && shares != nil {
			ent, exists := shares[item.Entity]
			if exists && ent.TargetEntity.Namespace() == entity.NamespacePost {
				allPostIDs = append(allPostIDs, ent.TargetEntity.ID())
			}
			continue
		}
	}
	return allPostIDs
}

func (d *DuploCacher) loadEntityPosts(ctx context.Context, items []masonry.Activity, shares map[entity.Entity]*duplo.Share, sharedByUsers []string) (map[entity.Entity]*duplo.Post, error) {
	allPostIDs := findPostIDs(items, shares)
	if len(allPostIDs) == 0 {
		return nil, nil
	}
	var options *duplo.GetPostsOptions
	if d.DuploCacherConfig.LoadShareInfo.Get() {
		options = &duplo.GetPostsOptions{
			Shares:     true,
			ShareUsers: sharedByUsers,
		}
	}
	dbPosts, err := d.Duplo.GetPostsWithOptions(ctx, allPostIDs, options)
	if err != nil {
		return nil, err
	}
	ret := make(map[entity.Entity]*duplo.Post, len(dbPosts.Items))
	for _, item := range dbPosts.Items {
		ret[entity.New(entity.NamespacePost, item.ID)] = item
	}
	return ret, nil
}

func (d *DuploCacher) loadEntityShares(ctx context.Context, items []masonry.Activity) (map[entity.Entity]*duplo.Share, error) {
	if len(items) == 0 {
		return nil, nil
	}
	shareIDs := make([]string, 0, len(items))
	for _, share := range items {
		if share.Entity.Namespace() == entity.NamespaceShare {
			shareIDs = append(shareIDs, share.Entity.ID())
		}
	}
	if len(shareIDs) == 0 {
		return nil, nil
	}
	shareReturn, err := d.Duplo.GetShares(ctx, shareIDs)
	if err != nil {
		return nil, err
	}
	ret := make(map[entity.Entity]*duplo.Share, len(shareReturn.Items))
	for _, share := range shareReturn.Items {
		if share.DeletedAt == nil {
			ret[entity.New(entity.NamespaceShare, share.ID)] = share
		}
	}
	return ret, nil
}
