package api

import (
	"code.justin.tv/feeds/log"
	users "code.justin.tv/web/users-service/client"
	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"
)

type stringSet map[string]struct{}

func (u stringSet) add(userID string) {
	u[userID] = struct{}{}
}

func (u stringSet) contains(userID string) bool {
	_, contains := u[userID]
	return contains
}

func (u stringSet) isEmpty() bool {
	return len(u) == 0
}

func (u stringSet) merge(from stringSet) {
	for id := range from {
		u[id] = struct{}{}
	}
}

func (u stringSet) removeAll(from stringSet) {
	for id := range from {
		delete(u, id)
	}
}

func (u stringSet) toArray() []string {
	ret := make([]string, 0, len(u))
	for s := range u {
		ret = append(ret, s)
	}
	return ret
}

// Filter will remove posts from users that shouldn't be in people's feed: for example DMCA or
// ToS banned users
type Filter struct {
	UsersClient users.Client
	Log         *log.ElevatedLog
}

// FilterPost will return nil if the post author is filtered, or will return the post filtered by bad users
func (f *Filter) FilterPost(ctx context.Context, post *Post, knownValidUsers stringSet) *Post {
	posts := f.FilterPosts(ctx, []*Post{post}, knownValidUsers)
	if len(posts) != 1 {
		return nil
	}
	return posts[0]
}

// FilterPosts returns the posts that we are allowed to show, with shares filtered as well
func (f *Filter) FilterPosts(ctx context.Context, posts []*Post, knownValidUsers stringSet) []*Post {
	allUserIDs := usersFromPosts(posts)
	validUsers := f.loadValidUsers(ctx, allUserIDs, knownValidUsers)
	newPosts := make([]*Post, 0, len(posts))
	for _, originalPost := range posts {
		if _, isValid := validUsers[originalPost.UserID]; !isValid {
			continue
		}
		post := *originalPost
		shareUsers := make([]string, 0, len(post.ShareSummary.UserIDs))
		for _, sharedUser := range post.ShareSummary.UserIDs {
			if _, isValid := validUsers[sharedUser]; !isValid {
				continue
			}
			shareUsers = append(shareUsers, sharedUser)
		}
		post.ShareSummary.UserIDs = shareUsers
		newPosts = append(newPosts, &post)
	}
	return newPosts
}

func (f *Filter) loadValidUsers(ctx context.Context, users stringSet, knownValidUsers stringSet) stringSet {
	users.removeAll(knownValidUsers)
	if users.isEmpty() {
		return knownValidUsers
	}
	params := &models.FilterParams{
		NotDeleted:      true,
		NoTOSViolation:  true,
		NoDMCAViolation: true,
		IDs:             users.toArray(),
	}
	res, err := f.UsersClient.GetUsers(ctx, params, nil)
	if err != nil {
		// If user service is down, just assume all the users are valid
		f.Log.LogCtx(ctx, "len(users)", len(users), "err", err)
		return users
	}
	validIDs := stringSet(make(map[string]struct{}, len(res.Results)))
	for _, result := range res.Results {
		validIDs.add(result.ID)
	}
	validIDs.merge(knownValidUsers)
	return validIDs
}

func usersFromPosts(posts []*Post) stringSet {
	allUserIDs := stringSet(make(map[string]struct{}, len(posts)))
	for _, post := range posts {
		allUserIDs.add(post.UserID)
		for _, sharedUsers := range post.ShareSummary.UserIDs {
			allUserIDs.add(sharedUsers)
		}
	}
	return allUserIDs
}
