package api

import (
	"fmt"
	"math"
	"strings"
	"time"

	"code.justin.tv/chat/emoticons/models"
	"code.justin.tv/feeds/clients/duplo"
	"code.justin.tv/feeds/clients/feed-settings"
	"code.justin.tv/feeds/clients/masonry"
	"code.justin.tv/feeds/feeds-common/entity"
	"golang.org/x/net/context"
)

// Filterable items
var _ canFilter = &ChannelFeed{}
var _ canFilter = &Post{}

// PostTracking contains post-level fields used for tracking
type PostTracking struct {
	RecGenerationID    string `json:"rec_generation_id,omitempty"`
	RecGenerationIndex *int   `json:"rec_generation_index,omitempty"`
	CardImpressionID   string `json:"card_impression_id,omitempty"`
	BatchID            string `json:"batch_id,omitempty"`
}

// ChannelFeed is the response payload for a channel feed request or news feed request
type ChannelFeed struct {
	Disabled bool    `json:"disabled,omitempty"`
	Cursor   string  `json:"cursor"`
	Posts    []*Post `json:"posts,omitempty"`

	// private variables used to know if a channel feed response should be cached
	cacheTime time.Duration
}

// Headers returns HTTP headers the ChannelFeed should use if the feed is safe to cache in a response
func (c *ChannelFeed) Headers() map[string]string {
	cacheTime := int(math.Ceil(c.cacheTime.Seconds()))
	if cacheTime == 0 {
		return nil
	}
	return map[string]string{
		"Cache-Control": fmt.Sprintf("public, max-age=%d", cacheTime),
	}
}

// Headers returns HTTP headers the ChannelFeed should use if the feed is safe to cache in a response
func (c *ChannelFeed) filter(ctx context.Context, f *Filter, knownIDs stringSet) interface{} {
	c.Posts = f.FilterPosts(ctx, c.Posts, knownIDs)
	return c
}

// Share is the returned object for a database share
type Share struct {
	ID           string        `json:"id"`
	UserID       string        `json:"user_id"`
	TargetEntity entity.Entity `json:"target_entity"`
	CreatedAt    time.Time     `json:"created_at"`
	DeletedAt    *time.Time    `json:"deleted_at,omitempty"`
}

// Shares is the JSON payload for multiple share objects
type Shares struct {
	Items []*Share
}

// postDebugInfo isn't a stable API and is used only by the feeds team
type postDebugInfo struct {
	Source masonry.Activity `json:"source"`
}

// Post is the JSON payload for a post object
type Post struct {
	ID            string           `json:"id"`
	Type          string           `json:"type"`
	CreatedAt     time.Time        `json:"created_at"`
	Deleted       bool             `json:"deleted,omitempty"`
	UserID        string           `json:"user_id"`
	Body          string           `json:"body"`
	EmbedEntities *[]entity.Entity `json:"embed_entities,omitempty"`
	// Deprecated: Use EmbedEntities
	EmbedURLs    *[]string        `json:"embed_urls,omitempty"`
	Emotes       []*Emote         `json:"emotes"`
	Embeds       []*Embed         `json:"embeds"`
	Permissions  *PostPermissions `json:"permissions"`
	Comments     *Comments        `json:"comments,omitempty"`
	Reactions    ReactionMap      `json:"reactions"`
	ShareSummary ShareSummary     `json:"share_summary"`

	// If Go had templates, this would be T<? embeds ReasonCommon>
	// This is like this for backwards compatibility.  I really wish it wasn't on the Post honestly.  If you fetch
	// a feed, there is a reason.  If you fetch the post directly, it is not here.
	Reasons []*ReasonSuperStruct `json:"reasons,omitempty"`

	// Tracking object on the post level.  Used for client-side spade tracking events
	Tracking *PostTracking `json:"tracking,omitempty"`

	// Only included in debugging requests.  Allows us to write tools that use feeds better
	Debug *postDebugInfo `json:"debug,omitempty"`
}

func (p *Post) filter(ctx context.Context, f *Filter, knownIDs stringSet) interface{} {
	return f.FilterPost(ctx, p, knownIDs)
}

// ReasonTypeEnum is the set of reason types we expose for feed items
type ReasonTypeEnum string

const (
	// ShareCreatedType is the Post.Reason.Type for created shares
	ShareCreatedType ReasonTypeEnum = "share_created"

	// PostCreatedType is the Post.Reason.Type for created posts
	PostCreatedType ReasonTypeEnum = "post_created"

	// FollowedType indicates that a piece of content is in a user's feed because the user
	// followed the content creator.
	FollowedType ReasonTypeEnum = "followed"

	// ViewedType indicates that a piece of content is in a user's feed because the user
	// viewed similar content.
	ViewedType ReasonTypeEnum = "viewed"

	// PopularType indicates that a piece of content is in a user's feed because it is
	// popular on Twitch.
	PopularType ReasonTypeEnum = "popular"
)

// ReasonSuperStruct should be every reason that can happen.  Honestly, this is the easiest way to do this in Go
type ReasonSuperStruct struct {
	Type      ReasonTypeEnum `json:"type"`
	UserID    string         `json:"user_id,omitempty"`
	CreatedAt *time.Time     `json:"created_at,omitempty"`
}

// ShareSummary contains summary information about shares
type ShareSummary struct {
	UserIDs    []string `json:"user_ids"`
	ShareCount int      `json:"share_count"`
}

// SharesSummaries is a JSON payload for multiple share summaries
type SharesSummaries struct {
	Items []*ShareSummary
}

// Embed contains information related to the RequestURL
type Embed struct {
	RequestURL      string      `json:"request_url,omitempty"`
	URL             string      `json:"url,omitempty"`
	Type            string      `json:"type,omitempty"`
	Title           string      `json:"title,omitempty"`
	Description     string      `json:"description,omitempty"`
	AuthorName      string      `json:"author_name,omitempty"`
	AuthorURL       string      `json:"author_url,omitempty"`
	AuthorThumbnail string      `json:"author_thumbnail_url,omitempty"`
	AuthorID        string      `json:"author_id,omitempty"`
	AuthorLogin     string      `json:"author_login,omitempty"`
	ThumbnailURL    string      `json:"thumbnail_url,omitempty"`
	Thumbnails      *Thumbnails `json:"thumbnail_urls,omitempty"`
	PlayerHTML      string      `json:"player_html,omitempty"`
	ProviderName    string      `json:"provider_name,omitempty"`

	// Twitch VOD/Clips specific
	CreatedAt       *time.Time `json:"created_at,omitempty"`
	Game            string     `json:"game,omitempty"`
	VideoLength     int        `json:"video_length,omitempty"`
	ViewCount       int        `json:"view_count,omitempty"`
	TwitchType      string     `json:"twitch_type,omitempty"`
	TwitchContentID string     `json:"twitch_content_id,omitempty"`
	StartTime       *time.Time `json:"start_time,omitempty"`
	EndTime         *time.Time `json:"end_time,omitempty"`

	// Twitch Stream specific
	StreamType string `json:"stream_type,omitempty"`
}

// Thumbnails provides additional thumbnail URLs
type Thumbnails struct {
	Large  string `json:"large,omitempty"`
	Medium string `json:"medium,omitempty"`
	Small  string `json:"small,omitempty"`
	Tiny   string `json:"tiny,omitempty"`
}

// GetEntity gets the entity for the post
func (p *Post) GetEntity() entity.Entity {
	ent, err := entity.Decode(p.ID)
	if err != nil {
		return entity.New(entity.NamespacePost, p.ID)
	}

	return ent
}

// GetBody gets a post's body
func (p *Post) GetBody() string { return p.Body }

// GetUserID gets a post's user ID
func (p *Post) GetUserID() string { return p.UserID }

// SetEmotes sets a post's emotes
func (p *Post) SetEmotes(es []*Emote) { p.Emotes = es }

// SetReactions sets a post's reactions
func (p *Post) SetReactions(r ReactionMap) { p.Reactions = r }

// Emote is the emote object for the API
type Emote struct {
	ID    int `json:"id"`
	Start int `json:"start"`
	End   int `json:"end"`
	Set   int `json:"set"`
}

// Comment is the comment object for the API
type Comment struct {
	ID            string              `json:"id"`
	UserID        string              `json:"user_id"`
	Body          string              `json:"body"`
	ParentEntity  entity.Entity       `json:"parent_entity"`
	CreatedAt     time.Time           `json:"created_at"`
	Deleted       bool                `json:"deleted,omitempty"`
	NeedsApproval bool                `json:"needs_approval"`
	Emotes        []*Emote            `json:"emotes"`
	Permissions   *CommentPermissions `json:"permissions"`
	Reactions     ReactionMap         `json:"reactions"`
}

// GetEntity gets the entity for the comment
func (c *Comment) GetEntity() entity.Entity { return entity.New(entity.NamespaceComment, c.ID) }

// GetBody gets a comment's body
func (c *Comment) GetBody() string { return c.Body }

// GetUserID gets a comment's user ID
func (c *Comment) GetUserID() string { return c.UserID }

// SetEmotes sets a comment's emotes
func (c *Comment) SetEmotes(es []*Emote) { c.Emotes = es }

// SetReactions sets a post's reactions
func (c *Comment) SetReactions(r ReactionMap) { c.Reactions = r }

// Entity represents a parent entity
// Deprecated: 1/24/17 Moved to feeds-common
type Entity struct {
	Type string `json:"type"`
	ID   string `json:"id"`
}

func (e *Entity) toString() string {
	return e.Type + ":" + e.ID
}

// Comments is the comment object that is returned for paginated comments
type Comments struct {
	ParentEntity entity.Entity `json:"-"`
	Total        int           `json:"total"`
	Cursor       string        `json:"cursor"`
	Comments     []*Comment    `json:"comments"`
}

// UserReactions is the object for user reactions
type UserReactions struct {
	UserID   string   `json:"user_id"`
	EmoteIDs []string `json:"emote_ids"`
}

// BatchReactionsResponse is the response to a batch reactions request
type BatchReactionsResponse struct {
	Reactions []*Reactions `json:"items"`
}

// Reactions is the collection of all reactions for the given parent entity
type Reactions struct {
	ParentEntity entity.Entity `json:"parent_entity"`
	Reactions    ReactionMap   `json:"reactions"`
}

// GetEntity gets the entity for the comment
func (t *Reactions) GetEntity() entity.Entity { return t.ParentEntity }

// SetReactions sets reactions
func (t *Reactions) SetReactions(r ReactionMap) { t.Reactions = r }

// ReactionMap is a map of emote ids to reaction items
type ReactionMap map[string]*ReactionItem

// ReactionItem is the count for a particular emote on a given parent entity
type ReactionItem struct {
	Emote   string   `json:"emote"`
	Count   int      `json:"count"`
	UserIDs []string `json:"user_ids"`
}

// PostPermissions are the permissions on a post
type PostPermissions struct {
	CanReply    bool `json:"can_reply"`
	CanDelete   bool `json:"can_delete"`
	CanModerate bool `json:"can_moderate"`
	CanShare    bool `json:"can_share"`
}

// CommentPermissions are the permissions on a comment
type CommentPermissions struct {
	CanDelete bool `json:"can_delete"`
}

// Settings contains user settings
type Settings struct {
	Entity                string    `json:"entity"` // User ID or feed ID
	CreatedAt             time.Time `json:"created_at"`
	UpdatedAt             time.Time `json:"updated_at"`
	SubsCanComment        bool      `json:"subs_can_comment"`
	FriendsCanComment     bool      `json:"friends_can_comment"`
	FollowersCanComment   bool      `json:"followers_can_comment"`
	UserDisabledComments  bool      `json:"user_disabled_comments"`
	AdminDisabledComments bool      `json:"admin_disabled_comments"`
	ChannelFeedEnabled    bool      `json:"channel_feed_enabled"`
}

func emotePtr(emote Emote) *Emote {
	return &emote
}

func createEmotes(matches []emoticonmodels.Match) []*Emote {
	emotes := make([]*Emote, len(matches))
	for i, match := range matches {
		emotes[i] = emotePtr(Emote(match))
	}
	return emotes
}

func createShareFromDuploShare(dShare *duplo.Share) *Share {
	return &Share{
		ID:           dShare.ID,
		UserID:       dShare.UserID,
		TargetEntity: dShare.TargetEntity,
		CreatedAt:    dShare.CreatedAt,
		DeletedAt:    dShare.DeletedAt,
	}
}

func createPostFromDuploPost(dPost *duplo.Post) *Post {
	return &Post{
		ID:            dPost.ID,
		Type:          "post",
		CreatedAt:     dPost.CreatedAt,
		UserID:        dPost.UserID,
		Body:          dPost.Body,
		Deleted:       dPost.DeletedAt != nil,
		EmbedURLs:     dPost.EmbedURLs,
		EmbedEntities: dPost.EmbedEntities,
		Emotes:        emotesFromDuploEmotes(dPost.Emotes),
		ShareSummary: ShareSummary{
			UserIDs:    []string{},
			ShareCount: 0,
		},
	}
}

func createPostFromVod(ent entity.Entity) *Post {
	shineVodURL := fmt.Sprintf("https://www.twitch.tv/videos/%s", ent.ID())
	ret := &Post{
		ID:            ent.Encode(),
		Type:          "vod",
		EmbedURLs:     &[]string{shineVodURL},
		EmbedEntities: &[]entity.Entity{ent},
	}
	return ret
}

func createPostFromClip(ent entity.Entity) *Post {
	shineClipURL := fmt.Sprintf("https://clips.twitch.tv/%s", ent.ID())
	ret := &Post{
		ID:            ent.Encode(),
		Type:          "clip",
		EmbedURLs:     &[]string{shineClipURL},
		EmbedEntities: &[]entity.Entity{ent},
	}
	return ret
}

func createPostFromStream(ent entity.Entity) *Post {
	streamID := ent.ID()
	// Stream id's come in the form of channelID-broadcastID
	idSplit := strings.Split(streamID, ":")

	if len(idSplit) != 2 {
		return nil
	}
	shineStreamURL := fmt.Sprintf("https://www.twitch.tv/broadcasts/%s/channel/%s", idSplit[1], idSplit[0])
	return &Post{
		ID:            ent.Encode(),
		Type:          "stream",
		EmbedURLs:     &[]string{shineStreamURL},
		EmbedEntities: &[]entity.Entity{ent},
	}
}

func emotesFromDuploEmotes(emotes *[]duplo.Emote) []*Emote {
	if emotes == nil || *emotes == nil {
		return nil
	}
	ret := make([]*Emote, len(*emotes))
	for i, emote := range *emotes {
		ret[i] = emotePtr(Emote(emote))
	}
	return ret
}

func createCommentFromDuploComment(dComment *duplo.Comment) *Comment {
	return &Comment{
		ID:            dComment.ID,
		ParentEntity:  dComment.ParentEntity,
		UserID:        dComment.UserID,
		Body:          dComment.Body,
		CreatedAt:     dComment.CreatedAt,
		Deleted:       dComment.DeletedAt != nil,
		NeedsApproval: dComment.NeedsApproval,
		Emotes:        emotesFromDuploEmotes(dComment.Emotes),
	}
}

func createCommentsFromDuploComments(parentEntity entity.Entity, dComments *duplo.Comments) *Comments {
	comments := make([]*Comment, len(dComments.Items))

	for i, dComment := range dComments.Items {
		comments[i] = createCommentFromDuploComment(dComment)
	}

	return &Comments{
		Comments:     comments,
		ParentEntity: parentEntity,
		Cursor:       dComments.Cursor,
	}
}

func createUserReactionsFromDuploReactions(dReactions *duplo.Reactions) *UserReactions {
	return &UserReactions{
		UserID:   dReactions.UserID,
		EmoteIDs: dReactions.EmoteIDs,
	}
}

func createReactionMapFromDuploReactionsSummary(dReactionsSummary *duplo.ReactionsSummary) ReactionMap {
	reactionMap := ReactionMap{}

	for emoteName, emoteSummary := range dReactionsSummary.EmoteSummaries {
		item := ReactionItem{
			Count:   emoteSummary.Count,
			UserIDs: emoteSummary.UserIDs,
		}
		if item.UserIDs == nil {
			// External API doesn't work well missing otherwise we would just use omitempty
			item.UserIDs = []string{}
		}
		reactionMap[emoteName] = &item
	}

	return reactionMap
}

func createSettingsFromFeedSettingsSettings(settings *feedsettings.Settings) *Settings {
	return (*Settings)(settings)
}
