package api

import (
	"math/rand"
	"sync"
	"time"

	"strings"

	"strconv"
	"sync/atomic"

	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/feeds/clients/duplo"
	"code.justin.tv/feeds/clients/masonry"
	"code.justin.tv/feeds/clients/shine"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"github.com/mvdan/xurls"
	uuid "github.com/satori/go.uuid"
	"golang.org/x/net/context"
	"golang.org/x/sync/errgroup"
)

func injectDebugging(isDebugEnabled bool, post *Post, item masonry.Activity) {
	if isDebugEnabled {
		post.Debug = &postDebugInfo{
			Source: item,
		}
	}
}

func (s *HTTPServer) feedItemsToPosts(ctx context.Context, feed *ChannelFeed, items []masonry.Activity, imap duploCache, feedTracking *masonry.FeedTracking) {
	_, isDebugEnabled := s.Ctxlog.ExtractDebugInfo(ctx)
	for _, item := range items {
		switch item.Entity.Namespace() {
		case entity.NamespacePost:
			dPost, exists := imap.posts[item.Entity]
			if !exists {
				continue
			}
			post := createPostFromDuploPost(dPost)
			post.Reasons = []*ReasonSuperStruct{{
				Type:      PostCreatedType,
				UserID:    post.UserID,
				CreatedAt: &dPost.CreatedAt,
			}}
			post.Tracking = getPostTracking(item, feedTracking)
			injectDebugging(isDebugEnabled, post, item)
			feed.Posts = append(feed.Posts, post)
		case entity.NamespaceShare:
			shareTarget, exists := imap.shares[item.Entity]
			if !exists {
				continue
			}
			postTarget, exists := imap.posts[shareTarget.TargetEntity]
			if !exists {
				continue
			}
			post := createPostFromDuploPost(postTarget)
			// We know this post has at least shareTarget.UserID in it.  Make sure it's there.
			post.ShareSummary.UserIDs = ensureContains(post.ShareSummary.UserIDs, shareTarget.UserID)

			post.Reasons = []*ReasonSuperStruct{{
				Type:      ShareCreatedType,
				UserID:    shareTarget.UserID,
				CreatedAt: &shareTarget.CreatedAt,
			}}
			injectDebugging(isDebugEnabled, post, item)
			feed.Posts = append(feed.Posts, post)
		case entity.NamespaceVod, entity.NamespaceClip, entity.NamespaceStream:
			postFromData, err2 := s.loadPostFromShineData(ctx, item.Entity)
			if err2 == nil && postFromData != nil {
				postFromData.Tracking = getPostTracking(item, feedTracking)
				postFromData.Reasons = loadReasons(item.RelevanceReason)
				feed.Posts = append(feed.Posts, postFromData)
			}
		}
	}
}

// getPostTracking returns a populated PostTracking object if tracking properties exist
func getPostTracking(item masonry.Activity, feedTracking *masonry.FeedTracking) *PostTracking {
	batchID := ""
	if feedTracking != nil {
		batchID = feedTracking.BatchID
	}

	return &PostTracking{
		RecGenerationID:    item.RecGenerationID,
		RecGenerationIndex: item.RecGenerationIndex,
		CardImpressionID:   uuid.NewV4().String(),
		BatchID:            batchID,
	}
}

// recReasonToEdgeReasonType maps "reason types" from the recommendation service to reason types supported by
// Feeds-Edge.
var recReasonToEdgeReasonType = map[string]ReasonTypeEnum{
	"top":                     PopularType,
	"followed_channel":        FollowedType,
	"followed_game":           FollowedType,
	"watched_channel":         ViewedType,
	"similar_watched_channel": ViewedType,
	"watched_vod":             ViewedType,
	"recommended_stream":      ViewedType,
}

func loadReasons(masonryReason *masonry.RelevanceReason) []*ReasonSuperStruct {
	if masonryReason == nil || masonryReason.Source != masonry.ReasonSourceRecommendations {
		return nil
	}

	edgeReasonType, ok := recReasonToEdgeReasonType[masonryReason.Kind]
	if !ok {
		return nil
	}

	return []*ReasonSuperStruct{
		{Type: edgeReasonType},
	}
}

func (s *HTTPServer) populateFeed(ctx context.Context, feed *ChannelFeed, items []masonry.Activity, userID string, masonryFeedID string, feedTracking *masonry.FeedTracking) error {
	feed.Posts = make([]*Post, 0, len(items))

	if len(items) == 0 {
		s.Log.DebugCtx(ctx, "no items to return")
		return nil
	}
	defer s.statDuration("populate.feed", time.Now())

	imap, err := s.DuploCacher.populateDuploCache(ctx, items, []string{userID})
	if err != nil {
		return err
	}
	s.feedItemsToPosts(ctx, feed, items, imap, feedTracking)
	err = s.populatePosts(ctx, feed.Posts, userID)
	feed.Posts = s.filterInvalidPosts(ctx, feed.Posts)
	return err
}

//  ensureContains will add item to original slice if it does not exist.  Returns a slice with origArray item inside it
func ensureContains(origArray []string, item string) []string {
	for _, i := range origArray {
		if i == item {
			return origArray
		}
	}
	return append(origArray, item)
}

func (s *HTTPServer) filterInvalidPosts(ctx context.Context, posts []*Post) []*Post {
	ret := make([]*Post, 0, len(posts))
	for _, post := range posts {
		if post.UserID != "" {
			ret = append(ret, post)
		}
	}
	return ret
}

func canLoadUserFromPost(post *Post) bool {
	if len(post.Embeds) != 1 {
		return false
	}
	embed := post.Embeds[0]
	if embed.TwitchType != entity.NamespaceVod && embed.TwitchType != entity.NamespaceClip && embed.TwitchType != entity.NamespaceStream {
		return false
	}

	return true
}

func (s *HTTPServer) prepopulatePosts(ctx context.Context, posts []*Post, userID string) error {
	// Content types have to have embeds populated first so we can fill in user id and creation time for
	// other populate steps
	contentPosts := make([]*Post, 0, len(posts))
	for _, post := range posts {
		if post.UserID == "" {
			contentPosts = append(contentPosts, post)
		}
	}
	if len(contentPosts) == 0 {
		return nil
	}
	s.populatePostEmbeds(ctx, contentPosts)

	for _, post := range contentPosts {
		if !canLoadUserFromPost(post) {
			continue
		}
		embed := post.Embeds[0]
		post.UserID = embed.AuthorID
		if embed.CreatedAt != nil {
			post.CreatedAt = *embed.CreatedAt
		}
		// TODO: Design needs to come back with what they want for post.Body
	}
	return nil
}

func (s *HTTPServer) populatePosts(ctx context.Context, posts []*Post, userID string) error {
	defer s.statDuration("populate.posts", time.Now())
	if err := s.prepopulatePosts(ctx, posts, userID); err != nil {
		return err
	}
	posts = s.filterInvalidPosts(ctx, posts)
	g, gCtx := errgroup.WithContext(ctx)
	g.Go(func() error {
		return s.populatePostPermissions(gCtx, posts, userID)
	})
	g.Go(func() error {
		return s.populatePostShares(gCtx, posts, userID)
	})
	g.Go(func() error {
		return s.populatePostReactions(gCtx, posts, []string{userID})
	})
	g.Go(func() error {
		s.populatePostEmotes(gCtx, posts)
		return nil
	})
	g.Go(func() error {
		s.populatePostEmbeds(gCtx, posts)
		return nil
	})
	return g.Wait()
}

func (s *HTTPServer) populatePostShares(ctx context.Context, posts []*Post, userID string) error {
	defer s.statDuration("populate.post_shares", time.Now())

	postsAsEntities := make([]entity.Entity, 0)
	for _, post := range posts {
		if !s.isContentPost(post.ID) {
			postsAsEntities = append(postsAsEntities, entity.New(entity.NamespacePost, post.ID))
		}
	}

	eg, egCtx := errgroup.WithContext(ctx)
	if userID != "" {
		eg.Go(func() error {
			return s.getSharesForPosts(egCtx, userID, postsAsEntities, posts)
		})
	}
	eg.Go(func() error {
		return s.getShareSummariesForPosts(egCtx, postsAsEntities, posts)
	})
	return eg.Wait()

}

func (s *HTTPServer) getSharesForPosts(egCtx context.Context, userID string, postsAsEntities []entity.Entity, posts []*Post) error {
	shares, err := s.Duplo.GetSharesByAuthor(egCtx, userID, postsAsEntities)
	if err != nil {
		return errors.Wrapf(err, "could not get shares from duplo user %s", userID)
	}
	for _, share := range shares.Items {
		foundAPost := false
		for _, post := range posts {
			if post.ID == share.TargetEntity.ID() {
				foundAPost = true
				post.ShareSummary.UserIDs = ensureContains(post.ShareSummary.UserIDs, userID)
				// Must finish loop because a post could be in posts twice
			}
		}
		if !foundAPost {
			s.Log.Log("target_entity", share.TargetEntity, "Got a response I didn't ask for")
		}
	}
	return nil
}

func (s *HTTPServer) getShareSummariesForPosts(egCtx context.Context, postsAsEntities []entity.Entity, posts []*Post) error {
	summaries, err := s.Duplo.GetSharesSummaries(egCtx, postsAsEntities)
	if err != nil {
		return err
	}
	for _, summary := range summaries.Items {
		foundAPost := false
		for _, post := range posts {
			if post.ID == summary.ParentEntity.ID() {
				foundAPost = true
				post.ShareSummary.ShareCount = summary.Total
			}
		}
		if !foundAPost {
			s.Log.Log("parent_entity", summary.ParentEntity, "Got a response I didn't ask for")
		}
	}
	return nil
}

func (s *HTTPServer) isContentPost(parentID string) bool {
	idAsEnt, err := entity.Decode(parentID)
	if err != nil {
		return false
	}
	return idAsEnt.Namespace() == entity.NamespaceVod || idAsEnt.Namespace() == entity.NamespaceStream || idAsEnt.Namespace() == entity.NamespaceClip
}

type permissionsEnvelope struct {
	permissions *PostPermissions
	author      string
}

func (s *HTTPServer) populatePostPermissions(ctx context.Context, posts []*Post, userID string) error {
	defer s.statDuration("populate.post_permissions", time.Now())

	authors := make(map[string]*PostPermissions)

	//Determine all unique authors.
	for _, post := range posts {
		// synthetic post permissions are hard coded to false and don't need to be looked up.
		if post.GetEntity().Namespace() != entity.NamespacePost {
			post.Permissions = &PostPermissions{
				CanShare:    false,
				CanReply:    false,
				CanModerate: false,
				CanDelete:   false,
			}
			continue
		}
		author := post.UserID
		authors[author] = nil
	}

	g, gCtx := errgroup.WithContext(ctx)
	c := make(chan permissionsEnvelope)
	for author := range authors {
		author := author
		g.Go(func() error {
			permissions, err := s.Authorization.GetPostPermissionsForUserToAuthor(gCtx, userID, author)
			if err != nil {
				s.Log.LogCtx(gCtx,
					"author", author,
					"user", userID,
					"err", err,
					"Error loading permissions for post author",
				)
				permissions = &PostPermissions{}
			}

			c <- permissionsEnvelope{
				permissions: permissions,
				author:      author,
			}
			return nil
		})
	}
	errorChan := make(chan error)
	defer close(errorChan)

	go func() {
		err := g.Wait()
		close(c)
		errorChan <- err
	}()

	for permissions := range c {
		authors[permissions.author] = permissions.permissions
	}

	err := <-errorChan
	if err != nil {
		return err
	}

	for _, post := range posts {
		// skip over synthetic posts
		if post.GetEntity().Namespace() != entity.NamespacePost {
			continue
		}
		permissions, ok := authors[post.UserID]
		if !ok {
			post.Permissions = &PostPermissions{}
			s.Log.LogCtx(gCtx,
				"post", post.ID,
				"user", userID,
				"err", err,
				"Populating uncalculated permissions",
			)
			continue
		}
		post.Permissions = permissions
	}

	return nil
}

// ReactionHaver is implemented by Post to populate reactions
type ReactionHaver interface {
	GetEntity() entity.Entity
	SetReactions(ReactionMap)
}

var _ ReactionHaver = &Post{}
var _ ReactionHaver = &Reactions{}

func (s *HTTPServer) populatePostReactions(ctx context.Context, posts []*Post, userIDs []string) error {
	defer s.statDuration("populate.post_reactions", time.Now())
	reactionHavers := make([]ReactionHaver, 0, len(posts))
	for _, post := range posts {
		reactionHavers = append(reactionHavers, post)
	}
	return s.populateReactions(ctx, reactionHavers, userIDs)
}

func (s *HTTPServer) populateReactions(ctx context.Context, reactionHavers []ReactionHaver, userIDs []string) error {
	if len(reactionHavers) == 0 {
		return nil
	}

	entities := make([]entity.Entity, 0, len(reactionHavers))
	// Note: Same entity could be in the map twice, so must be a list
	entityMap := map[entity.Entity][]ReactionHaver{}
	for _, item := range reactionHavers {
		itemEntity := item.GetEntity()
		if itemEntity.Namespace() == entity.NamespaceClip {
			itemEntity = normalizeClipEntityForReactions(itemEntity)
		}
		entities = append(entities, itemEntity)
		entityMap[itemEntity] = append(entityMap[itemEntity], item)
	}

	summaries, err := s.Duplo.GetReactionsSummariesByParent(ctx, entities, &duplo.GetReactionsSummariesByParentOptions{UserIDs: userIDs})
	if err != nil {
		return err
	}

	for _, summary := range summaries.Items {
		reactionHavers, ok := entityMap[summary.ParentEntity]
		if !ok {
			s.Log.LogCtx(ctx, "entity", summary.ParentEntity, "Could not find entity in map")
			continue
		}
		reactionMap := createReactionMapFromDuploReactionsSummary(summary)
		reactions, err := s.populateEmotesInReactionMap(ctx, reactionMap)
		if err != nil {
			s.Log.LogCtx(ctx,
				"entity", summary.ParentEntity,
				"err", err,
				"Could not populate emotes in reaction summaries",
			)
			continue
		}
		for _, reactionHaver := range reactionHavers {
			reactionHaver.SetReactions(reactions)
		}
	}

	return nil
}

func (s *HTTPServer) populateEmotesInReactionMap(ctx context.Context, reactionMap ReactionMap) (ReactionMap, error) {
	result := make(ReactionMap, len(reactionMap))
	for id, itemPtr := range reactionMap {
		// Make a pointer to clone so modifications don't affect the original copy
		item := *itemPtr
		result[id] = &item

		// TODO: Remove this call
		name, err := s.EmoteParser.GetEmoteName(id)
		if err != nil {
			// s.Log.Log("emote", id, "err", err, "could not lookup emote")
			continue
		}
		result[id].Emote = name
	}
	return result, nil
}

func (s *HTTPServer) hasInvalidEmotes(emotes []*Emote) bool {
	for _, emote := range emotes {
		isValid, isLoaded := s.EmoteParser.IsValidEmoteID(emote.ID)
		if isLoaded && !isValid {
			return true
		}
	}
	return false
}

func (s *HTTPServer) removePostsWithValidEmotes(posts []*Post) []*Post {
	ret := make([]*Post, 0, len(posts))
	for _, post := range posts {
		if post.Emotes == nil {
			ret = append(ret, post)
			continue
		}

		// Invalidate emotes that are currently invalid
		// This means also appending posts where post.Emotes is not nil, but some are
		// invalid
		if s.hasInvalidEmotes(post.Emotes) {
			post.Emotes = nil
			ret = append(ret, post)
		}
	}
	return ret
}

func (s *HTTPServer) populatePostEmotes(ctx context.Context, posts []*Post) {
	defer s.statDuration("populate.post_emotes", time.Now())
	posts = s.removePostsWithValidEmotes(posts)
	if len(posts) == 0 {
		// This means all the given posts already have valid cached emotes
		return
	}
	emoteHavers := make([]EmoteHaver, 0, len(posts))
	for _, post := range posts {
		// If there were emotes there, they are now invalid.  We set them to nil to make sure
		// we eventually populate them
		post.Emotes = nil
		emoteHavers = append(emoteHavers, post)
	}
	safeToSave := s.populateEmotes(ctx, emoteHavers)
	if safeToSave && s.Config.updateDuploEmotes.Get() {
		for _, post := range posts {
			if post.Emotes == nil {
				continue
			}
			item := postEmotesToUpdate{
				PostID: post.ID,
				Emotes: post.Emotes,
			}
			select {
			case s.postUpdateChan <- item:
				// Was able to send the item
			default:
				// Queue was too backed up.  Could not send the item.
				continue
			}
		}
	}
}

// EmoteHaver is implemented by Post to populate emotes
type EmoteHaver interface {
	GetUserID() string
	GetBody() string
	SetEmotes([]*Emote)
}

var _ EmoteHaver = &Post{}

func (s *HTTPServer) populateEmotes(ctx context.Context, emoteHavers []EmoteHaver) bool {
	defer s.statDuration("populate.emotes", time.Now())
	safeToSave := int64(1)
	wg := sync.WaitGroup{}
	wg.Add(len(emoteHavers))
	for _, emoteHaver := range emoteHavers {
		go func(emoteHaver EmoteHaver) {
			defer wg.Done()
			// This optimization is huge for recommendations, which do not have a body
			if emoteHaver.GetBody() == "" {
				return
			}
			resp, err := s.Zuma.ExtractMessage(ctx, api.ExtractMessageRequest{
				SenderID:     emoteHaver.GetUserID(),
				MessageText:  emoteHaver.GetBody(),
				AlwaysReturn: true,
			}, nil)
			if err != nil {
				// TODO: Temporary hack.  Do not log errors that are from spam.  Remove this hack later.
				if !strings.Contains(err.Error(), "spam") {
					s.Log.LogCtx(ctx, "err", err, "unable to extract zuma emotes")
				}
				atomic.StoreInt64(&safeToSave, 0)
				return
			}
			allEmotes := make([]*Emote, 0, len(resp.Content.Emoticons))
			for _, emoticon := range resp.Content.Emoticons {
				emoticonIDAsInt, err := strconv.Atoi(emoticon.ID)
				if err != nil {
					s.Log.LogCtx(ctx, "err", err, "emote_id", emoticon.ID, "unable to parse emoticon int")
					atomic.StoreInt64(&safeToSave, 0)
					continue
				}
				emoticonSetAsInt, err := strconv.Atoi(emoticon.SetID)
				if err != nil {
					s.Log.LogCtx(ctx, "err", err, "emote_set_id", emoticon.SetID, "unable to parse emoticon set as int")
					atomic.StoreInt64(&safeToSave, 0)
					continue
				}
				allEmotes = append(allEmotes, &Emote{
					ID:    emoticonIDAsInt,
					Start: emoticon.Start,
					End:   emoticon.End,
					Set:   emoticonSetAsInt,
				})
			}
			emoteHaver.SetEmotes(allEmotes)
		}(emoteHaver)
	}
	wg.Wait()
	return safeToSave == int64(1)
}

func (s *HTTPServer) populatePostEmbeds(ctx context.Context, posts []*Post) {
	defer s.statDuration("populate.post_embeds", time.Now())
	var wg sync.WaitGroup
	embedCtx, cancel := context.WithTimeout(ctx, s.Config.embedTimeout.Get())
	defer cancel()
	for _, post := range posts {
		wg.Add(1)
		go func(post *Post) {
			defer wg.Done()
			if post.Embeds != nil {
				// Sometimes we need to do embeds before everything else.  If they're already done,
				// don't do them again
				return
			}
			embedURLs := s.extractEmbedURLs(embedCtx, post)
			postEmbeds := s.requestEmbedURLs(embedCtx, embedURLs, nil)
			post.Embeds = removeNilEmbeds(postEmbeds)
			post.EmbedURLs = &embedURLs
			post.EmbedEntities = s.entitiesFromURLs(ctx, embedURLs)
		}(post)
	}
	wg.Wait()
}

func (s *HTTPServer) entitiesFromURLs(ctx context.Context, urls []string) *[]entity.Entity {
	if len(urls) == 0 {
		return nil
	}
	res, err := s.Shine.GetEntitiesForURLs(ctx, urls, nil)
	if err != nil || res == nil {
		return nil
	}
	ret := make([]entity.Entity, 0, len(res.Entities))
	for _, r := range res.Entities {
		ret = append(ret, r.Entity)
	}
	return &ret
}

// extractEmbedURLs returns the EmbedURLs for a post. If post.EmbedURLs is nil, it will
// attempt to extract from the post body and also backfill EmbedURLs into the post.
func (s *HTTPServer) extractEmbedURLs(ctx context.Context, post *Post) []string {
	if post.EmbedURLs != nil {
		// already populated. don't need to backfill
		return *post.EmbedURLs
	}

	urls := xurls.Relaxed.FindAllString(post.Body, -1)
	if urls == nil {
		urls = []string{}
	} else if len(urls) > 0 {
		urls = urls[:1]
	}

	if rand.Float64()*100 <= float64(s.Config.backfillEmbedURLsPct.Get()) {
		// TODO: avoid  race condition that can arise if the caller writes to urls
		go func() {
			updateCtx := context.Background()
			ents := make([]entity.Entity, 0, len(urls))

			if len(urls) > 0 {
				entsAndURLs, err2 := s.Shine.GetEntitiesForURLs(updateCtx, urls, nil)
				if err2 != nil {
					s.Log.LogCtx(updateCtx, "err", err2, "error backfilling EmbedURLs")
					return
				}
				for _, ent := range entsAndURLs.Entities {
					ents = append(ents, ent.Entity)
				}
			}

			err2 := s.Duplo.UpdatePost(updateCtx, post.ID, duplo.UpdatePostOptions{
				EmbedURLs:     &urls,
				EmbedEntities: &ents,
			})
			if err2 != nil {
				s.Log.LogCtx(updateCtx, "err", err2, "error backfilling EmbedURLs")
			}
		}()
	}
	return urls
}

func removeNilEmbeds(embeds []*Embed) []*Embed {
	result := make([]*Embed, 0)
	for _, embed := range embeds {
		if embed != nil {
			result = append(result, embed)
		}
	}
	return result
}

func (s *HTTPServer) requestEmbedURLs(embedCtx context.Context, embedURLs []string, autoplay *bool) []*Embed {
	var embedWG sync.WaitGroup
	postEmbeds := make([]*Embed, len(embedURLs))
	for index, embedURL := range embedURLs {
		embedWG.Add(1)
		go func(embedURL string, index int) {
			defer embedWG.Done()
			embed, err := s.requestEmbed(embedCtx, embedURL, autoplay)
			if err != nil {
				s.Log.DebugCtx(embedCtx, "err", err, "error requesting embed")
			}
			postEmbeds[index] = embed
		}(embedURL, index)
	}
	embedWG.Wait()
	return postEmbeds
}

func (s *HTTPServer) requestEmbed(ctx context.Context, url string, autoplay *bool) (*Embed, error) {
	shineEmbed, err := s.Shine.GetEmbed(ctx, url, &shine.GetEmbedOptions{
		Autoplay: autoplay,
	}, nil)
	if err != nil {
		if ctx.Err() == context.DeadlineExceeded {
			s.Stats.IncC("embed.timeout", 1, 1)
			return &Embed{RequestURL: url}, nil
		}
		if ctx.Err() == context.Canceled {
			s.Stats.IncC("embed.canceled", 1, 1)
			return &Embed{RequestURL: url}, nil
		}
		return nil, errors.Wrap(err, "cannot get embeds from shine")
	}
	if shineEmbed == nil {
		return nil, nil
	}
	var thumbnails *Thumbnails
	if shineEmbed.Thumbnails != nil {
		thumbnails = &Thumbnails{
			Medium: shineEmbed.Thumbnails.Medium,
			Small:  shineEmbed.Thumbnails.Small,
			Tiny:   shineEmbed.Thumbnails.Tiny,
		}
	}
	return &Embed{
		RequestURL:      shineEmbed.RequestURL,
		Type:            shineEmbed.Type,
		ProviderName:    shineEmbed.ProviderName,
		Title:           shineEmbed.Title,
		Description:     shineEmbed.Description,
		AuthorName:      shineEmbed.AuthorName,
		AuthorURL:       shineEmbed.AuthorURL,
		AuthorThumbnail: shineEmbed.AuthorThumbnail,
		AuthorID:        shineEmbed.AuthorID,
		AuthorLogin:     shineEmbed.AuthorLogin,
		ThumbnailURL:    shineEmbed.ThumbnailURL,
		Thumbnails:      thumbnails,
		PlayerHTML:      shineEmbed.PlayerHTML,
		VideoLength:     shineEmbed.VideoLength,
		ViewCount:       shineEmbed.ViewCount,
		Game:            shineEmbed.Game,
		CreatedAt:       shineEmbed.CreatedAt,
		TwitchType:      shineEmbed.TwitchType,
		TwitchContentID: shineEmbed.TwitchContentID,
		StartTime:       shineEmbed.StartTime,
		EndTime:         shineEmbed.EndTime,
		StreamType:      shineEmbed.StreamType,
	}, nil
}
