package v2

import (
	"encoding/json"
	"fmt"
	"math/rand"
	"net/http"
	"strings"
	"sync"
	"time"

	"code.justin.tv/chat/zuma/client"
	"code.justin.tv/feeds/clients/duplo"
	"code.justin.tv/feeds/clients/masonry"
	"code.justin.tv/feeds/clients/shine"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-edge/cmd/feeds-edge/internal/emotes"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/spade"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/web/twitter/client"
	users "code.justin.tv/web/users-service/client"
	"golang.org/x/net/context"
)

const (
	// DefaultPostLimit is how many posts we want to return by default if no limit is set
	DefaultPostLimit = 5

	iPhoneClientId           = "85lcqzxpb9bqu9z6ga1ol55du"
	iPadClientId             = "p9lhq6azjkdl72hs5xnt3amqu7vv8k2"
	androidClientId          = "kd1unb4b3q4t58fwlpcbzcbnm76a8fp"
	webClientClientID        = "jzkbprff40iqj646a697cyrvl0zt2m6"
	twilightInternalClientID = "ps1pu6on4i1x19vo6pgcjyzckr7kr1"
	twilightClientID         = "kimne78kx3ncx6brgo4mv6wki5h1ko"
	desktopAppClientID       = "jf3xu125ejjjt5cl4osdjci6oz6p93r"

	unknownPlatform          = "unknown"
	webClientPlatform        = "web"
	androidPlatform          = "Android"
	iPadPlatform             = "iPad"
	iPhonePlatform           = "iPhone"
	desktopPlatform          = "curse"
	twilightPlatform         = "twilight"
	twilightInternalPlatform = "twilightInternal"
)

var clientIDLookup = map[string]string{
	iPhoneClientId:           iPhonePlatform,
	iPadClientId:             iPadPlatform,
	androidClientId:          androidPlatform,
	webClientClientID:        webClientPlatform,
	twilightClientID:         twilightPlatform,
	twilightInternalClientID: twilightInternalPlatform,
	desktopAppClientID:       desktopPlatform,
	"":                       unknownPlatform,
}

func LookupClientPlatform(clientID string) string {
	p := clientIDLookup[clientID]
	if p == "" {
		return unknownPlatform
	}
	return p
}

// API in this package answers v2 HTTP requests
type API struct {
	Log           *log.ElevatedLog
	Config        *Config
	Duplo         *duplo.Client
	Masonry       *masonry.Client
	EmoteParser   *emotes.EmoteParser
	Authorization Authorization
	Shine         ShineClient
	Zuma          zuma.Client
	TwitterClient *twitter.Client
	UsersClient   users.Client
	SpadeClient   SpadeClient
	Cooldowns     Cooldowns
	Stats         *service_common.StatSender
}

// SpadeClient serves as an interface for mocking SpadeClient
type SpadeClient interface {
	QueueEvents(events ...spade.Event)
	Start() error
}

// ShineClient asserts what functions are needed from shine.Client to allow mocking
type ShineClient interface {
	GetEntitiesForURLs(ctx context.Context, urls []string, reqOpts *twitchclient.ReqOpts) (*shine.EntitiesForURLs, error)
	GetEmbedForEntity(ctx context.Context, ent entity.Entity, options *shine.GetEmbedOptions, reqOpts *twitchclient.ReqOpts) (*shine.Embed, error)
}

// Config configures the API
type Config struct {
	feedMaxLimit                    *distconf.Int
	cooldownExemptTestUsers         *distconf.Str
	createPostCooldown              *distconf.Duration
	validateFeedRolloutPct          *distconf.Float
	enableValidateFeedItemDebugLogs *distconf.Bool
}

// Authorization is anything that can authorize API access to feeds-edge
type Authorization interface {
	CanUserGetFeed(requestingUser string, feedID string) bool
	CanUserAddReaction(ctx context.Context, userID, emote string, isExistingReaction bool) (bool, error)
	CanUserDeletePostForAuthor(ctx context.Context, userID, authorID string) (bool, error)
	CanUserCreatePostForAuthor(ctx context.Context, userID, authorID string) (bool, error)
	CanUserCreateShareForAuthor(ctx context.Context, userID, authorID string) (bool, error)
	CanUserDeleteShareForAuthor(ctx context.Context, userID, authorID string) (bool, error)
}

// Load config from distconf
func (s *Config) Load(dconf *distconf.Distconf) error {
	s.feedMaxLimit = dconf.Int("feeds-edge.feed_max_limit", 20)
	s.cooldownExemptTestUsers = dconf.Str("feeds-edge.cooldown_exempt_test_users", "")
	s.createPostCooldown = dconf.Duration("feeds-edge.create_post_cooldown", 10*time.Second)
	s.validateFeedRolloutPct = dconf.Float("feeds-edge.validate_feed_items_rollout_pct", 0.0)
	s.enableValidateFeedItemDebugLogs = dconf.Bool("feeds-edge.enable_validate_feed_item_debug_logs", false)
	return nil
}

// GetSuggestedFeeds returns the feeds we think a user would like
func (s *API) GetSuggestedFeeds(req *http.Request) (interface{}, error) {
	userID, err := RequireUserID(req)
	if err != nil {
		return nil, err
	}
	return FeedIDs{
		FeedIDs: []string{
			// Not sure: just return the recommendation's feed
			fmt.Sprintf("r:%s", userID),
		},
	}, nil
}

// GetFeed returns a feed of entities
func (s *API) GetFeed(req *http.Request) (interface{}, error) {
	clientID := req.Header.Get("Twitch-Client-Id")

	feedID := req.URL.Query().Get("feed_id")
	userID := req.URL.Query().Get("user_id")
	cursor := req.URL.Query().Get("cursor")
	deviceID := req.URL.Query().Get("device_id")
	language := req.URL.Query().Get("language")
	if feedID == "" {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Errorf("Expected feed_id"),
		}
	}
	if permissionsCheck := s.Authorization.CanUserGetFeed(userID, feedID); !permissionsCheck {
		return nil, &service_common.CodedError{
			Code: http.StatusUnauthorized,
			Err:  errors.Errorf("User %s not allowed to load feed %s", userID, feedID),
		}
	}
	limit, err := ParseInt(req.URL.Query().Get("limit"), DefaultPostLimit)
	if err != nil {
		return nil, err
	}
	if limit > int(s.Config.feedMaxLimit.Get()) {
		limit = int(s.Config.feedMaxLimit.Get())
	}

	// Log problematic clients.
	if deviceID == "" {
		s.Stats.IncC(fmt.Sprintf("emptyDeviceID.%s", LookupClientPlatform(clientID)), 1, 1.0)
	}

	masonryFeed, err := s.getMasonryFeed(req, feedID, &masonry.GetFeedOptions{
		Cursor:   cursor,
		Limit:    limit,
		DeviceID: deviceID,
		Language: language,
	})
	if err != nil {
		return nil, err
	} else if masonryFeed == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("feed not found"),
		}
	}
	return s.masonryFeedToEntityFeed(masonryFeed), nil
}

func (s *API) getMasonryFeed(req *http.Request, feedID string, opts *masonry.GetFeedOptions) (*masonry.Feed, error) {
	if mockFeed := req.URL.Query().Get("mockFeed"); mockFeed != "" {
		return createMockMasonryFeed(mockFeed, feedID)
	}
	return nil, &service_common.CodedError{
		Code: http.StatusGone,
		Err:  errors.New("masonry is no more"),
	}
}

func createMockMasonryFeed(mockFeed string, feedID string) (*masonry.Feed, error) {
	var jsonMasonryFeed masonry.Feed
	if err := json.Unmarshal([]byte(mockFeed), &jsonMasonryFeed); err == nil {
		return &jsonMasonryFeed, nil
	}

	entities := strings.Split(mockFeed, ",")
	items := make([]masonry.Activity, 0, len(entities))
	for _, item := range entities {
		ent, err := entity.Decode(item)
		if err != nil {
			return nil, err
		}
		items = append(items, masonry.Activity{Entity: ent})
	}
	feed := &masonry.Feed{
		Items: items,
		ID:    feedID,
	}
	return feed, nil
}

func (s *API) masonryFeedToEntityFeed(masonryFeed *masonry.Feed) Feed {
	ret := Feed{}
	for _, activity := range masonryFeed.Items {
		tracking := &FeedItemTracking{
			RecGenerationID:    activity.RecGenerationID,
			RecGenerationIndex: activity.RecGenerationIndex,
			CardImpressionID:   CreateCardImpressionID(),
		}
		if masonryFeed.Tracking != nil {
			tracking.BatchID = masonryFeed.Tracking.BatchID
		}
		item := FeedItem{
			Entity:   activity.Entity,
			Tracking: tracking,
		}
		// TODO: masonry will need to return per item cursors
		switch activity.Entity.Namespace() {
		case entity.NamespaceShare, entity.NamespacePost:
			ret.Items = append(ret.Items, item)
		case entity.NamespaceVod, entity.NamespaceClip, entity.NamespaceStream:
			for _, r := range loadReasons(activity.RelevanceReason) {
				item.Reasons = append(item.Reasons, &FeedItemReason{
					Type:   r.Type,
					UserID: r.UserID,
				})
			}
			ret.Items = append(ret.Items, item)
		}
	}

	// TODO: remove this branch once the experiment is over
	if rand.Float64() < s.Config.validateFeedRolloutPct.Get() {
		validFeedItems := make([]FeedItem, len(ret.Items))
		var wg sync.WaitGroup

		for i, fi := range ret.Items {
			wg.Add(1)
			go func(fi FeedItem, i int) {
				defer wg.Done()
				e := fi.Entity
				if !s.isEntityValid(e) {
					s.Stats.IncC(fmt.Sprintf("%s.invalidFeedEntity", e.Namespace()), 1, 1.0)
					s.logInvalidEntity(fi)
					return
				}
				validFeedItems[i] = fi
			}(fi, i)
		}
		wg.Wait()
		ret.Items = filterEmptyFeedItems(validFeedItems)
	}

	// TODO: masonry will need to return per item cursors
	if len(ret.Items) > 0 {
		ret.Items[len(ret.Items)-1].Cursor = masonryFeed.Cursor
	}
	return ret
}

func (s *API) logInvalidEntity(fi FeedItem) {
	if s.Config.enableValidateFeedItemDebugLogs.Get() {
		if fi.Tracking != nil {
			s.Log.Log("entity", fi.Entity, "Rec-ID", fi.Tracking.RecGenerationID, "invalidFeedEntity")
		} else {
			s.Log.Log("entity", fi.Entity, "invalidFeedEntity")
		}
	}
}

func filterEmptyFeedItems(feedItems []FeedItem) []FeedItem {
	filteredFeedItems := make([]FeedItem, 0, len(feedItems))
	emptyEntity := entity.Entity{}
	for _, fi := range feedItems {
		if fi.Entity == emptyEntity {
			continue
		}
		filteredFeedItems = append(filteredFeedItems, fi)
	}
	return filteredFeedItems
}

// GetPostsByIDs returns a list of posts
func (s *API) GetPostsByIDs(req *http.Request) (interface{}, error) {
	const getBulkPostsByIDsLimit = 1000

	postIDs := ItemsFromQueryParam(req, "post_ids")
	if len(postIDs) == 0 {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("must provide at least one item in comma-separated post_ids"),
		}
	}

	if len(postIDs) > getBulkPostsByIDsLimit {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Errorf("cannot exceed %v post_ids requested", getBulkPostsByIDsLimit),
		}
	}

	duploPosts, err := s.Duplo.GetPosts(req.Context(), postIDs)
	if err != nil {
		return nil, err
	}

	posts := createPosts(duploPosts)
	return s.filterInvalidEmbedsFromPosts(posts), nil
}

func (s *API) filterInvalidEmbedsFromPosts(posts *Posts) *Posts {
	var mu sync.Mutex
	ret := &Posts{Items: make([]*Post, 0, len(posts.Items))}
	var wg sync.WaitGroup

	for _, p := range posts.Items {
		wg.Add(1)
		go func(p *Post) {
			defer wg.Done()

			if p.EmbedEntities != nil && len(*p.EmbedEntities) > 0 {
				validEmbedEntities := s.filterInvalidEmbeds(p.EmbedEntities, p.ID)
				p.EmbedEntities = validEmbedEntities
			}

			mu.Lock()
			ret.Items = append(ret.Items, p)
			mu.Unlock()
		}(p)
	}

	wg.Wait()
	return ret
}

func (s *API) filterInvalidEmbeds(embedEntities *[]entity.Entity, postID string) *[]entity.Entity {
	var mu sync.Mutex
	var wg sync.WaitGroup

	if embedEntities == nil {
		return nil
	}

	entities := *embedEntities
	validEmbedEntities := make([]entity.Entity, 0, len(entities))
	for _, e := range entities {
		wg.Add(1)
		go func(e entity.Entity) {
			defer wg.Done()

			if !s.isEntityValid(e) {
				s.Stats.IncC(fmt.Sprintf("%s.invalidEmbedEntity", e.Namespace()), 1, 1.0)
				return
			}
			mu.Lock()
			validEmbedEntities = append(validEmbedEntities, e)
			mu.Unlock()
		}(e)
	}

	wg.Wait()
	return &validEmbedEntities
}

func (s *API) isEntityValid(e entity.Entity) bool {
	switch e.Namespace() {
	case entity.NamespacePost:
		return true
	case entity.NamespaceShare:
		// Shine can't deal with Share's
		// Assume they are valid
		return true
	}

	embed, err := s.Shine.GetEmbedForEntity(context.Background(), e, nil, nil)
	if err != nil {
		s.Log.Log("err", err, "entity", e)
		return false
	}
	if embed == nil || isEmbedInvalid(embed, e) {
		return false
	}
	return true
}

func isEmbedInvalid(embed *shine.Embed, ent entity.Entity) bool {
	switch ent.Namespace() {
	case entity.NamespaceOembed:
		return false
	case entity.NamespaceVod, entity.NamespaceClip, entity.NamespaceStream, entity.NamespacePost:
		return embed == nil || embed.AuthorID == ""
	}
	return false
}

// GetPostsPermissionsByIDs returns a list of postIDs with their permissions for the specified user
func (s *API) GetPostsPermissionsByIDs(req *http.Request) (interface{}, error) {
	const getBulkPostsByIDsLimit = 1000

	userID, err := RequireUserID(req)
	if err != nil {
		return nil, err
	}

	postIDs := ItemsFromQueryParam(req, "post_ids")
	if len(postIDs) == 0 {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("must provide at least one item in comma-separated post_ids"),
		}
	}

	if len(postIDs) > getBulkPostsByIDsLimit {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Errorf("cannot exceed %v post_ids requested", getBulkPostsByIDsLimit),
		}
	}

	duploPosts, err := s.Duplo.GetPosts(req.Context(), postIDs)
	if err != nil {
		return nil, err
	}

	deletePermissions := make(map[string]bool)
	for _, post := range duploPosts.Items {
		if _, ok := deletePermissions[post.UserID]; !ok {
			deletePermissions[post.UserID], err = s.Authorization.CanUserDeletePostForAuthor(req.Context(), userID, post.UserID)
			if err != nil {
				return nil, err
			}
		}
	}

	permissions := make([]PostPermissions, len(duploPosts.Items))
	for i, post := range duploPosts.Items {
		permissions[i].PostID = post.ID
		permissions[i].CanDelete = deletePermissions[post.UserID]
	}

	return &PostsPermissions{
		Items: permissions,
	}, nil
}

func (s *API) isShareAllowed(ctx context.Context, targetEntity entity.Entity, userID string, authorID string) (bool, error) {
	canCreateShare, err := s.Authorization.CanUserCreateShareForAuthor(ctx, userID, authorID)
	if err != nil {
		return false, err
	}
	existingShares, serr := s.Duplo.GetSharesByAuthor(ctx, userID, []entity.Entity{targetEntity})
	if serr != nil {
		return false, serr
	}
	if len(existingShares.Items) != 0 {
		return false, &service_common.CodedError{
			Code: http.StatusConflict,
			Err:  errors.New("share already exists with this entity"),
		}
	}
	return canCreateShare, nil
}

// CreateShare creates a share
func (s *API) CreateShare(req *http.Request) (interface{}, error) {
	userID, err := RequireUserID(req)
	if err != nil {
		return nil, err
	}

	var params struct {
		TargetEntity entity.Entity `json:"target_entity"`
	}
	if err = json.NewDecoder(req.Body).Decode(&params); err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "could not parse body"),
		}
	}

	if params.TargetEntity.ID() == "" {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("target entity is required"),
		}
	}
	post, err := s.LoadAndBackfillDuploPost(req.Context(), params.TargetEntity.ID())
	if err != nil {
		return nil, err
	}
	if post == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("target entity does not exist"),
		}
	}
	canShare, canShareErr := s.isShareAllowed(req.Context(), params.TargetEntity, userID, post.UserID)
	if canShareErr != nil {
		return nil, canShareErr
	}
	if !canShare {
		return false, &service_common.CodedError{
			Code: http.StatusForbidden,
			Err:  errors.New("cannot create share"),
		}
	}
	duploShare, err := s.Duplo.CreateShare(req.Context(), userID, params.TargetEntity, nil)
	if err != nil {
		return nil, err
	}
	share := createShareFromDuploShare(duploShare)

	s.SpadeClient.QueueEvents(spade.Event{
		Name: "feed_server_share",
		Properties: ServerShareTracking{
			Action:       "create",
			UserID:       userID,
			TargetEntity: params.TargetEntity.Encode(),
			TargetType:   params.TargetEntity.Namespace(),
			TargetID:     params.TargetEntity.ID(),
		},
	})

	return share, nil
}

// DeleteShareShared is the shared part of DeleteShare in V1 and V2
func (s *API) DeleteShareShared(req *http.Request, shareID string) (interface{}, error) {
	userID, err := RequireUserID(req)
	if err != nil {
		return nil, err
	}

	duploShare, err := s.Duplo.GetShare(req.Context(), shareID)
	if err != nil {
		return nil, err
	}
	if duploShare == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errPostNotFound,
		}
	}
	share := createShareFromDuploShare(duploShare)

	canDelete, err := s.Authorization.CanUserDeleteShareForAuthor(req.Context(), userID, share.UserID)
	if err != nil {
		return nil, err
	}
	if !canDelete {
		return nil, &service_common.CodedError{
			Code: http.StatusForbidden,
			Err:  errors.New("unable to delete post"),
		}
	}

	duploShare, err = s.Duplo.DeleteShare(req.Context(), shareID)
	if err != nil {
		return nil, err
	}
	if duploShare == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errPostNotFound,
		}
	}
	share = createShareFromDuploShare(duploShare)

	s.SpadeClient.QueueEvents(spade.Event{
		Name: "feed_server_share",
		Properties: ServerShareTracking{
			Action:       "remove",
			UserID:       userID,
			TargetEntity: share.TargetEntity.Encode(),
			TargetType:   share.TargetEntity.Namespace(),
			TargetID:     share.TargetEntity.ID(),
		},
	})

	return share, nil
}

// DeleteShare deletes a share
func (s *API) DeleteShare(req *http.Request) (interface{}, error) {
	return s.DeleteShareShared(req, req.URL.Query().Get("share_id"))
}

// shared part of get shares in V1 nad V2
func (s *API) GetSharesShared(ctx context.Context, shareIDs []string) (interface{}, error) {
	const getBulkSharesByIDsLimit = 1000

	if len(shareIDs) == 0 {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("must provide at least one item in comma-separated share_ids"),
		}
	}

	if len(shareIDs) > getBulkSharesByIDsLimit {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Errorf("cannot exceed %v share_ids requested", getBulkSharesByIDsLimit),
		}
	}

	duploShares, err := s.Duplo.GetShares(ctx, shareIDs)
	if err != nil {
		return nil, err
	}

	return createShares(duploShares), nil
}

// GetSharesByIDs returns a list of shares
func (s *API) GetSharesByIDs(req *http.Request) (interface{}, error) {
	return s.GetSharesShared(req.Context(), ItemsFromQueryParam(req, "share_ids"))
}

// GetReactionsSummariesByEntities returns a list of reactions
func (s *API) GetReactionsSummariesByEntities(req *http.Request) (interface{}, error) {
	const getBulkReactionsSummariesByEntitiesLimit = 1000

	strong_consistency := req.URL.Query().Get("strong_consistency") == "true"

	userID := dedupeItems(ItemsFromQueryParam(req, "user_id"))
	if len(userID) > 1 {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Errorf("cannot exceed %v user_id requested", 1),
		}
	}

	entityStrings := dedupeItems(ItemsFromQueryParam(req, "entities"))
	if len(entityStrings) == 0 {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("must provide at least one item in comma-separated entities"),
		}
	}

	if len(entityStrings) > getBulkReactionsSummariesByEntitiesLimit {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Errorf("cannot exceed %v entities requested", getBulkReactionsSummariesByEntitiesLimit),
		}
	}

	return s.getReactionsSummariesByEntities(req.Context(), entityStrings, userID, strong_consistency)
}

// getReactionsSummariesByEntities contains the mechanics of the bulk get reactions
func (s *API) getReactionsSummariesByEntities(ctx context.Context, entityStrings []string, userID []string, strongConsistency bool) (*ReactionsSummaries, error) {
	entities := make([]entity.Entity, len(entityStrings))
	reactions := make([]*ReactionSummaries, len(entityStrings))
	reactionsMap := make(map[entity.Entity]*ReactionSummaries)
	for index, entityString := range entityStrings {
		// Clips entities need to be lowercased (normalized) in order to fetch their
		// reactions from duplo, but we must send back their original ID (unmodified)
		// to the caller. Thus we establish actualEntity as the original entity
		// to-be-returned and requestedEntity as the entity to send to duplo.

		actualEntity, err := entity.Decode(entityString)
		if err != nil {
			return nil, &service_common.CodedError{
				Code: http.StatusBadRequest,
				Err:  errors.Errorf("request contained an invalid entity"),
			}
		}

		reactions[index] = &ReactionSummaries{
			ParentEntity: actualEntity,
		}

		requestedEntity := actualEntity
		if actualEntity.Namespace() == entity.NamespaceClip {
			// Clips entities must be normalized to properly fetch reactions
			requestedEntity = normalizeClipEntityForReactions(actualEntity)
		}
		entities[index] = requestedEntity
		reactionsMap[requestedEntity] = reactions[index]
	}

	opts := &duplo.GetReactionsSummariesByParentOptions{
		UserIDs:           userID,
		StrongConsistency: strongConsistency,
	}
	reactionsSummaries, err := s.Duplo.GetReactionsSummariesByParent(ctx, entities, opts)
	if err != nil {
		return nil, err
	}

	for _, reactionsSummary := range reactionsSummaries.Items {
		ReactionsSummary, ok := reactionsMap[reactionsSummary.ParentEntity]
		if !ok {
			s.Log.LogCtx(ctx, "entity", reactionsSummary.ParentEntity, "Could not find parent entity in reactions map")
			continue
		}

		ReactionsSummary.Summaries = s.createSummariesFromDuploReactionsSummary(reactionsSummary.EmoteSummaries)
	}

	return &ReactionsSummaries{
		Items: reactions,
	}, nil
}

func (s *API) createSummariesFromDuploReactionsSummary(duploReactionsMap map[string]*duplo.EmoteSummary) []*ReactionSummary {
	summaries := make([]*ReactionSummary, 0, len(duploReactionsMap))

	for emoteID, emoteSummary := range duploReactionsMap {
		emoteName, err := s.EmoteParser.GetEmoteName(emoteID)
		// emoteName == "" means the emote was deleted by the owner.
		if err != nil || emoteName == "" {
			continue
		}

		summaries = append(summaries, &ReactionSummary{
			EmoteID:     emoteID,
			EmoteName:   emoteName,
			Count:       emoteSummary.Count,
			UserReacted: len(emoteSummary.UserIDs) > 0,
		})
	}

	return summaries
}

func (s *API) getCreatePostOptions(ctx context.Context, userID string, paramsEmbedURLs *[]string, body string, parseEmote bool) (*duplo.CreatePostOptions, error) {
	embedEnts, embedURLs, err := s.extractEntitiesFromPost(ctx, paramsEmbedURLs, body)
	if err != nil {
		return nil, err
	}
	createPostOptions := &duplo.CreatePostOptions{
		EmbedEntities: embedEnts,
		EmbedURLs:     embedURLs,
	}

	if parseEmote {
		var emotes *[]duplo.Emote
		emotes, err = s.parseEmotes(ctx, userID, body)
		if err != nil {
			return nil, err
		}
		createPostOptions.Emotes = emotes
	}

	return createPostOptions, nil
}

// The shared part of CreatePost in v1 and v2
func (s *API) CreatePostShared(req *http.Request, parseEmote bool) (bool, string, *duplo.Post, error) {
	userID, err := RequireUserID(req)
	if err != nil {
		return false, "", nil, err
	}

	var params createPostParams
	if err = json.NewDecoder(req.Body).Decode(&params); err != nil {
		return false, "", nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "could not parse body"),
		}
	}

	ctx := req.Context()

	if s.isCreatePostBeingThrottled(ctx, userID) {
		return false, "", nil, &service_common.CodedError{
			Code: http.StatusTooManyRequests,
			Err:  errors.New("user is creating posts too quickly"),
		}
	}

	canCreatePost, err := s.Authorization.CanUserCreatePostForAuthor(ctx, userID, params.UserID)
	if err != nil {
		return false, "", nil, err
	}
	if !canCreatePost {
		return false, "", nil, &service_common.CodedError{
			Code: http.StatusForbidden,
			Err:  errors.New("cannot create post"),
		}
	}

	createPostOptions, err := s.getCreatePostOptions(ctx, userID, params.EmbedURLs, params.Body, parseEmote)
	if err != nil {
		return false, "", nil, err
	}

	duploPost, err := s.Duplo.CreatePost(ctx, params.UserID, params.Body, createPostOptions)
	if err != nil {
		return false, "", nil, err
	}

	s.putUserOnPostCooldown(ctx, params.UserID)

	postEntity := entity.New(entity.NamespacePost, duploPost.ID)
	s.SpadeClient.QueueEvents(spade.Event{
		Name: "feed_server_post",
		Properties: ServerPostTracking{
			Entity:  postEntity.Encode(),
			PostID:  postEntity.ID(),
			Action:  "create",
			UserID:  userID,
			Content: params.Body,
		},
	})
	for i := 0; i < len(*createPostOptions.EmbedURLs) && i < len(*createPostOptions.EmbedEntities); i++ {
		s.SpadeClient.QueueEvents(spade.Event{
			Name: "feed_server_post_embed",
			Properties: ServerPostEmbedTracking{
				Entity:      postEntity.Encode(),
				PostID:      postEntity.ID(),
				EmbedIndex:  int64(i),
				EmbedEntity: (*createPostOptions.EmbedEntities)[i].Encode(),
				EmbedURL:    (*createPostOptions.EmbedURLs)[i],
				EmbedType:   (*createPostOptions.EmbedEntities)[i].Namespace(),
				EmbedID:     (*createPostOptions.EmbedEntities)[i].ID(),
			},
		})
	}

	return params.PostToTwitter, userID, duploPost, nil
}

// CreatePost creates a post and send it to twitter, WARNING: It's the same as the createSyndicatedPost in V1, not createPost.
func (s *API) CreatePost(req *http.Request) (interface{}, error) {
	postToTwitter, userID, duploPost, err := s.CreatePostShared(req, true)
	if err != nil {
		return nil, err
	}

	post := createPostFromDuploPost(duploPost)

	tweetStatus := 0
	tweetMessage := ""
	if postToTwitter {
		tweetStatus, tweetMessage = s.CreateTweetForPost(req.Context(), post.ID, post.Body, userID)
	}
	return &CreatePostResponse{post, tweetStatus, tweetMessage}, nil
}

// The shared part of DeletePost in v1 and v2
func (s *API) DeletePostShared(req *http.Request, postID string) (*duplo.Post, string, error) {
	userID, err := RequireUserID(req)
	if err != nil {
		return nil, userID, err
	}

	duploPost, err := s.RequireDuploPost(req.Context(), postID)
	if err != nil {
		return nil, userID, err
	}

	canDelete, err := s.Authorization.CanUserDeletePostForAuthor(req.Context(), userID, duploPost.UserID)
	if err != nil {
		return nil, userID, err
	}
	if !canDelete {
		return nil, userID, &service_common.CodedError{
			Code: http.StatusForbidden,
			Err:  errors.New("unable to delete post"),
		}
	}

	duploPost, err = s.Duplo.DeletePost(req.Context(), postID, nil)
	if err != nil {
		return nil, userID, err
	}
	if duploPost == nil {
		return nil, userID, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errPostNotFound,
		}
	}

	postEntity := entity.New(entity.NamespacePost, duploPost.ID)
	s.SpadeClient.QueueEvents(spade.Event{
		Name: "feed_server_post",
		Properties: ServerPostTracking{
			Entity: postEntity.Encode(),
			PostID: postEntity.ID(),
			Action: "remove",
			UserID: userID,
		},
	})

	return duploPost, userID, nil
}

// DeletePost deletes a post
func (s *API) DeletePost(req *http.Request) (interface{}, error) {
	// get post id. Legacy id no longer supported because all delete request would be using new ID.
	postID := req.URL.Query().Get("post_id")

	duploPost, _, err := s.DeletePostShared(req, postID)
	if err != nil {
		return nil, err
	}

	return createPostFromDuploPost(duploPost), nil
}

// CreateReaction creates a reaction on the parent_entity
func (s *API) CreateReaction(req *http.Request) (interface{}, error) {
	parentEntity, emoteID, userID, err := parseReactionRequest(req)
	if err != nil {
		return nil, err
	}

	rs, err := s.Duplo.GetReactionsSummary(req.Context(), parentEntity)
	if err != nil {
		return nil, err
	}
	_, isExistingReaction := rs.EmoteSummaries[emoteID]

	canAddReaction, err := s.Authorization.CanUserAddReaction(req.Context(), userID, emoteID, isExistingReaction)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusInternalServerError,
			Err:  errors.Wrap(err, "could not check if user can add reaction"),
		}
	} else if !canAddReaction {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("cannot use reaction: authorization failed"),
		}
	}

	if _, rerr := s.Duplo.CreateReaction(req.Context(), parentEntity, userID, emoteID); rerr != nil {
		return nil, rerr
	}

	s.SpadeClient.QueueEvents(spade.Event{
		Name: "feed_server_reaction",
		Properties: ServerReactionTracking{
			Action:       "create",
			UserID:       userID,
			ReactionID:   emoteID,
			TargetEntity: parentEntity.Encode(),
			TargetType:   parentEntity.Namespace(),
			TargetID:     parentEntity.ID(),
		},
	})

	return "OK", nil
}

// DeleteReaction removes a reaction on the parent_entity
func (s *API) DeleteReaction(req *http.Request) (interface{}, error) {
	parentEntity, emoteID, userID, err := parseReactionRequest(req)
	if err != nil {
		return nil, err
	}

	reactions, err := s.Duplo.DeleteReaction(req.Context(), parentEntity, userID, emoteID)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusInternalServerError,
			Err:  errors.Wrap(err, "could not delete reaction"),
		}
	} else if reactions == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("could not find reaction"),
		}
	}

	s.SpadeClient.QueueEvents(spade.Event{
		Name: "feed_server_reaction",
		Properties: ServerReactionTracking{
			Action:       "remove",
			UserID:       userID,
			ReactionID:   emoteID,
			TargetEntity: parentEntity.Encode(),
			TargetType:   parentEntity.Namespace(),
			TargetID:     parentEntity.ID(),
		},
	})

	return "OK", nil
}

// create a Twitter Post
func (s *API) CreateTweetForPost(ctx context.Context, postID, postBody, userID string) (int, string) {
	user, err := s.UsersClient.GetUserByID(ctx, userID, nil)
	if err != nil {
		s.Log.Log(
			"err", err,
			"userID", userID,
			"getting user info for userID failed",
		)
		return 500, "Error loading user"
	}

	if user.Login == nil {
		s.Log.Log(
			"userID", userID,
			"user login returned nil from users service",
		)
		return 500, "Error loading user"
	}
	deepURL := fmt.Sprintf("https://www.twitch.tv/%s/p/%s", *user.Login, postID)
	tweetURL, err := s.TwitterClient.PostLinkForTwitchUserID(ctx, postBody, deepURL, userID)
	if err != nil {
		s.Log.Log(
			"err", err,
			"userID", userID,
			"postID", postID,
			"postBody", postBody,
			"creating tweet for post failed",
		)
		return 500, "Failed to create tweet"
	}

	return 201, tweetURL
}

// The return envolope for created a post
type CreatePostResponse struct {
	Post        *Post  `json:"post"`
	TweetStatus int    `json:"tweet_status"`
	Tweet       string `json:"tweet"`
}

func (s *API) putUserOnPostCooldown(ctx context.Context, userID string) {
	if strings.Contains(s.Config.cooldownExemptTestUsers.Get(), ","+userID+",") {
		return
	}
	if err := s.Cooldowns.PutUserOnCooldown(ctx, userID, CreatePostCooldown, s.Config.createPostCooldown.Get()); err != nil {
		s.Log.LogCtx(ctx,
			"err", err,
			"userID", userID,
			"putting user on cooldown for 'create post' failed")
	}
}

func (s *API) isCreatePostBeingThrottled(ctx context.Context, userID string) bool {
	onCooldown, _, err := s.Cooldowns.IsUserOnCooldown(ctx, userID, CreatePostCooldown)
	if err != nil {
		s.Log.LogCtx(ctx,
			"err", err,
			"userID", userID,
			"checking whether user is on cooldown for 'create post' failed")
		return false
	}
	return onCooldown
}
