package v2

import (
	"fmt"
	"time"

	"code.justin.tv/feeds/log"
	"github.com/garyburd/redigo/redis"
	"golang.org/x/net/context"
)

// CreatePostCooldown identifies "create post" user actions.
const CreatePostCooldown string = "post"

// Cooldowns facilitates throttling a user's actions.
type Cooldowns interface {

	// IsUserOnCooldown gets whether a user's action is being throttled, and the remaining amount of time the
	// action should be throttled for.
	IsUserOnCooldown(ctx context.Context, userID string, action string) (bool, time.Duration, error)

	// PutUserOnCooldown records that the given user should not be allowed to perform the given action, until
	// the given duration of time has passed.
	PutUserOnCooldown(ctx context.Context, userID, action string, duration time.Duration) error
}

// RedisCooldowns is an implementation of Cooldowns that uses Redis to store its state.
type RedisCooldowns struct {
	RedisPool *redis.Pool
	Log       *log.ElevatedLog
	Prefix    string
}

// IsUserOnCooldown gets whether a user's action is being throttled, and the remaining amount of time the
// action should be throttled for.
func (r *RedisCooldowns) IsUserOnCooldown(ctx context.Context, userID, action string) (bool, time.Duration, error) {
	client := r.RedisPool.Get()
	defer func() {
		err := client.Close()
		if err != nil {
			r.Log.LogCtx(ctx,
				"err", err,
				"could not close redis client",
			)
		}
	}()

	key := r.generateKey(userID, action)

	ttl, err := redis.Int(client.Do("TTL", key))

	onCooldown := false
	duration := time.Duration(0)
	if err == nil && ttl > 0 {
		onCooldown = true
		duration = time.Duration(ttl) * time.Second
	}

	return onCooldown, duration, err
}

// PutUserOnCooldown records that the given user should not be allowed to perform the given action, until
// the given duration of time has passed.
func (r *RedisCooldowns) PutUserOnCooldown(ctx context.Context, userID, action string, duration time.Duration) error {
	if duration <= time.Nanosecond {
		return nil
	}

	client := r.RedisPool.Get()
	defer func() {
		err := client.Close()
		if err != nil {
			r.Log.LogCtx(ctx,
				"err", err,
				"could not close redis client",
			)
		}
	}()

	key := r.generateKey(userID, action)
	_, err := client.Do("SETEX", key, duration.Seconds(), "")

	return err
}

func (r *RedisCooldowns) generateKey(userID, action string) string {
	return fmt.Sprintf("%scooldown:%s:%s", r.Prefix, userID, action)
}
