package v2

import (
	"fmt"
	"net/http"
	"strconv"
	"strings"

	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/feeds/clients/duplo"
	"code.justin.tv/feeds/clients/masonry"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/service-common"
	"github.com/mvdan/xurls"
	"golang.org/x/net/context"
)

var (
	errPostNotFound = errors.New("post does not exist")
)

// recReasonToEdgeReasonType maps "reason types" from the recommendation service to reason types supported by
// Feeds-Edge.
var recReasonToEdgeReasonType = map[string]ReasonTypeEnum{
	"top":                     PopularType,
	"followed_channel":        FollowedType,
	"followed_game":           FollowedType,
	"watched_channel":         ViewedType,
	"similar_watched_channel": ViewedType,
	"watched_vod":             ViewedType,
}

func loadReasons(masonryReason *masonry.RelevanceReason) []*FeedItemReason {
	if masonryReason == nil || masonryReason.Source != masonry.ReasonSourceRecommendations {
		return nil
	}

	edgeReasonType, ok := recReasonToEdgeReasonType[masonryReason.Kind]
	if !ok {
		return nil
	}

	return []*FeedItemReason{
		{Type: edgeReasonType},
	}
}

// Lowercase ID required to fetch reactions for clip entities
func normalizeClipEntityForReactions(ent entity.Entity) entity.Entity {
	return entity.New(ent.Namespace(), strings.ToLower(ent.ID()))
}

// ParseInt tries to convert an optional int parameter from a string URL
func ParseInt(intParam string, defaultValue int) (int, error) {
	if intParam == "" {
		return defaultValue, nil
	}
	asInt, err := strconv.Atoi(intParam)
	if err != nil {
		return 0, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, fmt.Sprintf("expect integer param, saw %s", intParam)),
		}
	}
	return asInt, nil
}

// ItemsFromQueryParam converts a comma-separated list param into individual strings
func ItemsFromQueryParam(r *http.Request, paramName string) []string {
	items := strings.Split(r.URL.Query().Get(paramName), ",")
	ret := make([]string, 0, len(items))
	for _, val := range items {
		item := strings.TrimSpace(val)
		if item != "" {
			ret = append(ret, item)
		}
	}
	return ret
}

// RequireUserID ensures that a user_id is provided with the request
func RequireUserID(req *http.Request) (string, error) {
	userID := req.URL.Query().Get("user_id")
	if userID == "" {
		return "", &service_common.CodedError{
			Code: http.StatusForbidden,
			Err:  errors.New("you must provide a user_id"),
		}
	}
	return userID, nil
}

// convert URLs to entities
func (s *API) UrlsToEntities(ctx context.Context, embedURLs *[]string) (*[]entity.Entity, *[]string, error) {
	if embedURLs == nil {
		return nil, nil, nil
	}
	if len(*embedURLs) == 0 {
		return &[]entity.Entity{}, &[]string{}, nil
	}

	ents, err := s.Shine.GetEntitiesForURLs(ctx, *embedURLs, nil)
	if err != nil {
		return nil, nil, err
	}
	ret := make([]entity.Entity, 0, len(*embedURLs))
	urls := make([]string, 0, len(*embedURLs))
	for _, u := range *embedURLs {
		for _, ent := range ents.Entities {
			if ent.URL == u {
				ret = append(ret, ent.Entity)
				urls = append(urls, ent.URL)
				break
			}
		}
	}
	return &ret, &urls, nil
}

// populate embed URLs
func (s *API) PopulateEmbedURLs(body string) *[]string {
	urls := xurls.Relaxed.FindAllString(body, -1)
	if urls == nil {
		urls = []string{}
	} else if len(urls) > 0 {
		urls = urls[:1]
	}
	return &urls
}

func (s *API) extractEntitiesFromPost(ctx context.Context, postEmbedURLs *[]string, postBody string) (*[]entity.Entity, *[]string, error) {
	var embedURLs *[]string
	if postEmbedURLs == nil {
		embedURLs = s.PopulateEmbedURLs(postBody)
	} else {
		embedURLs = postEmbedURLs
	}

	return s.UrlsToEntities(ctx, embedURLs)
}

func (s *API) parseEmotes(ctx context.Context, userID string, body string) (*[]duplo.Emote, error) {
	parsed, err := s.Zuma.ExtractMessage(ctx, api.ExtractMessageRequest{
		SenderID:     userID,
		MessageText:  body,
		AlwaysReturn: true,
	}, nil)
	if err != nil {
		return nil, err
	}

	emotes := make([]duplo.Emote, 0, len(parsed.Content.Emoticons))
	for _, zumaEmote := range parsed.Content.Emoticons {
		id, err := strconv.Atoi(zumaEmote.ID)
		if err != nil {
			return nil, err
		}
		set, err := strconv.Atoi(zumaEmote.SetID)
		if err != nil {
			return nil, err
		}
		emotes = append(emotes, duplo.Emote{
			ID:    id,
			Start: zumaEmote.Start,
			End:   zumaEmote.End,
			Set:   set,
		})
	}
	return &emotes, nil
}

// backfill duplo post
func (s *API) BackfillDuploPost(ctx context.Context, duploPost *duplo.Post) (*duplo.Post, error) {
	if duploPost == nil {
		return nil, nil
	}
	if duploPost.EmbedURLs != nil && duploPost.EmbedEntities == nil {
		embedEnts, embedURLs, err := s.UrlsToEntities(ctx, duploPost.EmbedURLs)
		if err != nil {
			return nil, err
		}
		updatePostOptions := duplo.UpdatePostOptions{
			EmbedEntities: embedEnts,
			EmbedURLs:     embedURLs,
		}
		if err := s.Duplo.UpdatePost(ctx, duploPost.ID, updatePostOptions); err != nil {
			return nil, errors.Wrap(err, "unable to update post with new entities")
		}
		duploPost.EmbedEntities = embedEnts
		duploPost.EmbedURLs = embedURLs
	}
	return duploPost, nil
}

// load and backfill duplo post
func (s *API) LoadAndBackfillDuploPost(ctx context.Context, postID string) (*duplo.Post, error) {
	duploPost, err := s.Duplo.GetPost(ctx, postID)
	if err != nil {
		return nil, err
	}
	return s.BackfillDuploPost(ctx, duploPost)
}

// ensure the required duplo post exists and retrieve it
func (s *API) RequireDuploPost(ctx context.Context, postID string) (*duplo.Post, error) {
	duploPost, err := s.LoadAndBackfillDuploPost(ctx, postID)
	if err != nil {
		return nil, err
	}

	if duploPost == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errPostNotFound,
		}
	}
	return duploPost, nil
}

// dedupeItems removes all duplicates in a string slice
func dedupeItems(items []string) []string {
	itemsMap := make(map[string]bool)
	dedupedItems := make([]string, 0, len(items))
	for _, item := range items {
		if !itemsMap[item] {
			itemsMap[item] = true
			dedupedItems = append(dedupedItems, item)
		}
	}

	return dedupedItems
}

func parseReactionRequest(req *http.Request) (entity.Entity, string, string, error) {
	entityString := req.URL.Query().Get("parent_entity")
	parentEntity, err := entity.Decode(entityString)
	if err != nil {
		return entity.Entity{}, "", "", &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "must provide a valid parent_entity"),
		}
	}

	emoteID := req.URL.Query().Get("emote_id")
	if emoteID == "" {
		return entity.Entity{}, "", "", &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("must provide an emote_id"),
		}
	}

	userID, err := RequireUserID(req)
	if err != nil {
		return entity.Entity{}, "", "", err
	}

	if parentEntity.Namespace() == entity.NamespaceClip {
		parentEntity = normalizeClipEntityForReactions(parentEntity)
	}
	return parentEntity, emoteID, userID, nil
}
