package v2

import (
	"time"

	"code.justin.tv/feeds/clients/duplo"
	"code.justin.tv/feeds/feeds-common/entity"
	"github.com/satori/go.uuid"
)

// Feed is a feed of entities
type Feed struct {
	Items []FeedItem `json:"items"`
}

// FeedItem is the trimmed down entity item for a feed
type FeedItem struct {
	Entity   entity.Entity     `json:"entity"`
	Reasons  []*FeedItemReason `json:"reasons,omitempty"`
	Tracking *FeedItemTracking `json:"tracking,omitempty"`
	Cursor   string            `json:"cursor"`
}

// FeedItemReason should be every reason that can happen.  Honestly, this is the easiest way to do this in Go
type FeedItemReason struct {
	Type   ReasonTypeEnum `json:"type"`
	UserID string         `json:"user_id,omitempty"`
}

// FeedItemTracking contains feed item level fields used for tracking
type FeedItemTracking struct {
	RecGenerationID    string `json:"rec_generation_id,omitempty"`
	RecGenerationIndex *int   `json:"rec_generation_index,omitempty"`
	CardImpressionID   string `json:"card_impression_id,omitempty"`
	BatchID            string `json:"batch_id,omitempty"`
}

func CreateCardImpressionID() string {
	return uuid.NewV4().String()
}

// ReasonTypeEnum is the set of reason types we expose for feed items
type ReasonTypeEnum string

const (
	// FollowedType indicates that a piece of content is in a user's feed because the user
	// followed the content creator.
	FollowedType ReasonTypeEnum = "followed"

	// ViewedType indicates that a piece of content is in a user's feed because the user
	// viewed similar content.
	ViewedType ReasonTypeEnum = "viewed"

	// PopularType indicates that a piece of content is in a user's feed because it is
	// popular on Twitch.
	PopularType ReasonTypeEnum = "popular"
)

// Posts is a set of Post items
type Posts struct {
	Items []*Post `json:"items"`
}

// Post is a post, but only contains what is returned from duplo
type Post struct {
	ID            string           `json:"id"`
	CreatedAt     time.Time        `json:"created_at"`
	Deleted       bool             `json:"deleted,omitempty"`
	UserID        string           `json:"user_id"`
	Body          string           `json:"body"`
	Emotes        []*Emote         `json:"emotes"`
	EmbedEntities *[]entity.Entity `json:"embed_entities,omitempty"`
}

// PostPermissions is the permissions for a single post.
type PostPermissions struct {
	PostID    string `json:"post_id"`
	CanDelete bool   `json:"can_delete"`
}

// PostsPermissions is the permissions for multiple posts.
type PostsPermissions struct {
	Items []PostPermissions `json:"items"`
}

// Emote is the emote object for the API
type Emote struct {
	ID    int `json:"id"`
	Start int `json:"start"`
	End   int `json:"end"`
	Set   int `json:"set"`
}

func createPostFromDuploPost(dPost *duplo.Post) *Post {
	return &Post{
		ID:            dPost.ID,
		CreatedAt:     dPost.CreatedAt,
		Deleted:       dPost.DeletedAt != nil,
		UserID:        dPost.UserID,
		Body:          dPost.Body,
		Emotes:        emotesFromDuploEmotes(dPost.Emotes),
		EmbedEntities: dPost.EmbedEntities,
	}
}

func createPosts(duploPosts *duplo.Posts) *Posts {
	posts := make([]*Post, len(duploPosts.Items))

	for index, duploPost := range duploPosts.Items {
		post := createPostFromDuploPost(duploPost)
		posts[index] = post
	}

	return &Posts{
		Items: posts,
	}
}

type createPostParams struct {
	UserID        string    `json:"user_id"`
	Body          string    `json:"body"`
	PostToTwitter bool      `json:"post_to_twitter"`
	EmbedURLs     *[]string `json:"embed_urls"`
}

func emotePtr(emote Emote) *Emote {
	return &emote
}

func emotesFromDuploEmotes(emotes *[]duplo.Emote) []*Emote {
	if emotes == nil || *emotes == nil {
		return nil
	}
	ret := make([]*Emote, len(*emotes))
	for i, emote := range *emotes {
		ret[i] = emotePtr(Emote(emote))
	}
	return ret
}

// Shares is a set of Share items
type Shares struct {
	Items []*Share `json:"items"`
}

// FeedIDs is a list of encoded feed IDs
type FeedIDs struct {
	FeedIDs []string `json:"feed_ids"`
}

// Share is the returned object for a database share
type Share struct {
	ID           string        `json:"id"`
	UserID       string        `json:"user_id"`
	TargetEntity entity.Entity `json:"target_entity"`
	CreatedAt    time.Time     `json:"created_at"`
	DeletedAt    *time.Time    `json:"deleted_at,omitempty"`
}

func createShareFromDuploShare(dShare *duplo.Share) *Share {
	return &Share{
		ID:           dShare.ID,
		UserID:       dShare.UserID,
		TargetEntity: dShare.TargetEntity,
		CreatedAt:    dShare.CreatedAt,
		DeletedAt:    dShare.DeletedAt,
	}
}

func createShares(duploShares *duplo.Shares) *Shares {
	shares := make([]*Share, len(duploShares.Items))

	for index, duploShare := range duploShares.Items {
		share := createShareFromDuploShare(duploShare)
		shares[index] = share
	}

	return &Shares{
		Items: shares,
	}
}

// ReactionsSummaries is the response to a bulk get reactions
type ReactionsSummaries struct {
	Items []*ReactionSummaries `json:"items"`
}

// ReactionSummaries is the collection of all reaction summaries for the given parent entity
type ReactionSummaries struct {
	ParentEntity entity.Entity      `json:"parent_entity"`
	Summaries    []*ReactionSummary `json:"summaries"`
}

// ReactionSummary is the count for a particular emote on a given parent entity
type ReactionSummary struct {
	EmoteID     string `json:"emote_id"`
	EmoteName   string `json:"emote_name"`
	Count       int    `json:"count"`
	UserReacted bool   `json:"user_reacted"`
}

// ForEntity returns the ReactionSummaries associated with the entity passed in.
func (rs *ReactionsSummaries) ForEntity(e entity.Entity) *ReactionSummaries {
	if rs == nil {
		return nil
	}

	for _, item := range rs.Items {
		if pe := item.ParentEntity; (pe.ID() == e.ID()) &&
			(pe.Namespace() == e.Namespace()) {
			return item
		}
	}
	return nil
}

// Server-Tracked events
type ServerPostTracking struct {
	Entity  string `json:"entity"`
	PostID  string `json:"post_id"`
	Action  string `json:"action"`
	UserID  string `json:"user_id"`
	Content string `json:"content"`
}

type ServerReactionTracking struct {
	Action       string `json:"action"`
	UserID       string `json:"user_id"`
	ReactionID   string `json:"reaction_id"`
	TargetEntity string `json:"target_entity"`
	TargetType   string `json:"target_type"`
	TargetID     string `json:"target_id"`
}

type ServerReportTracking struct {
	UserID       string `json:"user_id"`
	TargetEntity string `json:"target_entity"`
	TargetType   string `json:"target_type"`
	TargetID     string `json:"target_id"`
	Reason       string `json:"reason"`
}

type ServerShareTracking struct {
	Action       string `json:"action"`
	UserID       string `json:"user_id"`
	TargetEntity string `json:"target_entity"`
	TargetType   string `json:"target_type"`
	TargetID     string `json:"target_id"`
}

type ServerPostEmbedTracking struct {
	Entity      string `json:"entity"`
	PostID      string `json:"post_id"`
	EmbedIndex  int64  `json:"embed_index"`
	EmbedURL    string `json:"embed_url"`
	EmbedEntity string `json:"embed_entity"`
	EmbedType   string `json:"embed_type"`
	EmbedID     string `json:"embed_id"`
}
