package emotes

import (
	"errors"
	"fmt"
	"strconv"
	"sync"
	"time"

	context2 "context"

	"code.justin.tv/chat/rails"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/xray"
	"golang.org/x/net/context"
)

// EmoteParserConfig stores configuration for EmoteParser
type EmoteParserConfig struct {
	RailsHostport  string
	TwitchClientID string
}

const (
	// EndorseEmote is "endorse"
	EndorseEmote = "endorse"
)

// Load configuration information
func (c *EmoteParserConfig) Load(dconf *distconf.Distconf) error {
	c.RailsHostport = dconf.Str("rails_hostport", "").Get()
	if c.RailsHostport == "" {
		return fmt.Errorf("cannot find rails_hostport")
	}

	c.TwitchClientID = dconf.Str("twitch_client_id", "").Get()
	if c.TwitchClientID == "" {
		return fmt.Errorf("cannot find twitch_client_id")
	}

	return nil
}

// EmoteParser retrieves and parses emotes
type EmoteParser struct {
	Config       *EmoteParserConfig
	Stats        *service_common.StatSender
	Log          log.Logger
	TwitchConfig twitchhttp.ClientConf `nilcheck:"ignore"`
	XRay         *xray.XRay
	Rails        rails.Rails `nilcheck:"ignore"`

	stopUpdateEmoteSetCache chan struct{}

	mu               sync.RWMutex
	validEmoticonIDs map[int]string
}

// Setup instantiates EmoteParser
func (T *EmoteParser) Setup() error {
	if T.Rails == nil {
		railsInst, err := rails.New(T.TwitchConfig, T.Config.TwitchClientID)
		if err != nil {
			return err
		}
		T.Rails = railsInst
	}

	T.updateEmoteSetCacheInBackground()

	return nil
}

// updateEmoteSetCacheInBackground calls UpdateEmoteSetCache once synchronously
// and then periodically in a goroutine
func (T *EmoteParser) updateEmoteSetCacheInBackground() {
	ctx := context.Background()
	T.updateEmoteIDSet(ctx)
	go func() {
		ticker := time.NewTicker(2 * time.Minute)
		for {
			select {
			case <-ticker.C:
				T.updateEmoteIDSet(ctx)
			case <-T.stopUpdateEmoteSetCache:
				return
			}
		}
	}()
}

func (T *EmoteParser) updateEmoteIDSet(ctxIn context.Context) {
	if err := T.XRay.Capture(ctxIn, "update_emote_set_ids", func(ctx context2.Context) error {
		ctx, cancel := context.WithTimeout(ctx, time.Second*20)
		defer cancel()
		emoticons, err := T.Rails.AllEmoticons(ctx, nil)
		if err != nil {
			return err
		}
		emoteIDs := make(map[int]string, len(emoticons.EmoticonSets)*10)
		for _, emoteList := range emoticons.EmoticonSets {
			for _, emote := range emoteList {
				emoteIDs[emote.Id] = emote.Pattern
			}
		}
		T.mu.Lock()
		T.validEmoticonIDs = emoteIDs
		T.mu.Unlock()
		return nil
	}); err != nil {
		T.Log.Log("err", err, "unable to fetch all emotes")
	}
}

// IsValidEmoteID returns if an emote is valid and not, as well as if it was able to know for sure.
// If the second value is false, then that means we could not trust the result of IsValidEmoteID (the cache was never
// able to be populated)
func (T *EmoteParser) IsValidEmoteID(emoteID int) (bool, bool) {
	T.mu.RLock()
	defer T.mu.RUnlock()
	if T.validEmoticonIDs == nil {
		return true, false
	}
	_, exists := T.validEmoticonIDs[emoteID]
	return exists, true
}

// GetEmoteName returns the emote name for a given ID string, e.g. "25" -> "Kappa"
func (T *EmoteParser) GetEmoteName(id string) (string, error) {
	if id == EndorseEmote {
		return id, nil
	}

	idInt, err := strconv.Atoi(id)
	if err != nil {
		return "", err
	}

	if _, ok := T.IsValidEmoteID(idInt); !ok {
		return "", errors.New("EmoteCache was never populated")
	}

	T.mu.RLock()
	defer T.mu.RUnlock()
	return T.validEmoticonIDs[idInt], nil
}
