package feedsedgeuser

import (
	"encoding/json"
	"io"
	"strings"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/service-common/feedsqs"
	"github.com/aws/aws-sdk-go/service/sqs"

	"golang.org/x/net/context"
)

type userHardDeleteMessage struct {
	UserID    string    `json:"user_id"`
	Timestamp time.Time `json:"timestamp"`
}

type UserHardDeleteSqsSourceConfig struct {
	feedsqs.SQSQueueProcessorConfig
	enableHardDeleteMessages *distconf.Bool
}

func (c *UserHardDeleteSqsSourceConfig) Load(dconf *distconf.Distconf) error {
	c.enableHardDeleteMessages = dconf.Bool("feeds-edge.enable-harddelete-msgs", true)
	return c.SQSQueueProcessorConfig.Verify(dconf, "feeds-edge.hard-delete", time.Second*3, 8, time.Minute*15)
}

type UserHardDeleteSqsSource struct {
	feedsqs.SQSQueueProcessor `nilcheck:"ignore"`
	HardDeleteConfig          *UserHardDeleteSqsSourceConfig
	UserDeleter               *Deleter
}

func (s *UserHardDeleteSqsSource) Setup() error {
	s.SQSQueueProcessor.ProcessMessage = s.processMessage
	return s.SQSQueueProcessor.Setup()
}

func parseHardDeleteMessage(in io.Reader) (userHardDeleteMessage, error) {
	msg := userHardDeleteMessage{}
	if err := json.NewDecoder(in).Decode(&msg); err != nil {
		return userHardDeleteMessage{}, err
	}
	return msg, nil
}

func (s *UserHardDeleteSqsSource) processHardDeleteMessage(ctx context.Context, msgBody io.Reader) error {
	fm, err := parseHardDeleteMessage(msgBody)
	if err != nil {
		return err
	}
	_, err = s.UserDeleter.HardDeleteUser(fm.UserID)
	return err
}

func (s *UserHardDeleteSqsSource) processMessage(ctx context.Context, m *sqs.Message) error {
	if !s.HardDeleteConfig.enableHardDeleteMessages.Get() {
		return nil
	}
	if m.Body != nil {
		return s.processHardDeleteMessage(ctx, strings.NewReader(*m.Body))
	}
	return nil
}
