package main

import (
	"errors"
	"net"
	"os"

	"net/http"

	"os/signal"

	friendship "code.justin.tv/chat/friendship/client"
	leviathan "code.justin.tv/chat/leviathan/client"
	"code.justin.tv/chat/rails"
	clue "code.justin.tv/chat/tmi/client"
	"code.justin.tv/chat/zuma/client"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/edge/client-incubator/nuclear_waste"
	"code.justin.tv/feeds/clients/duplo"
	"code.justin.tv/feeds/clients/feed-settings"
	"code.justin.tv/feeds/clients/masonry"
	"code.justin.tv/feeds/clients/shine"
	"code.justin.tv/feeds/ctxlog"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/feeds-edge/cmd/feeds-edge/internal/api"
	"code.justin.tv/feeds/feeds-edge/cmd/feeds-edge/internal/api/v2"
	"code.justin.tv/feeds/feeds-edge/cmd/feeds-edge/internal/emotes"
	"code.justin.tv/feeds/feeds-edge/cmd/feeds-edge/internal/user"
	"code.justin.tv/feeds/metrics/sfx/sfxstatsd"
	"code.justin.tv/feeds/metrics/statsdim"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/service-common/feedsqs"
	"code.justin.tv/feeds/service-common/redisold"
	"code.justin.tv/feeds/spade"
	"code.justin.tv/feeds/xray/plugins/ec2"
	"code.justin.tv/foundation/twitchclient"
	connections "code.justin.tv/identity/connections/client"
	twitter "code.justin.tv/web/twitter/client"
	users "code.justin.tv/web/users-service/client"
	"github.com/afex/hystrix-go/hystrix"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/garyburd/redigo/redis"
)

const (
	teamName    = "feeds"
	serviceName = "feeds-edge"
)

// CodeVersion is set by build script
var CodeVersion string

var instance = service{
	osExit: os.Exit,
	serviceCommon: service_common.ServiceCommon{
		ConfigCommon: service_common.ConfigCommon{
			Team:       teamName,
			Service:    serviceName,
			OsGetenv:   os.Getenv,
			OsHostname: os.Hostname,
		},
		CodeVersion:    CodeVersion,
		SfxSetupConfig: sfxStastdConfig(),
	},
}

func sfxStastdConfig() sfxstatsd.SetupConfig {
	return sfxstatsd.SetupConfig{
		CounterMetricRules: []*statsdim.ConfigurableDelimiterMetricRule{
			{
				MetricPath:    "api.*.code.*",
				DimensionsMap: "-.handler.-.http_code",
				MetricName:    "http.hits",
				AdditionalDimensions: map[string]string{
					"version": "v1",
				},
			},
			{
				MetricPath:    "api.drainCommentUpdateChan.*",
				DimensionsMap: "-.%.%",
			},
			{
				MetricPath:    "api.drainPostUpdateChan.*",
				DimensionsMap: "-.%.%",
			},
			{
				MetricPath:    "api.backfillCommentEmote.*",
				DimensionsMap: "-.%.%",
			},
			{
				MetricPath:    "embed.*",
				DimensionsMap: "%.error_type",
				MetricName:    "context_errors",
			},
			{
				MetricPath:    "api.v2.*.code.*",
				DimensionsMap: "-.version.handler.-.http_code",
				MetricName:    "http.hits",
			},
			{
				MetricPath:    "api.v2.*.invalidFeedEntity",
				DimensionsMap: "-.version.namespace.%",
				MetricName:    "conversion_error",
			},
			{
				MetricPath:    "api.v2.*.invalidEmbedEntity",
				DimensionsMap: "-.version.namespace.%",
				MetricName:    "conversion_error",
			},
			{
				MetricPath:    "api.get_feed.stats.*.*.*",
				DimensionsMap: "-.handler.-.feed_type.is_enabled.%",
				MetricName:    "feed.fetch_stat_counters",
			},
		},
		TimingMetricRules: []*statsdim.ConfigurableDelimiterMetricRule{
			{
				MetricPath:    "rails.kraken.*.*",
				DimensionsMap: "%.%.handler.http_code",
				MetricName:    "request_times",
			},
			{
				MetricPath:    "api.get_feed.stats.*.*.*",
				DimensionsMap: "-.handler.-.feed_type.is_enabled.%",
				MetricName:    "feed.fetch_stats",
			},
			{
				MetricPath:    "api.*.time",
				DimensionsMap: "%.handler.%",
				AdditionalDimensions: map[string]string{
					"version": "v1",
				},
			},
			{
				MetricPath:    "api.populate.*",
				DimensionsMap: "%.%.populate_step",
			},
			{
				MetricPath:    "api.v2.*.time",
				DimensionsMap: "%.version.handler.%",
			},
			{
				MetricPath:    "service.*.*.extract.*",
				DimensionsMap: "-.rpc_service.handler.%.http_code",
			},
			{
				MetricPath:    "authorization.*",
				DimensionsMap: "%.auth_type",
				MetricName:    "total_time",
			},
			{
				MetricPath:    "authorization.can_reply.*",
				DimensionsMap: "%.%.auth_check",
				MetricName:    "total_time",
			},
		},
	}
}

type injectables struct {
	FriendshipClient   api.FriendshipClient
	ShineClient        api.ShineClient
	NuclearWasteClient nuclear_waste.Client
	Rails              rails.Rails
	SpadeClient        v2.SpadeClient
}

type service struct {
	injectables
	osExit        func(code int)
	onListen      func(net.Addr)
	Runner        service_common.ServiceRunner
	serviceCommon service_common.ServiceCommon
	sigChan       chan os.Signal

	server            api.HTTPServer
	authorization     api.Authorization
	hardDeleteUserSQS feedsedgeuser.UserHardDeleteSqsSource

	clueClient         api.ClueClient
	duploClient        *duplo.Client
	feedSettingsClient *feedsettings.Client
	emoteParser        emotes.EmoteParser
	redisPool          *redis.Pool
	redisPrefix        string
	connectionsClient  connections.Client
	zumaClient         zuma.Client
	twitterClient      *twitter.Client
	usersClient        users.Client
	leviathanClient    leviathan.Client
	configs            struct {
		clientConfig         clientConfig
		httpConfig           api.HTTPConfig
		masonryConfig        masonry.Config
		duploConfig          duplo.Config
		feedSettingsConfig   feedsettings.Config
		emoteParserConfig    emotes.EmoteParserConfig
		redisConfig          feedsEdgeRedisConfig
		twitterConfig        twitterConfig
		leviathanConfig      leviathanConfig
		duploCacherConfig    api.DuploCacherConfig
		makoHystrixConfigs   makoHystrixConfigs
		spadeConfig          spade.Config
		apiConfigv2          v2.Config
		userHardDeleteConfig feedsedgeuser.UserHardDeleteSqsSourceConfig
	}
}

type clientConfig struct {
	friendshipURL   *distconf.Str
	clueURL         *distconf.Str
	zumaURL         *distconf.Str
	shineURL        *distconf.Str
	connectionsURL  *distconf.Str
	usersURL        *distconf.Str
	nuclearwasteURL *distconf.Str
}

func (c *clientConfig) Load(dconf *distconf.Distconf) error {
	c.friendshipURL = dconf.Str("feeds-edge.clients.friendship.url", "")
	if c.friendshipURL.Get() == "" {
		return errors.New("unable to find a valid friendship url")
	}
	c.clueURL = dconf.Str("feeds-edge.clients.clue.url", "")
	if c.clueURL.Get() == "" {
		return errors.New("unable to find a valid clue url")
	}
	c.connectionsURL = dconf.Str("feeds-edge.clients.connections.url", "")
	if c.connectionsURL.Get() == "" {
		return errors.New("unable to find a valid connections url")
	}
	c.usersURL = dconf.Str("feeds-edge.clients.users.url", "")
	if c.usersURL.Get() == "" {
		return errors.New("unable to find a valid users-service url")
	}
	c.zumaURL = dconf.Str("feeds-edge.clients.zuma.url", "")
	if c.zumaURL.Get() == "" {
		return errors.New("unable to find a valid zuma url")
	}

	c.shineURL = dconf.Str("shine.http_endpoint", "")
	if c.shineURL.Get() == "" {
		return errors.New("unable to find a valid shine url")
	}

	c.nuclearwasteURL = dconf.Str("feeds-edge.clients.nuclear_waste", "https://api.internal.twitch.tv")
	if c.nuclearwasteURL.Get() == "" {
		return errors.New("unable to find a valid nuclear waste URL")
	}
	return nil
}

type makoHystrixConfigs struct {
	timeout               *distconf.Int
	maxConcurrentRequests *distconf.Int
}

func (c *makoHystrixConfigs) Load(d *distconf.Distconf) error {
	c.timeout = d.Int("hystrix.mako_emote_entitlements.timeout", 1000)
	c.maxConcurrentRequests = d.Int("hystrix.mako_emote_entitlements.max_concurrent_requests", 5000)
	return nil
}

func (c *makoHystrixConfigs) Setup() {
	c.timeout.Watch(c.configureHystrix)
	c.maxConcurrentRequests.Watch(c.configureHystrix)
	c.configureHystrix()
}

func (c *makoHystrixConfigs) configureHystrix() {
	hystrix.ConfigureCommand("mako_emote_entitlements", hystrix.CommandConfig{
		Timeout:               int(c.timeout.Get()),
		MaxConcurrentRequests: int(c.maxConcurrentRequests.Get()),
	})
}

type twitterConfig struct {
	allowPosting   *distconf.Bool
	consumerKey    *distconf.Str
	consumerSecret *distconf.Str
}

func (t *twitterConfig) Load(d *distconf.Distconf) error {
	t.allowPosting = d.Bool("feeds-edge.enable_twitter_syndication", false)
	return nil
}

func (t *twitterConfig) LoadSecrets(d *distconf.Distconf) error {
	t.consumerKey = d.Str("twitter_consumer_key", "")
	t.consumerSecret = d.Str("twitter_consumer_secret", "")
	return nil
}

type feedsEdgeRedisConfig struct {
	redisold.RedisConfig
}

func (r *feedsEdgeRedisConfig) Load(d *distconf.Distconf) error {
	return r.RedisConfig.Verify("redis", d)
}

func (r *feedsEdgeRedisConfig) LoadSecrets(d *distconf.Distconf) error {
	return r.RedisConfig.VerifyAuth("redis", d)
}

type leviathanConfig struct {
	url            *distconf.Str
	allowReporting *distconf.Bool
	authToken      *distconf.Str
}

func (l *leviathanConfig) Load(d *distconf.Distconf) error {
	l.allowReporting = d.Bool("feeds-edge.clients.leviathan.allow_reports", false)
	l.url = d.Str("feeds-edge.clients.leviathan.url", "")

	if l.allowReporting.Get() && l.url.Get() == "" {
		return errors.New("unable to find a valid leviathan url")
	}

	return nil
}

func (l *leviathanConfig) LoadSecrets(d *distconf.Distconf) error {
	l.authToken = d.Str("leviathan_auth_token", "")

	if l.allowReporting.Get() && l.url.Get() == "" {
		return errors.New("unable to find a valid leviathan authorization token")
	}

	return nil
}

func (s *service) setupConfigs() error {
	if err := s.serviceCommon.Setup(); err != nil {
		return err
	}

	if err := service_common.LoadConfigs(
		s.serviceCommon.Config,
		&s.configs.clientConfig,
		&s.configs.httpConfig,
		&s.configs.masonryConfig,
		&s.configs.duploConfig,
		&s.configs.emoteParserConfig,
		&s.configs.redisConfig,
		&s.configs.feedSettingsConfig,
		&s.configs.leviathanConfig,
		&s.configs.duploCacherConfig,
		&s.configs.makoHystrixConfigs,
		&s.configs.apiConfigv2,
		&s.configs.userHardDeleteConfig,
		&s.configs.spadeConfig); err != nil {
		return err
	}

	if err := s.configs.redisConfig.LoadSecrets(s.serviceCommon.Secrets); err != nil {
		return err
	}
	if err := s.setupTwitterConfig(); err != nil {
		return err
	}
	if err := s.configs.leviathanConfig.LoadSecrets(s.serviceCommon.Secrets); err != nil {
		return nil
	}

	return nil
}

func (s *service) setupTwitterConfig() error {
	sandstormDistConf := s.serviceCommon.SetupAdditionalSandstormDistconf("identity", "connections")
	if err := s.configs.twitterConfig.LoadSecrets(sandstormDistConf); err != nil {
		return err
	}
	if err := s.configs.twitterConfig.Load(s.serviceCommon.Config); err != nil {
		return err
	}

	if s.configs.twitterConfig.allowPosting.Get() && (s.configs.twitterConfig.consumerKey.Get() == "" || s.configs.twitterConfig.consumerSecret.Get() == "") {
		return errors.New("invalid Twitter key or secret provided with Twitter posting enabled")
	}

	return nil
}

func (s *service) httpClient(host string) twitchhttp.ClientConf {
	rtWrappers := []func(http.RoundTripper) http.RoundTripper{
		s.serviceCommon.XRay.RoundTripper,
	}
	return twitchhttp.ClientConf{
		Host:                 host,
		Stats:                s.serviceCommon.Statsd,
		RoundTripperWrappers: rtWrappers,
		TurnOffXRay:          true,
		TurnOffChitin:        true,
		DimensionKey:         &s.serviceCommon.CtxDimensions,
	}
}

func (s *service) foundationClient(host string) twitchclient.ClientConf {
	rtWrappers := []func(http.RoundTripper) http.RoundTripper{
		s.serviceCommon.XRay.RoundTripper,
	}
	return twitchclient.ClientConf{
		Host:                 host,
		Stats:                s.serviceCommon.Statsd,
		RoundTripperWrappers: rtWrappers,
		TurnOffXRay:          true,
		TurnOffChitin:        true,
		ElevateKey:           s.serviceCommon.ElevateLogKey,
		Logger:               s.serviceCommon.Log,
		DimensionKey:         &s.serviceCommon.CtxDimensions,
	}
}

func (s *service) setupClients() error {
	var err error

	if err = s.setupInjectableClients(); err != nil {
		return err
	}

	if s.clueClient, err = clue.NewClient(s.foundationClient(s.configs.clientConfig.clueURL.Get())); err != nil {
		return err
	}

	if s.usersClient, err = users.NewClient(s.httpClient(s.configs.clientConfig.usersURL.Get())); err != nil {
		return err
	}

	if s.connectionsClient, err = connections.NewClient(s.foundationClient(s.configs.clientConfig.connectionsURL.Get())); err != nil {
		return err
	}

	if s.zumaClient, err = zuma.NewClient(s.httpClient(s.configs.clientConfig.zumaURL.Get())); err != nil {
		return err
	}

	if s.ShineClient, err = shine.NewClient(s.foundationClient(s.configs.clientConfig.shineURL.Get())); err != nil {
		return err
	}

	if s.NuclearWasteClient, err = nuclear_waste.NewClient(s.foundationClient(s.configs.clientConfig.nuclearwasteURL.Get())); err != nil {
		return err
	}

	s.twitterClient = twitter.NewClient(s.configs.twitterConfig.consumerKey.Get(), s.configs.twitterConfig.consumerSecret.Get(), s.connectionsClient)

	if err != nil {
		return err
	}

	return s.setupLeviathanClient()
}

func (s *service) setupInjectableClients() error {
	var err error

	if s.FriendshipClient == nil {
		if s.FriendshipClient, err = friendship.NewClient(s.httpClient(s.configs.clientConfig.friendshipURL.Get())); err != nil {
			return err
		}
	}
	return s.setupLeviathanClient()
}

func (s *service) setupLeviathanClient() error {
	// Leviathan is the service that handles creating reports for moderation.  Leviathan only has a Production
	// endpoint, and so we need to ensure that our dev and integration environments don't create reports.
	if !s.configs.leviathanConfig.allowReporting.Get() {
		s.leviathanClient = leviathan.NoopClient{}
		return nil
	}

	var err error
	s.leviathanClient, err = leviathan.NewClient(s.foundationClient(s.configs.leviathanConfig.url.Get()))
	return err
}

func (s *service) setup() error {
	err := s.setupConfigs()
	if err != nil {
		return err
	}

	err = s.setupClients()
	if err != nil {
		return err
	}

	s.configs.makoHystrixConfigs.Setup()

	s.emoteParser = emotes.EmoteParser{
		Config:       &s.configs.emoteParserConfig,
		Log:          s.serviceCommon.Log,
		TwitchConfig: s.httpClient(s.configs.emoteParserConfig.RailsHostport),
		XRay:         s.serviceCommon.XRay,
		Rails:        s.Rails,
	}
	if err = s.emoteParser.Setup(); err != nil {
		return err
	}

	s.redisPool = &redis.Pool{
		MaxIdle:   int(s.configs.redisConfig.RedisConfig.MaxConns.Get()),
		MaxActive: int(s.configs.redisConfig.RedisConfig.MaxConns.Get()),
		Dial:      s.configs.redisConfig.RedisConfig.RedisConn,
	}

	if err := s.serviceCommon.XRay.WithPlugin(&ec2.Plugin{}); err != nil {
		s.serviceCommon.Log.Log("err", err, "unable to load ec2 information")
	}

	return nil
}

func (s *service) inject() {
	httpClient := s.serviceCommon.XRay.Client(&http.Client{})

	s.duploClient = &duplo.Client{
		Config: &s.configs.duploConfig,
		RequestDoer: &ctxlog.LoggedDoer{
			Logger: s.serviceCommon.Log,
			C:      &s.serviceCommon.Ctxlog,
			Client: httpClient,
		},
		NewHTTPRequest: ctxlog.WrapHTTPRequestWithCtxlog(&s.serviceCommon.Ctxlog, http.NewRequest),
	}

	userDeleter := feedsedgeuser.Deleter{
		Duplo: s.duploClient,
	}

	s.feedSettingsClient = &feedsettings.Client{
		Config: &s.configs.feedSettingsConfig,
		RequestDoer: &ctxlog.LoggedDoer{
			Logger: s.serviceCommon.Log,
			C:      &s.serviceCommon.Ctxlog,
			Client: httpClient,
		},
		NewHTTPRequest: ctxlog.WrapHTTPRequestWithCtxlog(&s.serviceCommon.Ctxlog, http.NewRequest),
	}

	s.authorization = api.Authorization{
		SettingsClient:   s.feedSettingsClient,
		FriendshipClient: s.FriendshipClient,
		ClueClient:       s.clueClient,
		Zuma:             s.zumaClient,
		Log:              s.serviceCommon.Log,
		Stats: &service_common.StatSender{
			SubStatter:   s.serviceCommon.Statsd.NewSubStatter("authorization"),
			ErrorTracker: &s.serviceCommon.ErrorTracker,
		},
		Config: &s.configs.httpConfig,
	}

	s.emoteParser.Stats = &service_common.StatSender{
		SubStatter:   s.serviceCommon.Statsd.NewSubStatter("emotes"),
		ErrorTracker: &s.serviceCommon.ErrorTracker,
	}

	if s.SpadeClient == nil {
		spadeClient := &spade.Client{
			HTTPClient: httpClient,
			Config:     &s.configs.spadeConfig,
			Logger:     s.serviceCommon.Log,
		}
		spadeClient.Setup()
		s.SpadeClient = spadeClient
	}

	s.server = api.HTTPServer{
		BaseHTTPServer: service_common.BaseHTTPServer{
			Config: &s.configs.httpConfig.BaseHTTPServerConfig,
			Stats: &service_common.StatSender{
				SubStatter:   s.serviceCommon.Statsd.NewSubStatter("api"),
				ErrorTracker: &s.serviceCommon.ErrorTracker,
			},
			Dims:        &s.serviceCommon.CtxDimensions,
			Log:         s.serviceCommon.Log,
			ElevateKey:  s.serviceCommon.ElevateLogKey,
			Ctxlog:      &s.serviceCommon.Ctxlog,
			OnListen:    s.onListen,
			PanicLogger: s.serviceCommon.PanicLogger,
			XRay:        s.serviceCommon.XRay,
		},
		Config:      &s.configs.httpConfig,
		Zuma:        s.zumaClient,
		UserDeleter: &userDeleter,

		Masonry: &masonry.Client{
			Config: &s.configs.masonryConfig,
			RequestDoer: &ctxlog.LoggedDoer{
				Logger: s.serviceCommon.Log,
				C:      &s.serviceCommon.Ctxlog,
				Client: httpClient,
			},
			NewHTTPRequest: ctxlog.WrapHTTPRequestWithCtxlog(&s.serviceCommon.Ctxlog, http.NewRequest),
		},
		Duplo:              s.duploClient,
		Shine:              s.ShineClient,
		NuclearWaste:       s.NuclearWasteClient,
		Authorization:      &s.authorization,
		FeedSettingsClient: s.feedSettingsClient,
		EmoteParser:        &s.emoteParser,
		Cooldowns: &v2.RedisCooldowns{
			RedisPool: s.redisPool,
			Log:       s.serviceCommon.Log,
			Prefix:    s.redisPrefix,
		},
		DuploCacher: api.DuploCacher{
			Duplo:             s.duploClient,
			DuploCacherConfig: &s.configs.duploCacherConfig,
		},
		TwitterClient:   s.twitterClient,
		UsersClient:     s.usersClient,
		LeviathanClient: s.leviathanClient,
		ClueClient:      s.clueClient,
		SpadeClient:     s.SpadeClient,
		Filter: &api.Filter{
			UsersClient: s.usersClient,
			Log:         s.serviceCommon.Log,
		},
	}
	s.server.APIv2 = &v2.API{
		Log:           s.serviceCommon.Log,
		Config:        &s.configs.apiConfigv2,
		Duplo:         s.server.Duplo,
		Masonry:       s.server.Masonry,
		EmoteParser:   s.server.EmoteParser,
		Authorization: &s.authorization,
		Shine:         s.ShineClient,
		Zuma:          s.zumaClient,
		TwitterClient: s.twitterClient,
		UsersClient:   s.usersClient,
		SpadeClient:   s.SpadeClient,
		Cooldowns:     s.server.Cooldowns,
		Stats: &service_common.StatSender{
			SubStatter:   s.serviceCommon.Statsd.NewSubStatter("api.v2"),
			ErrorTracker: &s.serviceCommon.ErrorTracker,
		},
	}
	s.server.BaseHTTPServer.SetupRoutes = s.server.SetupRoutes

	session, awsConf := service_common.CreateAWSSession(s.serviceCommon.Config)
	s.hardDeleteUserSQS = feedsedgeuser.UserHardDeleteSqsSource{
		SQSQueueProcessor: feedsqs.SQSQueueProcessor{
			Log: s.serviceCommon.Log,
			Sqs: sqs.New(session, awsConf...),
			Stats: &service_common.StatSender{
				SubStatter:   s.serviceCommon.Statsd.NewSubStatter("sqs_source.user_hard_delete"),
				ErrorTracker: &s.serviceCommon.ErrorTracker,
			},
			Conf: &s.configs.userHardDeleteConfig.SQSQueueProcessorConfig,
			Ch:   &s.serviceCommon.Ctxlog,
		},
		HardDeleteConfig: &s.configs.userHardDeleteConfig,
		UserDeleter:      &userDeleter,
	}

	s.Runner = service_common.ServiceRunner{
		Log: s.serviceCommon.Log,
		Services: []service_common.Service{
			&s.server, &s.serviceCommon, s.SpadeClient, &s.hardDeleteUserSQS,
		},
		SigChan:      s.sigChan,
		SignalNotify: signal.Notify,
	}
	// Cannot just pass f because f contains private members that I cannot nil check via reflection
	res := (&service_common.NilCheck{
		IgnoredPackages: []string{"net/http", "garyburd/redigo/redis"},
	}).Check(s, s.Runner)
	res.MustBeEmpty(os.Stderr)
}

func (s *service) main() {
	if err := s.setup(); err != nil {
		service_common.SetupLogger.Log("err", err, "Unable to load initial config")
		s.osExit(1)
		return
	}
	s.inject()
	if err := s.Runner.Execute(); err != nil {
		service_common.SetupLogger.Log("err", err, "wait to end finished with an error")
		s.osExit(1)
		return
	}
	s.serviceCommon.Log.Log("done with service")
}

func main() {
	instance.main()
}
