// +build integration

package main

import (
	"encoding/json"
	"fmt"
	"math/rand"
	"net"
	"net/http"
	"os"
	"reflect"
	"strconv"
	"strings"
	"syscall"
	"testing"
	"time"

	"code.justin.tv/chat/friendship/client"
	"code.justin.tv/chat/rails"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/clients/masonry"
	"code.justin.tv/feeds/clients/shine"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-common/verb"
	"code.justin.tv/feeds/feeds-edge/cmd/feeds-edge/internal/api"
	"code.justin.tv/feeds/feeds-edge/cmd/feeds-edge/internal/api/mocks"
	"code.justin.tv/feeds/feeds-edge/cmd/feeds-edge/internal/api/v2"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/spade"
	users "code.justin.tv/web/users-service/client"
	"code.justin.tv/web/users-service/models"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/satori/go.uuid"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
	"golang.org/x/net/context"
	stack "gopkg.in/stack.v1"
)

const (
	testUserID                = "27184041"
	testUserID2               = "181631"
	testChannelID             = testUserID
	testChannelID2            = testUserID2
	testChannelIDWithEmoteSet = "26551727"  // mang0
	testMakoUserID            = "125199910" // richtang2; Mako entitled user
	testMakoChannelID         = testMakoUserID
	testStrictModerationID    = "153061467" // ihatebadwords, Bern's test user.
	testReporterUserID        = "11111111"  // arbitrary user ID that can be parsed into an int64.

	// FIXME: depends on staging for these reasons:
	// 1. this user exists on staging
	// 2. user has comments disabled
	// 3. has posts on his channel
	testChannelIDCommentsUnauthorized = "25242039"

	testEmoteKappa         = "25"
	testEmotePogChamp      = "88"
	testEmoteRegex1        = "6"     // O_o
	testEmoteMangoUSA      = "20387" // from mang0 channel emote set
	testEmoteIDNonExistent = "31337"
	testMakoEmoteRegex     = "160148"
	testVodID              = "vod:157063727" // vod hosted on twitch.tv/twitch
	testInvalidVodID       = "vod:168913010"
	testVodAuthorID        = "12826"
)

var sampleBirthday = models.Birthday{
	Day:   27,
	Month: time.November,
	Year:  1982,
}

func timestamp() string {
	return time.Now().Format("2006-01-02-15.04.05.000000000-MST")
}

func postEntity(id string) string {
	return "post:" + id
}

func userEntity(id string) string {
	return "user:" + id
}

func toMockFeed(postIDs ...string) *string {
	entities := make([]string, 0, len(postIDs))
	for _, postID := range postIDs {
		if _, err := entity.Decode(postID); err == nil {
			entities = append(entities, postID)
		} else {
			entities = append(entities, "post:"+postID)
		}

	}
	mockFeed := strings.Join(entities, ",")
	return &mockFeed
}

func toMockFeedOfActivity(feedID string, activity ...masonry.Activity) *string {
	ret := masonry.Feed{
		ID:     feedID,
		Cursor: "",
		Items:  activity,
	}
	retBytes, err := json.Marshal(ret)
	if err != nil {
		panic("strange")
	}
	return aws.String(string(retBytes))
}

type createPostParams struct {
	UserID    string    `json:"user_id"`
	Body      string    `json:"body"`
	EmbedURLs *[]string `json:"embed_urls"`
}

func createPostWithParams(ts *testSetup, userID string, params *createPostParams) (*api.Post, error) {
	url := "/v1/posts?user_id=" + userID

	var ret *api.Post
	err := request(ts, "POST", url, params, &ret)
	return ret, err
}

func createPost(ts *testSetup, channelID, body, userID string) (*api.Post, error) {
	params := &createPostParams{
		UserID: channelID,
		Body:   body,
	}

	return createPostWithParams(ts, userID, params)
}

// WARNING: Not the same as createPost
func createPostV2(ts *testSetup, userID string, params *createPostParams) (*v2.Post, error) {
	url := "/v2/create_post?user_id=" + userID

	var ret *v2.CreatePostResponse
	err := request(ts, "POST", url, params, &ret)
	if err != nil {
		return nil, err
	}
	return ret.Post, err
}

func deletePost(ts *testSetup, postID, userID string) (*api.Post, error) {
	url := fmt.Sprintf("/v1/posts/%s?user_id=%s", postID, userID)
	var ret *api.Post
	err := request(ts, "DELETE", url, nil, &ret)
	return ret, err
}

func deletePostV2(ts *testSetup, postID, userID string) (*v2.Post, error) {
	url := fmt.Sprintf("/v2/delete_post?post_id=%s&user_id=%s", postID, userID)
	var ret *v2.Post
	err := request(ts, "DELETE", url, nil, &ret)
	return ret, err
}

func createShareV2(ts *testSetup, userID string, targetEntity entity.Entity) (*v2.Share, error) {
	url := "/v2/create_share?user_id=" + userID

	params := struct {
		TargetEntity entity.Entity `json:"target_entity"`
	}{
		TargetEntity: targetEntity,
	}
	var ret *v2.Share
	err := request(ts, "POST", url, params, &ret)
	return ret, err
}

func createShare(ts *testSetup, userID string, targetEntity entity.Entity) (*api.Share, error) {
	url := "/v1/shares?user_id=" + userID

	params := struct {
		TargetEntity entity.Entity `json:"target_entity"`
	}{
		TargetEntity: targetEntity,
	}
	var ret *api.Share
	err := request(ts, "POST", url, params, &ret)
	return ret, err
}

func deleteShareV2(ts *testSetup, shareID, userID string) (*v2.Share, error) {
	url := fmt.Sprintf("/v2/delete_share?share_id=%s&user_id=%s", shareID, userID)
	var ret *v2.Share
	err := request(ts, "DELETE", url, nil, &ret)
	return ret, err
}

func deleteShare(ts *testSetup, shareID, userID string) (*api.Share, error) {
	url := fmt.Sprintf("/v1/shares/%s?user_id=%s", shareID, userID)
	var ret *api.Share
	err := request(ts, "DELETE", url, nil, &ret)
	return ret, err
}

func getShare(ts *testSetup, shareID, userID string) (*api.Share, error) {
	url := fmt.Sprintf("/v1/shares/%s?user_id=%s", shareID, userID)
	var ret *api.Share
	err := request(ts, "GET", url, nil, &ret)
	return ret, err
}

func getSharesByIDs(ts *testSetup, shareIDs []string) (*v2.Shares, error) {
	url := fmt.Sprintf("/v2/get_shares_by_ids?share_ids=%s", strings.Join(shareIDs, ","))
	var ret *v2.Shares
	err := request(ts, "GET", url, nil, &ret)
	return ret, err
}

func getSharesSummaries(ts *testSetup, userID string, parentEntity entity.Entity) (*api.SharesSummaries, error) {
	url := fmt.Sprintf("/v1/shares_summaries?author_ids=%s&parent_entities=%s", userID, parentEntity.Encode())
	var ret *api.SharesSummaries
	err := request(ts, "GET", url, nil, &ret)
	return ret, err
}

func getEmbed(ts *testSetup, embedURL string, autoplay bool) (*api.Embed, error) {
	url := "/v1/embed?url=" + embedURL + "&autoplay=" + strconv.FormatBool(autoplay)
	var ret *api.Embed
	err := request(ts, "GET", url, nil, &ret)
	return ret, err
}

func getPost(ts *testSetup, postID, userID string) (*api.Post, error) {
	url := fmt.Sprintf("/v1/posts/%s?user_id=%s", postID, userID)

	var ret *api.Post
	err := request(ts, "GET", url, nil, &ret)
	return ret, err
}

func getPostsByIDs(ts *testSetup, postIDs []string) (*v2.Posts, error) {
	url := fmt.Sprintf("/v2/get_posts_by_ids?post_ids=%s", strings.Join(postIDs, ","))

	var ret *v2.Posts
	err := request(ts, "GET", url, nil, &ret)
	return ret, err
}

func getPostsPermissionsByIDs(ts *testSetup, userID string, postIDs []string) (*v2.PostsPermissions, error) {
	url := fmt.Sprintf("/v2/get_posts_permissions_by_ids?user_id=%s&post_ids=%s", userID, strings.Join(postIDs, ","))

	var ret v2.PostsPermissions
	err := request(ts, "GET", url, nil, &ret)
	return &ret, err
}

func unshareByTarget(ts *testSetup, userID string, authorID string, targetEntity entity.Entity) (*api.Shares, error) {
	url := fmt.Sprintf("/v1/unshare_by_target?author_id=%s&user_id=%s&target_entities=%s", authorID, userID, targetEntity.Encode())
	var ret *api.Shares
	err := request(ts, "DELETE", url, nil, &ret)
	return ret, err
}

func getReactionsByParents(ts *testSetup, parents []string, userID string) (*api.BatchReactionsResponse, error) {
	url := fmt.Sprintf("/v1/reactions?parent_entity=%s&user_id=%s", strings.Join(parents, ","), userID)
	var ret *api.BatchReactionsResponse
	err := request(ts, "GET", url, nil, &ret)
	return ret, err
}

func getReactionsByEntities(ts *testSetup, entities []string, userID string) (*v2.ReactionsSummaries, error) {
	url := fmt.Sprintf("/v2/get_reactions_by_entities?entities=%s&user_id=%s&strong_consistency=true", strings.Join(entities, ","), userID)
	var ret *v2.ReactionsSummaries
	err := request(ts, "GET", url, nil, &ret)
	return ret, err
}

func createReaction(ts *testSetup, parentEntity string, emoteID, userID string) (*api.UserReactions, error) {
	url := fmt.Sprintf("/v1/reactions/%s/%s?user_id=%s", parentEntity, emoteID, userID)
	var ret *api.UserReactions
	err := request(ts, "PUT", url, nil, &ret)
	return ret, err
}

func createReactionV2(ts *testSetup, parentEntity string, emoteID, userID string) (*string, error) {
	url := fmt.Sprintf("/v2/create_reaction?parent_entity=%s&user_id=%s&emote_id=%s", parentEntity, userID, emoteID)
	var ret *string
	err := request(ts, "PUT", url, nil, &ret)
	return ret, err
}

func deleteReaction(ts *testSetup, parentEntity string, emoteID, userID string) (*api.UserReactions, error) {
	url := fmt.Sprintf("/v1/reactions/%s/%s?user_id=%s", parentEntity, emoteID, userID)
	var ret *api.UserReactions
	err := request(ts, "DELETE", url, nil, &ret)
	return ret, err
}

func deleteReactionV2(ts *testSetup, parentEntity string, emoteID, userID string) (*string, error) {
	url := fmt.Sprintf("/v2/delete_reaction?parent_entity=%s&user_id=%s&emote_id=%s", parentEntity, userID, emoteID)
	var ret *string
	err := request(ts, "DELETE", url, nil, &ret)
	return ret, err
}

func getFeed(ts *testSetup, feedID, userID string, mockFeed *string) (*api.ChannelFeed, error) {
	ret, _, err := getFeedAndHeaders(ts, feedID, userID, nil, mockFeed)
	return ret, err
}

func getFeedWithLimit(ts *testSetup, feedID, userID string, limit string) (*api.ChannelFeed, error) {
	ret, _, err := getFeedAndHeaders(ts, feedID, userID, &limit, nil)
	return ret, err
}

func getFeedEntities(ts *testSetup, feedID, userID string, mockFeed *string) (*v2.Feed, error) {
	url := fmt.Sprintf("/v2/get_feed?feed_id=%s&user_id=%s", feedID, userID)
	if mockFeed != nil {
		url += "&mockFeed=" + *mockFeed
	}
	var ret *v2.Feed
	err := request(ts, "GET", url, nil, &ret)
	return ret, err
}

func getSuggestedFeeds(ts *testSetup, userID string) (*v2.FeedIDs, error) {
	url := fmt.Sprintf("/v2/suggested_feeds?user_id=%s", userID)
	var ret *v2.FeedIDs
	err, _ := requestAndHeaders(ts, "GET", url, nil, &ret)
	return ret, err
}

func getFeedAndHeaders(ts *testSetup, feedID, userID string, limit, mockFeed *string) (*api.ChannelFeed, http.Header, error) {
	url := fmt.Sprintf("/v1/feeds/%s?user_id=%s", feedID, userID)
	if limit != nil {
		url += "&limit=" + *limit
	}
	if mockFeed != nil {
		url += "&mockFeed=" + *mockFeed
	}
	var ret *api.ChannelFeed
	err, headers := requestAndHeaders(ts, "GET", url, nil, &ret)
	return ret, headers, err
}

func getPermissions(ts *testSetup, feedID string, userID string) (*api.PostPermissions, error) {
	url := fmt.Sprintf("/v1/permissions/%s?user_id=%s", feedID, userID)
	var ret *api.PostPermissions
	err := request(ts, "GET", url, nil, &ret)
	return ret, err
}

func getSettings(ts *testSetup, settingsEntity string, userID string) (*api.Settings, error) {
	url := fmt.Sprintf("/v1/settings/%s?user_id=%s", settingsEntity, userID)
	var ret *api.Settings
	err := request(ts, "GET", url, nil, &ret)
	return ret, err
}

type updateSettingsParams struct {
	ChannelFeedEnabled *bool `json:"channel_feed_enabled,omitempty"`
}

func updateSettings(ts *testSetup, settingsEntity string, userID string, opts *updateSettingsParams) (*api.Settings, error) {
	url := fmt.Sprintf("/v1/settings/%s?user_id=%s", settingsEntity, userID)
	var ret *api.Settings
	err := request(ts, "POST", url, opts, &ret)
	return ret, err
}

func reportPost(ts *testSetup, postID, userID, reason string) error {
	url := fmt.Sprintf("/v1/posts/%s/report?user_id=%s", postID, userID)

	params := api.ReportParams{
		Reason: reason,
	}

	return request(ts, "POST", url, params, nil)
}

func TestIntegration_Post(t *testing.T) {
	t.Parallel()
	ts := startServer(t, injectables{})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func(c C) {
		So(ts.Setup(), ShouldBeNil)
		c.Reset(ts.cancelFunc)

		Convey("Should be able to create a post and get the post back", func() {
			testBody := "TestIntegration_Post " + timestamp()
			created, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.Body, ShouldEqual, testBody)
			So(created.ShareSummary.ShareCount, ShouldEqual, 0)
			So(created.ShareSummary.UserIDs, ShouldBeEmpty)

			post, err := getPost(ts, created.ID, testUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			So(post.ID, ShouldEqual, created.ID)
			So(post.Body, ShouldEqual, testBody)
			So(post.ShareSummary.ShareCount, ShouldEqual, 0)
			So(post.ShareSummary.UserIDs, ShouldBeEmpty)
		})

		Convey("Should not be able to create a post as a different user", func() {
			testBody := "TestIntegration_Post " + timestamp()
			created, err := createPost(ts, testChannelID, testBody, testUserID2)
			So(errorCode(err), ShouldEqual, http.StatusForbidden)
			So(created, ShouldBeNil)
		})

		Convey("Should be able to create a post and get the post back in V2", func() {
			testBody := "TestIntegration_Post " + timestamp()
			created, err := createPostV2(ts, testUserID, &createPostParams{UserID: testChannelID, Body: testBody})
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.Body, ShouldEqual, testBody)

			post, err := getPost(ts, created.ID, testUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			So(post.ID, ShouldEqual, created.ID)
			So(post.Body, ShouldEqual, testBody)
		})

		Convey("Should be able to create a post with emote", func() {
			testBody := "TestIntegration_Post_Emote :) " + timestamp()
			created, err := createPostV2(ts, testUserID, &createPostParams{UserID: testChannelID, Body: testBody})
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)

			posts, err := getPostsByIDs(ts, []string{created.ID})
			So(err, ShouldBeNil)
			So(posts.Items[0], ShouldNotBeNil)
			So(posts.Items[0].Emotes, ShouldNotBeNil)
			So(len(posts.Items[0].Emotes), ShouldEqual, 1)
		})

		Convey("Should not be able to create a post as a different user in V2", func() {
			testBody := "TestIntegration_Post " + timestamp()
			created, err := createPostV2(ts, testUserID2, &createPostParams{UserID: testChannelID, Body: testBody})
			So(errorCode(err), ShouldEqual, http.StatusForbidden)
			So(created, ShouldBeNil)
		})

		Convey("Should be able to bulk get multiple posts by id", func() {
			testBody := "TestIntegration_Post " + timestamp()
			created1, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(created1, ShouldNotBeNil)

			created2, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(created2, ShouldNotBeNil)

			posts, err := getPostsByIDs(ts, []string{created1.ID, created2.ID})
			So(err, ShouldBeNil)
			So(posts, ShouldNotBeNil)
			So(posts.Items, ShouldHaveLength, 2)

			postMap := make(map[string]*v2.Post)
			for _, post := range posts.Items {
				postMap[post.ID] = post
			}

			So(postMap[created1.ID], ShouldNotBeNil)
			So(postMap[created1.ID].Body, ShouldEqual, created1.Body)
			So(postMap[created1.ID].CreatedAt, ShouldResemble, created1.CreatedAt)

			So(postMap[created2.ID], ShouldNotBeNil)
			So(postMap[created2.ID].Body, ShouldEqual, created2.Body)
			So(postMap[created2.ID].CreatedAt, ShouldResemble, created2.CreatedAt)
		})

		Convey("Should be able to get multiple posts' delete permissions", func() {
			testBody := "TestIntegration_Post " + timestamp()
			created0, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(created0, ShouldNotBeNil)

			created1, err := createPost(ts, testChannelID2, testBody, testUserID2)
			So(err, ShouldBeNil)
			So(created1, ShouldNotBeNil)

			permissions, err := getPostsPermissionsByIDs(ts, testUserID, []string{created0.ID, created1.ID})
			So(err, ShouldBeNil)
			So(permissions, ShouldNotBeNil)
			So(permissions.Items, ShouldHaveLength, 2)
			if permissions.Items[0].PostID != created0.ID {
				tmp := permissions.Items[0]
				permissions.Items[0] = permissions.Items[1]
				permissions.Items[1] = tmp
			}
			So(permissions.Items[0].PostID, ShouldEqual, created0.ID)
			So(permissions.Items[1].PostID, ShouldEqual, created1.ID)
			So(permissions.Items[0].CanDelete, ShouldBeTrue)
			So(permissions.Items[1].CanDelete, ShouldBeFalse)
		})

		Convey("Should be able to bulk get a post with emotes and embeds", func() {
			testBody := "TestIntegration_Post Kappa " + timestamp()
			testUrl := "https://clips.twitch.tv/SecretiveMildOkapiYouDontSay"
			embedURLs := []string{testUrl}
			params := &createPostParams{
				UserID:    testChannelID,
				Body:      testBody,
				EmbedURLs: &embedURLs,
			}
			created, err := createPostWithParams(ts, testUserID, params)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.Embeds, ShouldNotBeNil)
			So(created.Emotes, ShouldNotBeNil)

			posts, err := getPostsByIDs(ts, []string{created.ID})
			So(err, ShouldBeNil)
			So(posts, ShouldNotBeNil)
			So(posts.Items, ShouldHaveLength, 1)

			So(posts.Items[0], ShouldNotBeNil)
			So(posts.Items[0].Emotes, ShouldHaveLength, 1)
			So(posts.Items[0].Emotes[0], ShouldResemble, &v2.Emote{
				Set:   0,
				ID:    25,
				Start: 21,
				End:   25,
			})
			So(posts.Items[0].EmbedEntities, ShouldHaveLength, 1)
			So((*posts.Items[0].EmbedEntities)[0].Namespace(), ShouldEqual, entity.NamespaceClip)
			So((*posts.Items[0].EmbedEntities)[0].ID(), ShouldEqual, "SecretiveMildOkapiYouDontSay")
		})

		Convey("Should not be able to bulk get a deleted post", func() {
			testBody := "TestIntegration_Post " + timestamp()
			created, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			deleted, err := deletePost(ts, created.ID, testUserID)
			So(err, ShouldBeNil)
			So(deleted, ShouldNotBeNil)
			So(deleted.Deleted, ShouldBeTrue)

			posts, err := getPostsByIDs(ts, []string{created.ID})
			So(err, ShouldBeNil)
			So(posts, ShouldNotBeNil)
			So(posts.Items, ShouldHaveLength, 0)
		})

		// testing deleting post using the new API
		Convey("Should not be able to get a deleted post in V2", func() {
			testBody := "TestIntegration_Post " + timestamp()
			created, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)

			deleted, err := deletePostV2(ts, created.ID, testUserID)
			So(err, ShouldBeNil)
			So(deleted, ShouldNotBeNil)
			So(deleted.Deleted, ShouldBeTrue)

			_, err = getPost(ts, created.ID, testUserID)
			So(errorCode(err), ShouldEqual, http.StatusNotFound)
		})

		Convey("Should be able to parse emotes in a post body", func() {
			testBody := "TestIntegration_Post Kappa PogChamp " + timestamp()
			created, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)

			post, err := getPost(ts, created.ID, testUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			So(post, ShouldResemble, created)
			So(post.Emotes, ShouldResemble, []*api.Emote{
				{Set: 0, ID: 25, Start: 21, End: 25},
				{Set: 0, ID: 88, Start: 27, End: 34},
			})
		})

		Convey("Should be able to parse mako emotes in a post body", func() {
			testBody := "TestIntegration_Post SeemsMinton Kappa" + timestamp()
			created, err := createPost(ts, testMakoChannelID, testBody, testMakoUserID)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)

			post, err := getPost(ts, created.ID, testMakoUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			So(post, ShouldResemble, created)
			So(post.Emotes, ShouldResemble, []*api.Emote{
				{Set: 21889, ID: 160148, Start: 21, End: 31},
			})
		})

		Convey("Should be able to parse emotes in a post body with Unicode (CF-180)", func() {
			testBody := "TestIntegration_Post 'ä' Kappa " + timestamp()
			created, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)

			post, err := getPost(ts, created.ID, testUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			So(post, ShouldResemble, created)
			So(post.Emotes, ShouldResemble, []*api.Emote{{Set: 0, ID: 25, Start: 25, End: 29}})
		})

		Convey("Should parse no emotes in a post body with no emotes", func() {
			testBody := "TestIntegration_Post " + timestamp()
			created, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)

			post, err := getPost(ts, created.ID, testUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			So(post, ShouldResemble, created)
			So(post.Emotes, ShouldBeEmpty)
		})

		Convey("Should be able to report a post", func() {
			testBody := "TestIntegration_Post " + timestamp()
			created, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)

			err = reportPost(ts, created.ID, testReporterUserID, "other")
			So(err, ShouldBeNil)
		})

		Convey("Should have the correct permissions for your own posts", func() {
			permissions, err := getPermissions(ts, testChannelID, testUserID)
			So(err, ShouldBeNil)
			So(permissions.CanDelete, ShouldBeTrue)
			So(permissions.CanModerate, ShouldBeTrue)
			So(permissions.CanShare, ShouldBeFalse)
		})

		Convey("Should have the correct permissions for other people's posts", func() {
			permissions, err := getPermissions(ts, testChannelID, testUserID2)
			So(err, ShouldBeNil)
			So(permissions.CanDelete, ShouldBeFalse)
			So(permissions.CanModerate, ShouldBeFalse)
			So(permissions.CanShare, ShouldBeTrue)
		})

		Convey("Should have the correct permissions if you aren't logged in", func() {
			permissions, err := getPermissions(ts, testChannelID, "")
			So(err, ShouldBeNil)
			So(permissions.CanDelete, ShouldBeFalse)
			So(permissions.CanModerate, ShouldBeFalse)
			So(permissions.CanShare, ShouldBeFalse)
		})

		Convey("Should not be able to share synthetic posts", func() {
			postID := testVodID
			post, err := getPost(ts, postID, testUserID)
			So(err, ShouldBeNil)
			So(post.Permissions.CanShare, ShouldBeFalse)
			So(post.Permissions.CanReply, ShouldBeFalse)
			So(post.Permissions.CanDelete, ShouldBeFalse)
		})

		Convey("Reporting a post that doesn't exist should return a 404", func() {
			// Using a post ID that won't exist.
			postID := timestamp()

			err := reportPost(ts, postID, testReporterUserID, "other")
			So(err, ShouldNotBeNil)
			So(errorCode(err), ShouldEqual, http.StatusNotFound)
		})

		Convey("Post loading should work for vods", func() {
			postID := testVodID
			post, err := getPost(ts, postID, testUserID)
			So(err, ShouldBeNil)
			So(post.ID, ShouldEqual, postID)
			So(post.UserID, ShouldEqual, testVodAuthorID)
		})

		Convey("Invalid vods should 404", func() {
			postID := "vod:76846095123423423423"
			_, err := getPost(ts, postID, testUserID)
			So(errorCode(err), ShouldEqual, http.StatusNotFound)
		})

		Convey("Post loading should work for clips", func() {
			postID := "clip:EvilPolishedCurrySoonerLater"
			post, err := getPost(ts, postID, testUserID)
			So(err, ShouldBeNil)
			So(post.ID, ShouldEqual, postID)
			So(post.UserID, ShouldEqual, "76846095")
		})
	})

	ts.onFinish(time.Second * 3)
}

func TestIntegration_Embeds(t *testing.T) {
	t.Parallel()
	ts := startServer(t, injectables{})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func(c C) {
		So(ts.Setup(), ShouldBeNil)
		c.Reset(ts.cancelFunc)

		Convey("Post with Twitch Clip URL should return with populated embed properties", func() {
			testUrl := "https://clips.twitch.tv/SecretiveMildOkapiYouDontSay"
			testBody := "TestIntegration_Post " + testUrl + " " + timestamp()
			post, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			So(post.Embeds, ShouldNotBeNil)
			So(post.Embeds, ShouldHaveLength, 1)
			So(post.Embeds[0].RequestURL, ShouldEqual, testUrl)
			So(post.EmbedURLs, ShouldNotBeNil)
			So(*post.EmbedURLs, ShouldHaveLength, 1)
			So((*post.EmbedURLs)[0], ShouldEqual, testUrl)

			So(post.EmbedEntities, ShouldNotBeNil)
			So(*post.EmbedEntities, ShouldHaveLength, 1)
			So((*post.EmbedEntities)[0].Namespace(), ShouldEqual, entity.NamespaceClip)
			So((*post.EmbedEntities)[0].ID(), ShouldEqual, "SecretiveMildOkapiYouDontSay")
		})

		Convey("Post with Twitch VOD URL should return with populated embed properties", func() {
			testUrl := "https://www.twitch.tv/qa_flv_vods_transcoded/v/43848179"
			testBody := "TestIntegration_Post " + testUrl + " " + timestamp()
			post, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			So(post.Embeds, ShouldNotBeNil)
			So(post.Embeds, ShouldHaveLength, 1)
			So(post.Embeds[0].RequestURL, ShouldEqual, testUrl)
			So(post.EmbedURLs, ShouldNotBeNil)
			So(*post.EmbedURLs, ShouldHaveLength, 1)
			So((*post.EmbedURLs)[0], ShouldEqual, testUrl)

			So(post.EmbedEntities, ShouldNotBeNil)
			So(*post.EmbedEntities, ShouldHaveLength, 1)
			So((*post.EmbedEntities)[0].Namespace(), ShouldEqual, entity.NamespaceVod)
			So((*post.EmbedEntities)[0].ID(), ShouldEqual, "43848179")
		})

		Convey("EmbedURLs field with Twitch VOD URL should return with populated embed properties", func() {
			testUrl := "https://www.twitch.tv/qa_flv_vods_transcoded/v/43848179"
			embedURLs := []string{testUrl}
			testBody := "TestIntegration_Post " + timestamp()

			params := &createPostParams{
				UserID:    testChannelID,
				Body:      testBody,
				EmbedURLs: &embedURLs,
			}
			post, err := createPostWithParams(ts, testUserID, params)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			So(post.Embeds, ShouldNotBeNil)
			So(post.Embeds, ShouldHaveLength, 1)
			So(post.Embeds[0].RequestURL, ShouldEqual, testUrl)
			So(post.EmbedURLs, ShouldNotBeNil)
			So(*post.EmbedURLs, ShouldHaveLength, 1)
			So((*post.EmbedURLs)[0], ShouldEqual, testUrl)
		})

		Convey("EmbedURLs field with multiple URLs should return with populated embed properties in the same order", func() {
			testUrl1 := "https://www.twitch.tv/qa_flv_vods_transcoded/v/43848179"
			testUrl2 := "https://clips.twitch.tv/SecretiveMildOkapiYouDontSay"
			testUrl3 := "https://www.youtube.com/watch?v=dQw4w9WgXcQ"
			embedURLs := []string{testUrl1, testUrl2, testUrl3}
			testBody := "TestIntegration_Post " + timestamp()

			params := &createPostParams{
				UserID:    testChannelID,
				Body:      testBody,
				EmbedURLs: &embedURLs,
			}
			post, err := createPostWithParams(ts, testUserID, params)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			So(post.Embeds, ShouldNotBeNil)
			So(post.Embeds, ShouldHaveLength, len(embedURLs))
			So(post.EmbedURLs, ShouldNotBeNil)
			So(*post.EmbedURLs, ShouldHaveLength, len(embedURLs))
			// Check that embed order is preserved
			for index, embedURL := range embedURLs {
				So(embedURL, ShouldEqual, post.Embeds[index].RequestURL)
				So(embedURL, ShouldEqual, (*post.EmbedURLs)[index])
			}
		})

		Convey("Post with Test URL should return no embeds", func() {
			testUrl := "https://test.com"
			testBody := "TestIntegration_Post " + testUrl + " " + timestamp()
			post, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			So(post.Embeds, ShouldBeEmpty)
		})

		Convey("Post with invalid embed url returns no embeds", func() {
			invalidVodURL := "https://www.twitch.tv/videos/168913010"
			testBody := strings.Join([]string{"TestIntegration_Post", invalidVodURL, timestamp()}, "  ")
			post, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			So(post.Embeds, ShouldBeEmpty)
		})

		Convey("Embed endpoint returns a populated embed object", func() {
			testUrl := "https://www.twitch.tv/qa_flv_vods_transcoded/v/43848179"
			embed, err := getEmbed(ts, testUrl, true)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeEmpty)
			So(embed.Type, ShouldEqual, "video")
		})

		Convey("Post with empty embedURLs should return no embeds", func() {
			testBody := "TestIntegration_Post " + timestamp()
			embedURLs := []string{}

			params := &createPostParams{
				UserID:    testChannelID,
				Body:      testBody,
				EmbedURLs: &embedURLs,
			}
			created, err := createPostWithParams(ts, testUserID, params)
			So(err, ShouldBeNil)
			So(created.Embeds, ShouldBeEmpty)
			So(created.EmbedURLs, ShouldNotBeNil)
			So(*created.EmbedURLs, ShouldBeEmpty)
		})
	})

	ts.onFinish(time.Second * 3)
}

func TestIntegration_Embed_Timeout(t *testing.T) {
	ts := startServer(t, injectables{}, map[string][]byte{
		"feeds-edge.embed_timeout": []byte("1ms"),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With 1ms embed timeout "+ts.host, t, func(c C) {
		So(ts.Setup(), ShouldBeNil)
		c.Reset(ts.cancelFunc)

		Convey("Post with Test URL in body should return embed", func() {
			testUrl := "https://www.youtube.com/watch?v=dQw4w9WgXcQ"
			testBody := "TestIntegration_Post " + testUrl + " " + timestamp()
			created, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.Embeds, ShouldHaveLength, 1)
			So(created.Embeds[0].RequestURL, ShouldEqual, testUrl)
			So(created.EmbedURLs, ShouldNotBeNil)
			So(*created.EmbedURLs, ShouldHaveLength, 1)
			So((*created.EmbedURLs)[0], ShouldEqual, testUrl)
		})
	})

	ts.onFinish(time.Second * 3)
}

func TestIntegration_Post_Cooldown(t *testing.T) {
	t.Parallel()
	cooldownTime := time.Second * 2
	ts := startServer(t, injectables{}, map[string][]byte{
		"feeds-edge.create_post_cooldown":       []byte(cooldownTime.String()),
		"feeds-edge.cooldown_exempt_test_users": []byte("," + testUserID + ","),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func(c C) {
		So(ts.Setup(), ShouldBeNil)
		c.Reset(ts.cancelFunc)

		Convey("Given a post created by a user 1", func() {
			userID1 := newUserID()
			channelID1 := userID1

			_, err := createPost(ts, channelID1, "arbitrary body", userID1)
			So(err, ShouldBeNil)

			Convey("User 1 should not be able to create another post during cooldown", func() {
				_, err := createPost(ts, channelID1, "arbitrary body", userID1)
				So(err, ShouldNotBeNil)
				So(errorCode(err), ShouldEqual, http.StatusTooManyRequests)
			})

			Convey("User 1 should be able to post again after cooldown", func() {
				time.Sleep(cooldownTime)

				_, err := createPost(ts, channelID1, "arbitrary body", userID1)
				So(err, ShouldBeNil)
			})

			Convey("User 2 should be able to post during User 1's cooldown", func() {
				userID2 := newUserID()
				channelID2 := userID2

				_, err := createPost(ts, channelID2, "arbitrary body", userID2)
				So(err, ShouldBeNil)
			})
		})

		Convey("Given a post created by a cooldown-exempt user", func() {
			_, err := createPost(ts, testChannelID, "arbitrary body", testUserID)
			So(err, ShouldBeNil)

			Convey("Exempt user should be able to create another post right away", func() {
				_, err := createPost(ts, testChannelID, "arbitrary body", testUserID)
				So(err, ShouldBeNil)
			})
		})
	})

	ts.onFinish(time.Second * 3)
}

func TestIntegrationShares(t *testing.T) {
	t.Parallel()
	ts := startServer(t, injectables{})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func(c C) {
		So(ts.Setup(), ShouldBeNil)
		c.Reset(ts.cancelFunc)

		Convey("Should create a post then share it", func() {
			testBody := "TestIntegration_Share " + timestamp()
			createdPost, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(createdPost, ShouldNotBeNil)

			postEnt := entity.New("post", createdPost.ID)
			share, err := createShare(ts, testUserID2, postEnt)
			So(err, ShouldBeNil)
			So(share, ShouldNotBeNil)

			thenCheckItWasRemoved := func() {
				Convey("Removing again should 404", func() {
					share2, err := deleteShare(ts, share.ID, testUserID2)
					So(share2, ShouldBeNil)
					So(errorCode(err), ShouldEqual, http.StatusNotFound)
				})

				Convey("fetches should fail", func() {
					share3, err := getShare(ts, share.ID, testUserID2)
					So(errorCode(err), ShouldEqual, http.StatusNotFound)
					So(share3, ShouldBeNil)
				})
				Convey("unshare by target should show it already removed", func() {
					res, err := unshareByTarget(ts, testUserID2, testUserID2, postEnt)
					So(err, ShouldBeNil)
					So(res.Items, ShouldBeEmpty)
				})
				Convey("summary should be empty", func() {
					summary, err := getSharesSummaries(ts, testUserID2, postEnt)
					So(err, ShouldBeNil)
					So(summary.Items[0].ShareCount, ShouldEqual, 0)
					So(summary.Items[0].UserIDs, ShouldBeEmpty)
				})
				Convey("post should have no shares", func() {
					newPost, err := getPost(ts, createdPost.ID, testUserID2)
					So(err, ShouldBeNil)
					So(newPost.ShareSummary.ShareCount, ShouldEqual, 0)
					So(newPost.ShareSummary.UserIDs, ShouldBeEmpty)
				})
			}

			Convey("Then remove it", func() {
				share2, err := deleteShare(ts, share.ID, testUserID2)
				So(err, ShouldBeNil)
				So(share.ID, ShouldEqual, share2.ID)
				thenCheckItWasRemoved()
			})

			Convey("Removing someone else's share should check permissions", func() {
				_, err := deleteShare(ts, share.ID, testUserID)
				So(errorCode(err), ShouldEqual, http.StatusForbidden)

				_, err = unshareByTarget(ts, testUserID2, testUserID, postEnt)
				So(errorCode(err), ShouldEqual, http.StatusForbidden)
			})

			Convey("Then remove it by unshare by target", func() {
				res, err := unshareByTarget(ts, testUserID2, testUserID2, postEnt)
				So(err, ShouldBeNil)
				So(res.Items, ShouldHaveLength, 1)
				So(res.Items[0].ID, ShouldEqual, share.ID)
				thenCheckItWasRemoved()
			})

			Convey("and get the share back", func() {
				newPost, err := getPost(ts, createdPost.ID, testUserID2)
				So(err, ShouldBeNil)
				So(newPost.ShareSummary.ShareCount, ShouldEqual, 1)
				So(newPost.ShareSummary.UserIDs, ShouldContain, testUserID2)

				share4, err := getShare(ts, share.ID, testUserID2)
				So(err, ShouldBeNil)
				So(share4.TargetEntity.Encode(), ShouldEqual, "post:"+createdPost.ID)
				So(share4.ID, ShouldEqual, share.ID)

				summary, err := getSharesSummaries(ts, testUserID2, postEnt)
				So(err, ShouldBeNil)
				So(summary.Items, ShouldHaveLength, 1)
				So(summary.Items[0].ShareCount, ShouldEqual, 1)
				So(summary.Items[0].UserIDs, ShouldResemble, []string{testUserID2})
			})
		})

		Convey("test create and delete in V2", func() {
			testBody := "TestIntegration_Share V2" + timestamp()
			createdPost, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(createdPost, ShouldNotBeNil)

			postEnt := entity.New("post", createdPost.ID)
			share, err := createShareV2(ts, testUserID2, postEnt)
			So(err, ShouldBeNil)
			So(share, ShouldNotBeNil)

			share2, err := deleteShareV2(ts, share.ID, testUserID2)
			So(err, ShouldBeNil)
			So(share.ID, ShouldEqual, share2.ID)

			share2, err = deleteShareV2(ts, share.ID, testUserID2)
			So(share2, ShouldBeNil)
			So(errorCode(err), ShouldEqual, http.StatusNotFound)
		})

		Convey("Should not be able to share your own posts", func() {
			testBody := "TestIntegration_Share " + timestamp()
			createdPost, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(createdPost, ShouldNotBeNil)

			postEnt := entity.New("post", createdPost.ID)
			share, err := createShare(ts, testUserID, postEnt)
			So(share, ShouldBeNil)
			So(err, ShouldNotBeNil)
			So(errorCode(err), ShouldEqual, http.StatusForbidden)
		})

		Convey("Should not be able to share a post that does not exist", func() {
			postEnt := entity.New("post", "aldsfjhlasjkdfhlajshdflkj")
			share, err := createShare(ts, testUserID, postEnt)
			So(share, ShouldBeNil)
			So(err, ShouldNotBeNil)
			So(errorCode(err), ShouldEqual, http.StatusNotFound)
		})

		Convey("Should be able to bulk get multiple shares", func() {
			testBody := "TestIntegration_Share " + timestamp()
			post, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			postEnt := entity.New("post", post.ID)

			share1, err := createShare(ts, testUserID2, postEnt)
			So(err, ShouldBeNil)
			So(share1, ShouldNotBeNil)

			share2, err := createShare(ts, testStrictModerationID, postEnt)
			So(err, ShouldBeNil)
			So(share2, ShouldNotBeNil)

			shares, err := getSharesByIDs(ts, []string{share1.ID, share2.ID})
			So(err, ShouldBeNil)
			So(shares, ShouldNotBeNil)
			So(shares.Items, ShouldHaveLength, 2)

			shareMap := make(map[string]*v2.Share)

			for _, share := range shares.Items {
				shareMap[share.ID] = share
			}

			So(shareMap[share1.ID], ShouldNotBeNil)
			So(shareMap[share1.ID].UserID, ShouldEqual, share1.UserID)
			So(shareMap[share1.ID].TargetEntity, ShouldResemble, share1.TargetEntity)
			So(shareMap[share1.ID].CreatedAt, ShouldResemble, share1.CreatedAt)

			So(shareMap[share2.ID], ShouldNotBeNil)
			So(shareMap[share2.ID].UserID, ShouldEqual, share2.UserID)
			So(shareMap[share2.ID].TargetEntity, ShouldResemble, share2.TargetEntity)
			So(shareMap[share2.ID].CreatedAt, ShouldResemble, share2.CreatedAt)
		})

		Convey("Should not be able to bulk get a deleted share", func() {
			testBody := "TestIntegration_Share " + timestamp()
			post, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			postEnt := entity.New("post", post.ID)

			share, err := createShare(ts, testUserID2, postEnt)
			So(err, ShouldBeNil)
			So(share, ShouldNotBeNil)

			_, err = deleteShare(ts, share.ID, testUserID2)
			So(err, ShouldBeNil)

			shares, err := getSharesByIDs(ts, []string{share.ID})
			So(err, ShouldBeNil)
			So(shares, ShouldNotBeNil)
			So(shares.Items, ShouldHaveLength, 0)
		})
	})

	ts.onFinish(time.Second * 3)
}

func verifyReactionSummaries(actual interface{}, expected interface{}) (*v2.ReactionSummaries, string, string) {
	rs, ok := actual.(*v2.ReactionSummaries)
	if !ok {
		return nil, "", "actual should be of type *v2.ReactionSummaries, not " + reflect.TypeOf(actual).String()
	}

	wantID, ok := expected.(string)
	if !ok {
		return nil, "", "expected: is not of type string"
	}
	return rs, wantID, ""
}

func containsEmoteID(ts *testSetup, ent string, userID string, emoteID string) {
	defer func() {
		if p := recover(); p != nil {
			Println("Failure at ", stack.Caller(7).String())
			panic(p)
		}
	}()

	reactions, err := getReactionsByEntities(ts, []string{ent}, userID)
	So(err, ShouldBeNil)
	So(reactions, ShouldNotBeNil)
	So(reactions.Items, ShouldHaveLength, 1)
	So(reactions.Items[0], ShouldContainEmoteID, emoteID)
}

func doesNotContainsEmoteID(ts *testSetup, ent string, userID string, emoteID string) {
	reactions, err := getReactionsByEntities(ts, []string{ent}, userID)
	So(err, ShouldBeNil)
	So(reactions, ShouldNotBeNil)
	So(reactions.Items, ShouldHaveLength, 1)
	So(reactions.Items[0], ShouldNotContainEmoteID, emoteID)
}

func ShouldContainEmoteID(actual interface{}, expected ...interface{}) string {
	if len(expected) != 1 {
		return "expected should contain only one element"
	}

	rs, wantID, err := verifyReactionSummaries(actual, expected[0])
	if err != "" {
		return err
	}

	for _, s := range rs.Summaries {
		if s.EmoteID == wantID {
			return ""
		}
	}
	return fmt.Sprintf("actual ReactionSummaries: %#v, does not contain emoteID: %s", rs, wantID)
}

func ShouldNotContainEmoteID(actual interface{}, expected ...interface{}) string {
	if len(expected) != 1 {
		return "expected should contain only one element"
	}

	rs, wantID, err := verifyReactionSummaries(actual, expected[0])
	if err != "" {
		return err
	}

	for _, s := range rs.Summaries {
		if s.EmoteID == wantID {
			return fmt.Sprintf("actual ReactionSummaries: %#v, contains emoteID: %s", rs, wantID)
		}
	}
	return ""
}

func TestIntegration_Reaction(t *testing.T) {
	t.Parallel()
	//ts := startServer(t, injectables{})
	ts := startServer(t, injectables{}, map[string][]byte{
		"feeds-edge.create_comment_cooldown": []byte(time.Nanosecond.String()),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func(c C) {
		So(ts.Setup(), ShouldBeNil)
		c.Reset(ts.cancelFunc)

		Convey("Should be able to react to a post", func() {
			testPostBody := "TestIntegration_Reaction Post " + timestamp()
			createdPost, err := createPost(ts, testChannelID, testPostBody, testUserID)
			So(err, ShouldBeNil)
			So(createdPost, ShouldNotBeNil)

			Convey("With v1 createReaction", func() {
				reaction1, err := createReaction(ts, postEntity(createdPost.ID), testEmoteKappa, testUserID)
				So(err, ShouldBeNil)
				So(reaction1, ShouldNotBeNil)
				So(reaction1.EmoteIDs, ShouldContain, testEmoteKappa)

				reaction2, err := createReaction(ts, postEntity(createdPost.ID), testEmoteRegex1, testUserID)
				So(err, ShouldBeNil)
				So(reaction2, ShouldNotBeNil)
				So(reaction2.EmoteIDs, ShouldContain, testEmoteRegex1)

				post, err := getPost(ts, createdPost.ID, testUserID)
				So(err, ShouldBeNil)
				So(post.Reactions, ShouldContainKey, testEmoteKappa)
				So(post.Reactions, ShouldContainKey, testEmoteRegex1)
				So(post.Reactions[testEmoteKappa].UserIDs, ShouldContain, testUserID)
			})

			Convey("With v2 createReaction", func() {
				_, err := createReactionV2(ts, postEntity(createdPost.ID), testEmoteKappa, testUserID)
				So(err, ShouldBeNil)
				containsEmoteID(ts, postEntity(createdPost.ID), testUserID, testEmoteKappa)

				_, err = createReactionV2(ts, postEntity(createdPost.ID), testEmoteRegex1, testUserID)
				So(err, ShouldBeNil)
				containsEmoteID(ts, postEntity(createdPost.ID), testUserID, testEmoteRegex1)

				post, err := getPost(ts, createdPost.ID, testUserID)
				So(err, ShouldBeNil)
				So(post.Reactions, ShouldContainKey, testEmoteKappa)
				So(post.Reactions, ShouldContainKey, testEmoteRegex1)
				So(post.Reactions[testEmoteKappa].UserIDs, ShouldContain, testUserID)
			})
		})

		Convey("With a post", func() {
			createdPost, err := createPost(ts, testChannelID, "arbitrary body", testUserID)
			So(err, ShouldBeNil)
			pEntity := postEntity(createdPost.ID)

			Convey("That multiple users reacted on", func() {
				testEmoteID5 := "5"

				Convey("Using v1 and v2 Reactions", func() {
					_, err = createReaction(ts, pEntity, testEmoteKappa, testUserID)
					So(err, ShouldBeNil)

					_, err := createReactionV2(ts, pEntity, testEmoteID5, testUserID2)
					So(err, ShouldBeNil)
					containsEmoteID(ts, pEntity, testUserID2, testEmoteID5)

					_, err = createReaction(ts, pEntity, testEmoteKappa, testUserID2)
					So(err, ShouldBeNil)

					Convey("Removing one the reactions should succeed and decrement the emote count", func() {
						Convey("Using v1 and v2 Reactions", func() {
							reactions2, err := deleteReaction(ts, pEntity, testEmoteKappa, testUserID)
							So(err, ShouldBeNil)
							So(reactions2.EmoteIDs, ShouldNotContain, testEmoteKappa)
							So(reactions2.UserID, ShouldEqual, testUserID)

							_, err = deleteReactionV2(ts, pEntity, testEmoteID5, testUserID2)
							So(err, ShouldBeNil)
							doesNotContainsEmoteID(ts, pEntity, testUserID2, testEmoteID5)

							post, err := getPost(ts, createdPost.ID, testUserID)
							So(err, ShouldBeNil)
							So(post.Reactions, ShouldNotContainKey, testEmoteID5)
							So(post.Reactions, ShouldContainKey, testEmoteKappa)
							So(post.Reactions[testEmoteKappa].Count, ShouldEqual, 1)
						})
					})
				})

				// Disable test until feeds/clients can be revendored to 7118d38.
				SkipConvey("Removing an emote that the user didn't post should fail", func() {
					_, err := deleteReaction(ts, pEntity, testEmotePogChamp, testUserID)
					So(err, ShouldNotBeNil)
					So(errorCode(err), ShouldEqual, http.StatusNotFound)
				})
			})
		})

		Convey("Should not be able to react to a post with a special emote, but should be able to if it was already added", func() {
			testPostBody := "TestIntegration_Reaction Post " + timestamp()
			Convey("With v1 Reaction", func() {
				createdPost, err := createPost(ts, testChannelIDWithEmoteSet, testPostBody, testChannelIDWithEmoteSet)
				So(err, ShouldBeNil)
				So(createdPost, ShouldNotBeNil)

				failedReaction, err := createReaction(ts, postEntity(createdPost.ID), testEmoteMangoUSA, testUserID)
				So(errorCode(err), ShouldEqual, http.StatusBadRequest)
				So(failedReaction, ShouldBeNil)

				reactionByChannel, err := createReaction(ts, postEntity(createdPost.ID), testEmoteMangoUSA, testChannelIDWithEmoteSet)
				So(err, ShouldBeNil)
				So(reactionByChannel, ShouldNotBeNil)
				So(reactionByChannel.EmoteIDs, ShouldContain, testEmoteMangoUSA)

				// do the same exact thing as for failedReaction, but this time it should succeed since the reaction is already there
				reaction, err := createReaction(ts, postEntity(createdPost.ID), testEmoteMangoUSA, testUserID)
				So(err, ShouldBeNil)
				So(reaction, ShouldNotBeNil)
				So(reaction.EmoteIDs, ShouldContain, testEmoteMangoUSA)

				post, err := getPost(ts, createdPost.ID, testUserID)
				So(err, ShouldBeNil)
				So(post.Reactions[testEmoteMangoUSA].Count, ShouldEqual, 2)
			})

			Convey("With v2 Reaction", func() {
				createdPost, err := createPost(ts, testChannelIDWithEmoteSet, testPostBody, testChannelIDWithEmoteSet)
				So(err, ShouldBeNil)
				So(createdPost, ShouldNotBeNil)

				failedReaction, err := createReactionV2(ts, postEntity(createdPost.ID), testEmoteMangoUSA, testUserID)
				So(errorCode(err), ShouldEqual, http.StatusBadRequest)
				So(failedReaction, ShouldBeNil)

				_, err = createReactionV2(ts, postEntity(createdPost.ID), testEmoteMangoUSA, testChannelIDWithEmoteSet)
				So(err, ShouldBeNil)
				containsEmoteID(ts, postEntity(createdPost.ID), testChannelIDWithEmoteSet, testEmoteMangoUSA)

				// do the same exact thing as for failedReaction, but this time it should succeed since the reaction is already there
				_, err = createReactionV2(ts, postEntity(createdPost.ID), testEmoteMangoUSA, testUserID)
				So(err, ShouldBeNil)
				containsEmoteID(ts, postEntity(createdPost.ID), testUserID, testEmoteMangoUSA)

				post, err := getPost(ts, createdPost.ID, testUserID)
				So(err, ShouldBeNil)
				So(post.Reactions[testEmoteMangoUSA].Count, ShouldEqual, 2)
			})
		})

		Convey("Should not be able to react to a post with an invalid emote", func() {
			testPostBody := "TestIntegration_Reaction Post " + timestamp()
			createdPost, err := createPost(ts, testChannelID, testPostBody, testUserID)
			So(err, ShouldBeNil)
			So(createdPost, ShouldNotBeNil)

			Convey("With v1 Reaction", func() {
				reaction, err := createReaction(ts, postEntity(createdPost.ID), testEmoteIDNonExistent, testUserID)
				So(errorCode(err), ShouldEqual, http.StatusBadRequest)
				So(reaction, ShouldBeNil)

				post, err := getPost(ts, createdPost.ID, testUserID)
				So(err, ShouldBeNil)
				So(post.Reactions, ShouldNotContainKey, testEmoteIDNonExistent)
			})

			Convey("With v2 Reaction", func() {
				reaction, err := createReactionV2(ts, postEntity(createdPost.ID), testEmoteIDNonExistent, testUserID)
				So(errorCode(err), ShouldEqual, http.StatusBadRequest)
				So(reaction, ShouldBeNil)

				post, err := getPost(ts, createdPost.ID, testUserID)
				So(err, ShouldBeNil)
				So(post.Reactions, ShouldNotContainKey, testEmoteIDNonExistent)
			})
		})

		Convey("Should be able to create, get, and delete reactions on a foreign entity", func() {
			e := entity.New("test", timestamp())

			created, err := createReaction(ts, e.Encode(), testEmoteKappa, testUserID)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.EmoteIDs, ShouldContain, testEmoteKappa)

			reactions, err := getReactionsByParents(ts, []string{e.Encode()}, testUserID)
			So(err, ShouldBeNil)
			So(reactions, ShouldNotBeNil)
			So(reactions.Reactions, ShouldHaveLength, 1)
			r := reactions.Reactions[0]
			So(r.ParentEntity, ShouldResemble, e)
			So(r.Reactions, ShouldContainKey, testEmoteKappa)
			So(r.Reactions[testEmoteKappa].Count, ShouldEqual, 1)
			So(r.Reactions[testEmoteKappa].UserIDs, ShouldContain, testUserID)

			deleted, err := deleteReaction(ts, e.Encode(), testEmoteKappa, testUserID)
			So(err, ShouldBeNil)
			So(deleted, ShouldNotBeNil)

			Convey("With v2 endpoints", func() {
				_, err = createReactionV2(ts, e.Encode(), testEmoteKappa, testUserID2)
				So(err, ShouldBeNil)
				containsEmoteID(ts, e.Encode(), testUserID2, testEmoteKappa)

				reactions, err := getReactionsByParents(ts, []string{e.Encode()}, testUserID2)
				So(err, ShouldBeNil)
				So(reactions, ShouldNotBeNil)
				So(reactions.Reactions, ShouldHaveLength, 1)
				r := reactions.Reactions[0]
				So(r.ParentEntity, ShouldResemble, e)
				So(r.Reactions, ShouldContainKey, testEmoteKappa)
				So(r.Reactions[testEmoteKappa].Count, ShouldEqual, 1)
				So(r.Reactions[testEmoteKappa].UserIDs, ShouldContain, testUserID2)

				deleted, err := deleteReactionV2(ts, e.Encode(), testEmoteKappa, testUserID2)
				So(err, ShouldBeNil)
				So(deleted, ShouldNotBeNil)
			})
		})

		Convey("Fetching two non-existing foreign entities should return two empty objects", func() {
			e1 := entity.New("test1", timestamp())
			e2 := entity.New("test2", timestamp())

			reactions, err := getReactionsByParents(ts, []string{e1.Encode(), e2.Encode()}, testUserID)
			So(err, ShouldBeNil)
			So(reactions, ShouldNotBeNil)
			So(reactions.Reactions, ShouldHaveLength, 2)
			So(reactions.Reactions[0].Reactions, ShouldBeEmpty)
			So(reactions.Reactions[1].Reactions, ShouldBeEmpty)
		})

		Convey("Should be able to react to a clip regardless of clip id capitalization", func() {
			clipEntity := "clip:SparklingDepressedRamenPlanking"
			clipEntityLower := strings.ToLower(clipEntity)
			reaction1, err := createReaction(ts, clipEntity, testEmoteKappa, testUserID)
			So(err, ShouldBeNil)
			So(reaction1, ShouldNotBeNil)
			So(reaction1.EmoteIDs, ShouldContain, testEmoteKappa)

			reaction2, err := createReaction(ts, clipEntityLower, testEmotePogChamp, testUserID)
			So(err, ShouldBeNil)
			So(reaction2, ShouldNotBeNil)
			So(reaction2.EmoteIDs, ShouldContain, testEmoteKappa)
			So(reaction2.EmoteIDs, ShouldContain, testEmotePogChamp)

			reactions, err := getReactionsByParents(ts, []string{clipEntity, clipEntityLower}, testUserID)
			So(err, ShouldBeNil)
			So(reactions.Reactions[0].Reactions, ShouldResemble, reactions.Reactions[1].Reactions)

			reaction3, err := deleteReaction(ts, clipEntity, testEmotePogChamp, testUserID)
			So(err, ShouldBeNil)
			So(reaction3, ShouldNotBeNil)
			So(reaction3.EmoteIDs, ShouldNotContain, testEmotePogChamp)
			So(reaction3.EmoteIDs, ShouldContain, testEmoteKappa)

			reaction4, err := deleteReaction(ts, clipEntityLower, testEmoteKappa, testUserID)
			So(err, ShouldBeNil)
			So(reaction4, ShouldNotBeNil)
			So(reaction4.EmoteIDs, ShouldBeEmpty)
		})

		Convey("Should be able to bulk get reactions from multiple entity types", func() {
			clipEntity := "clip:AstuteStylishRabbitPeteZarollTie"
			reaction1, err := createReaction(ts, clipEntity, testEmoteKappa, testUserID)
			So(err, ShouldBeNil)
			So(reaction1, ShouldNotBeNil)
			So(reaction1.EmoteIDs, ShouldContain, testEmoteKappa)

			vodEntity := testVodID
			_, err = createReactionV2(ts, vodEntity, testEmoteKappa, testUserID)
			So(err, ShouldBeNil)
			containsEmoteID(ts, vodEntity, testUserID, testEmoteKappa)

			testPostBody := "TestIntegration_Reaction Post " + timestamp()
			createdPost, err := createPost(ts, testChannelID, testPostBody, testUserID)
			So(err, ShouldBeNil)
			So(createdPost, ShouldNotBeNil)
			postEntity := "post:" + createdPost.ID

			_, err = createReactionV2(ts, postEntity, testEmoteKappa, testUserID)
			So(err, ShouldBeNil)
			containsEmoteID(ts, postEntity, testUserID, testEmoteKappa)

			reactions, err := getReactionsByEntities(ts, []string{clipEntity, vodEntity, postEntity}, testUserID)
			So(err, ShouldBeNil)
			So(reactions, ShouldNotBeNil)
			So(reactions.Items, ShouldHaveLength, 3)

			reactionMap := make(map[string]*v2.ReactionSummaries)

			for _, reaction := range reactions.Items {
				reactionMap[reaction.ParentEntity.String()] = reaction
			}

			expectedReaction := &v2.ReactionSummary{
				EmoteID:     "25",
				EmoteName:   "Kappa",
				Count:       1,
				UserReacted: true,
			}

			So(reactionMap[clipEntity], ShouldNotBeNil)
			So(reactionMap[clipEntity].Summaries, ShouldHaveLength, 1)
			So(reactionMap[clipEntity].Summaries[0], ShouldResemble, expectedReaction)

			So(reactionMap[vodEntity], ShouldNotBeNil)
			So(reactionMap[vodEntity].Summaries, ShouldHaveLength, 1)
			So(reactionMap[vodEntity].Summaries[0], ShouldResemble, expectedReaction)

			So(reactionMap[postEntity], ShouldNotBeNil)
			So(reactionMap[postEntity].Summaries, ShouldHaveLength, 1)
			So(reactionMap[postEntity].Summaries[0], ShouldResemble, expectedReaction)

			reaction4, err := deleteReaction(ts, clipEntity, testEmoteKappa, testUserID)
			So(err, ShouldBeNil)
			So(reaction4, ShouldNotBeNil)
			So(reaction4.EmoteIDs, ShouldBeEmpty)

			reaction5, err := deleteReaction(ts, vodEntity, testEmoteKappa, testUserID)
			So(err, ShouldBeNil)
			So(reaction5, ShouldNotBeNil)
			So(reaction5.EmoteIDs, ShouldBeEmpty)
		})
	})

	ts.onFinish(time.Second * 3)
}

func TestIntegration_WithShineMock(t *testing.T) {
	t.Parallel()
	shineMock := &mocks.ShineClient{}
	ts := startServer(t, injectables{ShineClient: shineMock})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func(c C) {
		So(ts.Setup(), ShouldBeNil)
		c.Reset(ts.cancelFunc)
		resetMock(&shineMock.Mock)

		testEmbed := &shine.Embed{AuthorID: testUserID, TwitchType: "clip"}
		testClipID := "clip:LightEntertainingRedpandaPraiseIt"
		testClipURL := "https://clips.twitch.tv/abc"
		shineMock.On("GetEmbed", mock.Anything, testClipURL, mock.Anything).Return(testEmbed, nil)
		shineMock.On("GetEntitiesForURLs", mock.Anything, mock.Anything).Return(nil, nil)

		Convey("Post permissions should be false", func() {
			post, err := getPost(ts, testClipID, testUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			So(post.Permissions, ShouldNotBeNil)
			So(post.Permissions.CanDelete, ShouldBeFalse)
			So(post.Permissions.CanModerate, ShouldBeFalse)
			So(post.Permissions.CanReply, ShouldBeFalse)
			So(post.Permissions.CanShare, ShouldBeFalse)
		})

		// There was a strange bug (CF-706) where if a clip and a feed post shared a user_id, the
		// clip would have the same permissions as the post.  This tests to make sure that
		Convey("Feed permissions should be false", func() {
			testBody := "TestIntegration " + timestamp()
			post, err := createPost(ts, testUserID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)

			feed, err := getFeed(ts, "n:"+testUserID, testUserID, aws.String("post:"+post.ID+","+testClipID))
			So(err, ShouldBeNil)
			So(feed, ShouldNotBeNil)
			So(feed.Posts, ShouldHaveLength, 2)

			post1 := feed.Posts[0]
			So(post1, ShouldNotBeNil)
			So(post1.ID, ShouldEqual, post.ID)
			So(post1.Permissions, ShouldNotBeNil)
			So(post1.Permissions.CanDelete, ShouldBeTrue)
			So(post1.Permissions.CanModerate, ShouldBeTrue)

			post2 := feed.Posts[1]
			So(post2, ShouldNotBeNil)
			So(post2.ID, ShouldEqual, testClipID)
			So(post2.Permissions, ShouldNotBeNil)
			So(post2.Permissions.CanDelete, ShouldBeFalse)
			So(post2.Permissions.CanModerate, ShouldBeFalse)
		})

		Convey("Should be able to react to a post", func() {
			reaction1, err := createReaction(ts, testClipID, testEmoteKappa, testUserID)
			So(err, ShouldBeNil)
			So(reaction1, ShouldNotBeNil)
			So(reaction1.EmoteIDs, ShouldContain, testEmoteKappa)

			post, err := getPost(ts, testClipID, testUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			So(post.Reactions, ShouldContainKey, testEmoteKappa)
			So(post.Reactions[testEmoteKappa].UserIDs, ShouldContain, testUserID)
		})
	})

	ts.onFinish(time.Second * 3)
}

func createTemporaryUser(ctx context.Context, usersClient users.Client, c C) string {
	res, err := usersClient.CreateUser(ctx, &models.CreateUserProperties{
		Login:    uuid.NewV4().String()[:16],
		Email:    timestamp() + "@" + timestamp() + ".com",
		Birthday: sampleBirthday,
	}, nil)
	c.So(err, ShouldBeNil)
	c.So(res.ID, ShouldNotEqual, "")
	c.Reset(func() {
		So(usersClient.ExpireUserByID(ctx, res.ID, nil), ShouldBeNil)
	})
	return res.ID
}

func TestIntegration_SuggestedFeeds(t *testing.T) {
	t.Parallel()
	ts := startServer(t, injectables{}, nil)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func(c C) {
		So(ts.Setup(), ShouldBeNil)
		c.Reset(ts.cancelFunc)

		Convey("Should be able to fetch some feed IDs", func() {
			feed, err := getSuggestedFeeds(ts, testUserID)
			So(err, ShouldBeNil)
			So(feed, ShouldNotBeNil)
			So(len(feed.FeedIDs), ShouldBeGreaterThan, 0)
		})
		Convey("Should require user ID", func() {
			_, err := getSuggestedFeeds(ts, "")
			So(err, ShouldNotBeNil)
		})
	})
}

func TestIntegration_Feed(t *testing.T) {
	t.Parallel()
	ts := startServer(t, injectables{})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func(c C) {
		So(ts.Setup(), ShouldBeNil)
		c.Reset(ts.cancelFunc)

		SkipConvey("Non existent feed should return an empty feed", func() {
			feedID := "c:" + timestamp()
			feed, err := getFeed(ts, feedID, testUserID, nil)
			So(err, ShouldBeNil)
			So(feed, ShouldNotBeNil)
			So(feed.Posts, ShouldBeEmpty)
		})

		Convey("with a created share", func() {
			testBody := "TestIntegration_Share " + timestamp()
			createdPost, err := createPost(ts, testUserID2, testBody, testUserID2)
			So(err, ShouldBeNil)
			So(createdPost, ShouldNotBeNil)

			share, err := createShare(ts, testUserID, entity.New("post", createdPost.ID))
			So(err, ShouldBeNil)
			So(share, ShouldNotBeNil)

			Convey("Should not be able to share it again", func() {
				_, err := createShare(ts, testUserID, entity.New("post", createdPost.ID))
				So(errorCode(err), ShouldEqual, http.StatusConflict)
			})

			Convey("Should see shares in feed", func() {
				feedID := "c:" + testUserID
				feed, err := getFeed(ts, feedID, testUserID, toMockFeedOfActivity(feedID, masonry.Activity{
					Entity: entity.New(entity.NamespaceShare, share.ID),
					Verb:   verb.Create,
					Actor:  entity.New(entity.NamespaceUser, share.UserID),
				}))
				So(err, ShouldBeNil)
				So(feed.Posts, ShouldHaveLength, 1)
				So(feed.Posts[0].ID, ShouldEqual, createdPost.ID)
				So(feed.Posts[0].Reasons[0].Type, ShouldEqual, api.ShareCreatedType)
				So(feed.Posts[0].Reasons[0].UserID, ShouldEqual, testUserID)
				So(feed.Posts[0].ShareSummary.UserIDs, ShouldResemble, []string{testUserID})
				So(feed.Posts[0].ShareSummary.ShareCount, ShouldEqual, 1)
			})
			Convey("Should see share counts on a loaded post", func() {
				feedID := "c:" + testUserID
				feed, err := getFeed(ts, feedID, testUserID, aws.String("post:"+createdPost.ID))
				So(err, ShouldBeNil)
				So(feed.Posts, ShouldHaveLength, 1)
				So(feed.Posts[0].ID, ShouldEqual, createdPost.ID)
				So(feed.Posts[0].ShareSummary, ShouldNotBeNil)
				So(feed.Posts[0].ShareSummary.UserIDs, ShouldResemble, []string{testUserID})
				So(feed.Posts[0].ShareSummary.ShareCount, ShouldEqual, 1)
			})
		})

		// TODO: Remove skip: Friendship service broken in staging
		SkipConvey("With two setup users that are friends", func(c C) {
			feedOwnerID := createTemporaryUser(ts.ctx, ts.thisInstance.usersClient, c)
			feedOwnerFriend := createTemporaryUser(ts.ctx, ts.thisInstance.usersClient, c)

			var err error
			for i := 0; i < 3; i++ {
				err = ts.thisInstance.FriendshipClient.(friendship.Client).BulkAddFriends(ts.ctx, feedOwnerID, []string{feedOwnerFriend}, nil)
				if err == nil {
					break
				}
				time.Sleep(time.Second)
			}
			So(err, ShouldBeNil)

			Convey("If someone makes a post to their feed", func() {
				testBody := "TestIntegration_Feed " + timestamp()
				post, err := createPost(ts, feedOwnerID, testBody, feedOwnerID)
				So(err, ShouldBeNil)
				So(post, ShouldNotBeNil)
				Convey("And someone comments on that post", func() {
					Convey("If the second user is ToS banned", func() {
						So(ts.thisInstance.usersClient.BanUserByID(ts.ctx, &models.BanUserProperties{
							TargetUserID: feedOwnerFriend,
							Type:         "tos",
							Reason:       "integration_test",
						}, nil), ShouldBeNil)
					})
				})
			})
		})

		Convey("Should get mock items as entities", func() {
			feedID := "1234"
			userID := "1234"
			postID := "post:12345"
			feed, err := getFeedEntities(ts, feedID, userID, toMockFeed(postID))
			So(err, ShouldBeNil)
			So(len(feed.Items), ShouldEqual, 1)
			So(feed.Items[0].Entity.Encode(), ShouldEqual, "post:12345")
			So(feed.Items[0].Tracking, ShouldNotBeNil)
			So(feed.Items[0].Tracking.CardImpressionID, ShouldNotBeNil)
		})

		Convey("Should be able to create a post and get a feed back with that post", func() {
			testBody := "TestIntegration_Feed " + timestamp()
			post, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)

			feedID := "c:" + testUserID
			feed, headers, err := getFeedAndHeaders(ts, feedID, testUserID, nil, toMockFeedOfActivity(feedID, masonry.Activity{
				Entity: entity.New(entity.NamespacePost, post.ID),
				Verb:   verb.Create,
				Actor:  entity.New(entity.NamespaceUser, post.UserID),
			}))
			So(err, ShouldBeNil)
			So(feed, ShouldNotBeNil)
			So(headers.Get("Cache-Control"), ShouldBeEmpty)
			So(feed.Posts, ShouldNotBeEmpty)
			So(feed.Posts[0].ShareSummary.ShareCount, ShouldEqual, 0)
			post.ShareSummary.UserIDs = turnNilIntoEmpty(post.ShareSummary.UserIDs)

			So(feed.Posts[0].Reasons[0].Type, ShouldEqual, api.PostCreatedType)
			So(feed.Posts[0].Reasons[0].UserID, ShouldEqual, testUserID)

			// The originally fetched post won't have a Reason
			feed.Posts[0].Reasons = nil
			assertPostsAreEquivalent(t, feed.Posts[0], post)

			Convey("and anon users should see a cached feed", func() {
				feedID := "c:" + testUserID
				feed, headers, err := getFeedAndHeaders(ts, feedID, "", nil, toMockFeed(post.ID))
				So(err, ShouldBeNil)
				So(feed, ShouldNotBeNil)
				So(headers.Get("Cache-Control"), ShouldNotBeEmpty)
				So(headers.Get("Cache-Control"), ShouldContainSubstring, "public")
				So(feed.Posts, ShouldNotBeEmpty)
				So(feed.Posts[0].ShareSummary.ShareCount, ShouldEqual, 0)
				post.ShareSummary.UserIDs = turnNilIntoEmpty(post.ShareSummary.UserIDs)
				So(feed.Posts[0].ID, ShouldEqual, post.ID)
			})
		})

		Convey("Should be able to create a post with emotes", func() {
			testBody := "TestIntegration_Feed Kappa PogChamp " + timestamp()
			post, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)

			feedID := "c:" + testUserID
			feed, err := getFeed(ts, feedID, testUserID, toMockFeed(post.ID))
			So(err, ShouldBeNil)
			So(feed, ShouldNotBeNil)
			So(feed.Posts, ShouldNotBeEmpty)
			So(feed.Posts[0].ShareSummary.ShareCount, ShouldEqual, 0)
			post.ShareSummary.UserIDs = turnNilIntoEmpty(post.ShareSummary.UserIDs)
			feed.Posts[0].Reasons = nil
			assertPostsAreEquivalent(t, feed.Posts[0], post)
			So(post.Emotes, ShouldResemble, []*api.Emote{
				{Set: 0, ID: 25, Start: 21, End: 25},
				{Set: 0, ID: 88, Start: 27, End: 34},
			})
		})

		Convey("Should be able to get a feed with posts", func() {
			testPostBody := "TestIntegration_Feed " + timestamp()
			post, err := createPost(ts, testChannelID, testPostBody, testUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)

			feedID := "c:" + testChannelID
			Convey("and get those posts back in the feed", func() {
				feed, err := getFeed(ts, feedID, testUserID, toMockFeed(post.ID))
				So(err, ShouldBeNil)
				So(feed, ShouldNotBeNil)
				So(feed.Posts, ShouldNotBeEmpty)

				So(feed.Posts[0].ID, ShouldEqual, post.ID)
			})
		})

		Convey("Should be able to get a feed with reactions", func() {
			testPostBody := "TestIntegration_Feed " + timestamp()
			post, err := createPost(ts, testChannelID, testPostBody, testUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)

			postReaction, err := createReaction(ts, postEntity(post.ID), testEmoteKappa, testUserID)
			So(err, ShouldBeNil)
			So(postReaction, ShouldNotBeNil)
			So(postReaction.EmoteIDs, ShouldContain, testEmoteKappa)

			feedID := "c:" + testUserID
			feed, err := getFeed(ts, feedID, testUserID, toMockFeed(post.ID))
			So(err, ShouldBeNil)
			So(feed, ShouldNotBeNil)
			So(feed.Posts, ShouldHaveLength, 1)

			So(feed.Posts[0].ID, ShouldEqual, post.ID)
			post = feed.Posts[0]
			So(post.Reactions[testEmoteKappa].Count, ShouldEqual, 1)
		})

		Convey("Whitelisted news feed should return a feed", func() {
			testBody := "TestIntegration_Feed " + timestamp()
			post, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)

			feedID := "n:" + testUserID
			feed, err := getFeed(ts, feedID, testUserID, toMockFeed(post.ID))
			So(err, ShouldBeNil)
			So(feed, ShouldNotBeNil)
			So(feed.Posts, ShouldNotBeEmpty)
			post.ShareSummary.UserIDs = turnNilIntoEmpty(post.ShareSummary.UserIDs)

			// The originally fetched post won't have a Reason
			feed.Posts[0].Reasons = nil
			assertPostsAreEquivalent(t, feed.Posts[0], post)
		})

		Convey("Should be able to populate a feed of vod or clips", func() {
			feedID := "r:" + testUserID
			vod1 := testVodID
			clip1 := "clip:EvilPolishedCurrySoonerLater"
			feed, err := getFeed(ts, feedID, testUserID, toMockFeed(vod1, clip1))
			So(err, ShouldBeNil)
			So(feed.Posts, ShouldHaveLength, 2)
			So(feed.Posts[0].UserID, ShouldEqual, testVodAuthorID)
		})

		SkipConvey("Should be able to get a recommended feed with post tracking info populated", func() {
			feedID := "r:" + testUserID

			feed, err := getFeedWithLimit(ts, feedID, testUserID, "10")
			So(err, ShouldBeNil)
			So(len(feed.Posts), ShouldBeGreaterThanOrEqualTo, 2)

			post0 := feed.Posts[0]
			So(post0, ShouldNotBeNil)
			So(post0.Tracking, ShouldNotBeNil)
			So(post0.Tracking.CardImpressionID, ShouldNotBeBlank)
			So(post0.Tracking.BatchID, ShouldNotBeBlank)
			So(post0.Tracking.RecGenerationID, ShouldNotBeBlank)
			So(post0.Tracking.RecGenerationIndex, ShouldNotBeNil)
			So(*post0.Tracking.RecGenerationIndex, ShouldEqual, 0)

			post1 := feed.Posts[1]
			So(post1, ShouldNotBeNil)
			So(post1.Tracking, ShouldNotBeNil)
			So(post1.Tracking.CardImpressionID, ShouldNotEqual, post0.Tracking.CardImpressionID)
		})

		Convey("Should filter out invalid content from a feed", func() {
			feedID := "r:" + testUserID
			vod1 := testInvalidVodID
			clip1 := "clip:EvilPolishedCurrySoonerLater"
			feed, err := getFeedEntities(ts, feedID, testUserID, toMockFeed(vod1, clip1))
			So(err, ShouldBeNil)
			So(feed.Items, ShouldHaveLength, 1)
			So(feed.Items[0].Entity.Encode(), ShouldEqual, clip1)
		})
	})

	ts.onFinish(time.Second * 3)
}

// turnNilIntoEmpty is needed so ShouldResemble will work correctly
func turnNilIntoEmpty(s []string) []string {
	if s == nil {
		return []string{}
	}
	return s
}

func TestIntegration_Settings(t *testing.T) {
	ts := startServer(t, injectables{})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func(c C) {
		So(ts.Setup(), ShouldBeNil)
		c.Reset(ts.cancelFunc)

		Convey("Should be able to get settings for nonexistent user", func() {
			settingsUserID := "feeds-edge_integrationtest:" + timestamp()
			settingsEntity := userEntity(settingsUserID)
			settings, err := getSettings(ts, settingsEntity, settingsUserID)
			So(err, ShouldBeNil)
			So(settings.Entity, ShouldEqual, settingsEntity)
		})

		Convey("Should not be able to get settings for different user", func() {
			settingsUserID := "feeds-edge_integrationtest:" + timestamp()
			settingsEntity := userEntity(settingsUserID)
			settings, err := getSettings(ts, settingsEntity, testUserID)
			So(errorCode(err), ShouldEqual, http.StatusForbidden)
			So(settings, ShouldBeNil)
		})

		Convey("Should be able to update settings for nonexistent user", func() {
			settingsUserID := "feeds-edge_integrationtest:" + timestamp()
			settingsEntity := userEntity(settingsUserID)
			updates := &updateSettingsParams{ChannelFeedEnabled: aws.Bool(true)}
			settings, err := updateSettings(ts, settingsEntity, settingsUserID, updates)
			So(err, ShouldBeNil)
			So(settings.Entity, ShouldEqual, settingsEntity)
			So(settings.ChannelFeedEnabled, ShouldEqual, *updates.ChannelFeedEnabled)
		})

		Convey("Should not be able to update settings for different user", func() {
			settingsUserID := "feeds-edge_integrationtest:" + timestamp()
			settingsEntity := userEntity(settingsUserID)
			updates := &updateSettingsParams{ChannelFeedEnabled: aws.Bool(true)}
			settings, err := updateSettings(ts, settingsEntity, testUserID, updates)
			So(errorCode(err), ShouldEqual, http.StatusForbidden)
			So(settings, ShouldBeNil)
		})

		Convey("Should be able to update all settings", func() {
			settingsUserID := "feeds-edge_integrationtest:" + timestamp()
			settingsEntity := userEntity(settingsUserID)
			oldSettings, err := getSettings(ts, settingsEntity, settingsUserID)
			So(err, ShouldBeNil)
			updates := &updateSettingsParams{
				ChannelFeedEnabled: aws.Bool(!oldSettings.ChannelFeedEnabled),
			}

			newSettings, err := updateSettings(ts, settingsEntity, settingsUserID, updates)
			So(err, ShouldBeNil)
			So(newSettings.ChannelFeedEnabled, ShouldEqual, *(updates.ChannelFeedEnabled))
		})
	})

	ts.onFinish(time.Second * 3)
}

type nilCheckMock struct {
	*mocks.SpadeClient `nilcheck:"nodepth"`
}

func TestIntegration_WithSpadeMock(t *testing.T) {
	t.Parallel()

	spadeMock := nilCheckMock{&mocks.SpadeClient{}}
	spadeMock.On("Start").Return()

	ts := startServer(t, injectables{SpadeClient: spadeMock})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	checkForSpadeEvent := func(expectedEvent spade.Event) {
		events := make([]spade.Event, 0)
		for _, call := range spadeMock.Mock.Calls {
			if call.Method == "QueueEvents" {
				if event, ok := call.Arguments[0].(spade.Event); ok {
					events = append(events, event)
				}
			}
		}
		count := 0
		for _, event := range events {
			if reflect.DeepEqual(event, expectedEvent) {
				count++
			}
		}
		So(count, ShouldEqual, 1)
	}

	Convey("With "+ts.host, t, func(c C) {
		So(ts.Setup(), ShouldBeNil)
		c.Reset(ts.cancelFunc)
		resetMock(&spadeMock.Mock)
		spadeMock.On("Start").Return()
		spadeMock.On("QueueEvents", mock.Anything).Return()

		Convey("Create and deleted post with embed spade events", func() {
			userID := testUserID
			postBody := "TestIntegration_WithSpadeMock_Post " + timestamp()
			testUrl := "https://clips.twitch.tv/SecretiveMildOkapiYouDontSay"
			embedURLs := []string{testUrl}
			params := &createPostParams{
				UserID:    userID,
				Body:      postBody,
				EmbedURLs: &embedURLs,
			}
			post, cErr := createPostV2(ts, testChannelID, params)
			So(cErr, ShouldBeNil)
			So(post, ShouldNotBeNil)

			checkForSpadeEvent(spade.Event{
				Name: "feed_server_post",
				Properties: v2.ServerPostTracking{
					Entity:  "post:" + post.ID,
					PostID:  post.ID,
					Action:  "create",
					UserID:  userID,
					Content: postBody,
				},
			})

			So(post.EmbedEntities, ShouldNotBeNil)
			So(len(*post.EmbedEntities), ShouldBeGreaterThanOrEqualTo, 1)

			checkForSpadeEvent(spade.Event{
				Name: "feed_server_post_embed",
				Properties: v2.ServerPostEmbedTracking{
					Entity:      "post:" + post.ID,
					PostID:      post.ID,
					EmbedIndex:  int64(0),
					EmbedURL:    testUrl,
					EmbedEntity: (*post.EmbedEntities)[0].Encode(),
					EmbedType:   (*post.EmbedEntities)[0].Namespace(),
					EmbedID:     (*post.EmbedEntities)[0].ID(),
				},
			})

			_, dErr := deletePostV2(ts, post.ID, userID)
			So(dErr, ShouldBeNil)

			checkForSpadeEvent(spade.Event{
				Name: "feed_server_post",
				Properties: v2.ServerPostTracking{
					Entity: "post:" + post.ID,
					PostID: post.ID,
					Action: "remove",
					UserID: userID,
				},
			})
		})

		Convey("Create a post with embed spade events", func() {
			userID := testUserID
			testUrl := "https://clips.twitch.tv/SecretiveMildOkapiYouDontSay"
			postBody := "TestIntegration_WithSpadeMock_Post " + timestamp() + "  " + testUrl
			params := &createPostParams{
				UserID:    userID,
				Body:      postBody,
				EmbedURLs: nil,
			}

			post, cErr := createPostV2(ts, testChannelID, params)
			So(cErr, ShouldBeNil)
			So(post, ShouldNotBeNil)

			checkForSpadeEvent(spade.Event{
				Name: "feed_server_post_embed",
				Properties: v2.ServerPostEmbedTracking{
					Entity:      "post:" + post.ID,
					PostID:      post.ID,
					EmbedIndex:  int64(0),
					EmbedURL:    testUrl,
					EmbedEntity: (*post.EmbedEntities)[0].Encode(),
					EmbedType:   (*post.EmbedEntities)[0].Namespace(),
					EmbedID:     (*post.EmbedEntities)[0].ID(),
				},
			})
		})

		Convey("test reaction related spade events", func() {
			testPostBody := "TestIntegration_WithSpadeMock_Reaction " + timestamp()
			createdPost, err := createPost(ts, testChannelID, testPostBody, testUserID)
			So(err, ShouldBeNil)
			So(createdPost, ShouldNotBeNil)

			_, err = createReactionV2(ts, postEntity(createdPost.ID), testEmoteKappa, testUserID)
			So(err, ShouldBeNil)

			checkForSpadeEvent(spade.Event{
				Name: "feed_server_reaction",
				Properties: v2.ServerReactionTracking{
					Action:       "create",
					UserID:       testUserID,
					ReactionID:   testEmoteKappa,
					TargetEntity: "post:" + createdPost.ID,
					TargetType:   "post",
					TargetID:     createdPost.ID,
				},
			})

			_, err = deleteReactionV2(ts, postEntity(createdPost.ID), testEmoteKappa, testUserID)
			So(err, ShouldBeNil)

			checkForSpadeEvent(spade.Event{
				Name: "feed_server_reaction",
				Properties: v2.ServerReactionTracking{
					Action:       "remove",
					UserID:       testUserID,
					ReactionID:   testEmoteKappa,
					TargetEntity: "post:" + createdPost.ID,
					TargetType:   "post",
					TargetID:     createdPost.ID,
				},
			})
		})

		Convey("test share related spade events", func() {
			testBody := "TestIntegration_WithSpadeMock_Share " + timestamp()
			createdPost, err := createPost(ts, testChannelID, testBody, testUserID)
			So(err, ShouldBeNil)
			So(createdPost, ShouldNotBeNil)

			postEnt := entity.New("post", createdPost.ID)
			share, err := createShareV2(ts, testUserID2, postEnt)
			So(err, ShouldBeNil)
			So(share, ShouldNotBeNil)

			checkForSpadeEvent(spade.Event{
				Name: "feed_server_share",
				Properties: v2.ServerShareTracking{
					Action:       "create",
					UserID:       testUserID2,
					TargetEntity: postEnt.Encode(),
					TargetType:   postEnt.Namespace(),
					TargetID:     postEnt.ID(),
				},
			})

			_, err = deleteShareV2(ts, share.ID, testUserID2)
			So(err, ShouldBeNil)

			checkForSpadeEvent(spade.Event{
				Name: "feed_server_share",
				Properties: v2.ServerShareTracking{
					Action:       "remove",
					UserID:       testUserID2,
					TargetEntity: postEnt.Encode(),
					TargetType:   postEnt.Namespace(),
					TargetID:     postEnt.ID(),
				},
			})
		})
	})

	ts.onFinish(time.Second * 3)
}

type HTTPError interface {
	error
	HTTPCode() int
}

func errorCode(err error) int {
	if errHttp, ok := errors.Cause(err).(HTTPError); ok {
		return errHttp.HTTPCode()
	}
	return -1
}

func resetMock(m *mock.Mock) {
	m.ExpectedCalls = nil
	m.Calls = nil
}

func newUserID() string {
	return "user" + timestamp()
}

type testSetup struct {
	ctx          context.Context
	cancelFunc   func()
	client       *http.Client
	host         string
	onFinish     func(timeToWait time.Duration)
	thisInstance *service
}

func (t *testSetup) Setup() error {
	t.ctx, t.cancelFunc = context.WithTimeout(context.Background(), time.Second*5)
	t.client = &http.Client{}
	return nil
}

func request(ts *testSetup, method, url string, body interface{}, into interface{}) error {
	return clients.DoHTTP(ts.ctx, ts.client, method, ts.host+url, nil, body, into, nil)
}

type cachedReq struct {
	resp   *http.Response
	client *http.Client
}

func (c *cachedReq) Do(req *http.Request) (*http.Response, error) {
	resp, err := c.client.Do(req)
	c.resp = resp
	return resp, err
}

func requestAndHeaders(ts *testSetup, method, url string, body interface{}, into interface{}) (error, http.Header) {
	c := &cachedReq{
		client: ts.client,
	}
	err := clients.DoHTTP(ts.ctx, c, method, ts.host+url, nil, body, into, nil)
	return err, c.resp.Header
}

func addMapValues(m *distconf.InMemory, vals map[string][]byte) error {
	for k, v := range vals {
		if err := m.Write(k, v); err != nil {
			return err
		}
	}
	return nil
}

type panicPanic struct{}

func (p panicPanic) OnPanic(pnc interface{}) {
	panic(pnc)
}

func assertPostsAreEquivalent(t *testing.T, actual, expected *api.Post) {
	if actual == nil {
		So(actual, ShouldResemble, expected)
		return
	}

	// Clear Tracking field because it contains guids that will be different for every post.
	actual.Tracking = nil
	expected.Tracking = nil

	So(actual, ShouldResemble, expected)
}

type unitTestRails struct {
	rails.Rails
}

func (u *unitTestRails) AllEmoticons(ctx context.Context, reqOpts *twitchhttp.ReqOpts) (*rails.EmoticonsResp, error) {
	return &rails.EmoticonsResp{
		EmoticonSets: map[string][]rails.Emoticon{
			"1": {
				{
					Id:      25,
					Pattern: "Kappa",
				},
				{
					Id:      88,
					Pattern: "PogChamp",
				},
			},
		},
	}, nil
}

func startServer(t *testing.T, i injectables, configs ...map[string][]byte) *testSetup {
	localConf := &distconf.InMemory{}
	err := addMapValues(localConf, map[string][]byte{
		"feeds-edge.listen_addr": []byte(":0"),
		"rollbar.access_token":   []byte(""),
		"statsd.hostport":        []byte(""),
		"debug.addr":             []byte(":0"),
		"logging.to_stdout":      []byte("false"),
		"logging.to_stderr":      []byte("true"),

		// Disable "create post" cooldowns so that tests can run more quickly.
		"feeds-edge.create_comment_cooldown": []byte("0"),
		"feeds-edge.create_post_cooldown":    []byte("0"),

		// Disable creating reports, because Leviathan does not have a staging env, and we don't want our
		// tests to create false reports in Production.
		"feeds-edge.clients.leviathan.allow_reports": []byte("false"),
		"feeds-edge.validate_feed_items_rollout_pct": []byte("1.0"),
	})
	if err != nil {
		t.Error(err)
		return nil
	}
	for _, config := range configs {
		err := addMapValues(localConf, config)
		if err != nil {
			t.Error(err)
			return nil
		}
	}

	i.Rails = &unitTestRails{}

	started := make(chan string)
	finished := make(chan struct{})
	signalToClose := make(chan os.Signal)
	exitCalled := make(chan struct{})
	elevateKey := "hi"
	thisInstance := service{
		injectables: i,
		osExit: func(i int) {
			if i != 0 {
				t.Error("Invalid osExit status code", i)
			}
			close(exitCalled)
		},
		serviceCommon: service_common.ServiceCommon{
			ConfigCommon: service_common.ConfigCommon{
				Team:          teamName,
				Service:       serviceName,
				CustomReaders: []distconf.Reader{localConf},
				BaseDirectory: "../../",
				OsGetenv:      os.Getenv,
				OsHostname:    os.Hostname,
			},
			CodeVersion: CodeVersion,
			Log: &log.ElevatedLog{
				ElevateKey: elevateKey,
				NormalLog: log.ContextLogger{
					Logger: t,
				},
				DebugLog: log.ContextLogger{
					Logger: log.Discard,
				},
				LogToDebug: func(vals ...interface{}) bool {
					// TODO: Move context.Cancel to debug log
					return false
				},
			},
			PanicLogger:    panicPanic{},
			SfxSetupConfig: sfxStastdConfig(),
		},
		redisPrefix: strconv.Itoa(rand.Int()) + ":",
		sigChan:     signalToClose,
		onListen: func(listeningAddr net.Addr) {
			started <- fmt.Sprintf("http://localhost:%d", listeningAddr.(*net.TCPAddr).Port)
		},
	}
	thisInstance.serviceCommon.Log.NormalLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	thisInstance.serviceCommon.Log.DebugLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	go func() {
		thisInstance.main()
		close(finished)
	}()

	var addressForIntegrationTests string
	select {
	case <-exitCalled:
		return nil
	case addressForIntegrationTests = <-started:
	case <-time.After(time.Second * 35):
		t.Error("Took to long to start service")
		return nil
	}

	onFinish := func(timeToWait time.Duration) {
		signalToClose <- syscall.SIGTERM
		select {
		case <-finished:
			return
		case <-time.After(timeToWait):
			t.Error("Timed out waiting for server to end")
		}
	}
	return &testSetup{
		host:         addressForIntegrationTests,
		onFinish:     onFinish,
		thisInstance: &thisInstance,
	}
}
