locals {
  owner   = "jlindamo@twitch.tv"
  team    = "feeds"
  region  = "us-west-2"
  service = "feeds-edge"

  account_ids = {
    integration = 724951484461
    staging     = 724951484461
    production  = 914569885343
  }

  profiles = {
    integration = "twitch-feed-dev"
    staging     = "twitch-feed-dev"
    production  = "twitch-feed-aws"
  }

  min_counts = {
    integration = "1"
    staging     = "2"
    production  = "7"
  }

  desired_counts = {
    integration = "2"
    staging     = "3"
    production  = "8"
  }

  max_counts = {
    integration = "3"
    staging     = "4"
    production  = "24"
  }

  auto_promotes = {
    integration = "true"
    staging     = "true"
    production  = "false"
  }

  promote_froms = {
    integration = "latest"
    staging     = "integration"
    production  = "canary"
  }

  redis_instance_types = {
    integration = "cache.r3.xlarge"
    staging     = "cache.r3.xlarge"
    production  = "cache.r3.4xlarge"
  }

  redis_cache_nodes_counts = {
    integration = 1
    staging     = 1
    production  = 1
  }

  pushy_comment_sqs_arns = {
    integration = ""
    staging     = "arn:aws:sqs:us-west-2:603200399373:pushy_darklaunch_feedcomment"
    production  = "arn:aws:sqs:us-west-2:603200399373:pushy_production_feedcomment"
  }

  hard_delete_user_sns_arns = {
    integration = "arn:aws:sns:us-west-2:465369119046:notification-user-destroyed"
    staging     = "arn:aws:sns:us-west-2:465369119046:notification-user-destroyed"
    production  = "arn:aws:sns:us-west-2:641044725657:notification-user-destroyed"
  }

  hard_delete_user_sns_arn = "${lookup(local.hard_delete_user_sns_arns, terraform.workspace)}"
  pushy_comment_sqs_arn    = "${lookup(local.pushy_comment_sqs_arns, terraform.workspace)}"
  redis_cache_nodes        = "${lookup(local.redis_cache_nodes_counts, terraform.workspace)}"
  redis_instance_type      = "${lookup(local.redis_instance_types, terraform.workspace)}"
  promote_from             = "${lookup(local.promote_froms, terraform.workspace)}"
  auto_promote             = "${lookup(local.auto_promotes, terraform.workspace)}"
  min_count                = "${lookup(local.min_counts, terraform.workspace)}"
  max_count                = "${lookup(local.max_counts, terraform.workspace)}"
  desired_count            = "${lookup(local.desired_counts, terraform.workspace)}"
  profile                  = "${lookup(local.profiles, terraform.workspace)}"
}

provider "aws" {
  profile             = "${local.profile}"
  allowed_account_ids = ["${lookup(local.account_ids, terraform.workspace)}"]
  region              = "${local.region}"
}

terraform {
  backend "s3" {
    bucket  = "twitch-feed-aws"
    key     = "feeds/feeds-edge-us-west-2.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

provider "jenkins" {
  url                = "https://jenkins.internal.justin.tv"
  shared_credentials = "${file(pathexpand("~/.jenkins_auth"))}"
}

provider "consul" {
  address    = "consul.internal.justin.tv"
  datacenter = "us-west2"
}

data "terraform_remote_state" "account" {
  backend = "s3"

  config {
    bucket  = "${local.profile}"
    key     = "tfstate/feed/terraform/accounts/${local.profile}"
    region  = "us-west-2"
    profile = "${local.profile}"
  }
}

data "terraform_remote_state" "account_core" {
  backend   = "s3"
  workspace = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "${local.team}/core-account.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "region_core" {
  backend   = "s3"
  workspace = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "${local.team}/core-region-${local.region}.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

resource "aws_elasticache_subnet_group" "primary_cache" {
  name        = "${local.service}-${terraform.workspace}-cache-subnet-group"
  description = "Subnet group for Redis cluster ${local.service} ${terraform.workspace}"
  subnet_ids  = ["${split(",", "${data.terraform_remote_state.account.private_subnets}")}"]

  lifecycle {
    prevent_destroy = true
  }
}

resource "aws_elasticache_cluster" "primary_cache" {
  cluster_id           = "fe-${terraform.workspace}"
  engine               = "redis"
  node_type            = "${local.redis_instance_type}"
  port                 = 6379
  num_cache_nodes      = "${local.redis_cache_nodes}"
  parameter_group_name = "default.redis3.2"
  subnet_group_name    = "${aws_elasticache_subnet_group.primary_cache.name}"
  security_group_ids   = ["${data.terraform_remote_state.account.twitch_subnets_sg}"]

  lifecycle {
    prevent_destroy = true
  }

  tags {
    Environment = "${terraform.workspace}"
    Service     = "${local.service}"
    Team        = "${local.team}"
  }
}

# Create SQS queues for hard deleting user
resource "aws_sqs_queue" "hard_delete_user" {
  name           = "hard_delete_user-${local.service}-${terraform.workspace}"
  redrive_policy = "{\"deadLetterTargetArn\":\"${aws_sqs_queue.hard_delete_user_dead_letter.arn}\",\"maxReceiveCount\":10}"
}

module "consulconf_harddelete_registry" {
  source      = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//consulconf?ref=v0.8.20"
  team        = "${local.team}"
  service     = "${local.service}"
  environment = "${terraform.workspace}"
  key         = "hard-delete.sqssource.queue_url"
  value       = "${aws_sqs_queue.hard_delete_user.id}"
}

resource "aws_sqs_queue" "hard_delete_user_dead_letter" {
  name = "hard_delete_user-dead-${local.service}-${terraform.workspace}"

  tags {
    Team        = "${local.team}"
    Service     = "${local.service}"
    Environment = "${terraform.workspace}"
  }
}

# Create policies for the hard delete user SQS topic
data "aws_iam_policy_document" "sqs_policy" {
  statement {
    sid = "VisualEditor0"

    actions = [
      "sqs:DeleteMessage",
      "sqs:DeleteMessageBatch",
      "sqs:ReceiveMessage",
      "sqs:SendMessage",
    ]

    resources = [
      "${aws_sqs_queue.hard_delete_user.arn}",
    ]
  }
}

resource "aws_iam_policy" "sqs_policy" {
  name        = "hard_delete_queue_policy_${local.service}_${terraform.workspace}"
  description = "Allows roles to process hard delete SQS queue for ${local.service} in ${terraform.workspace}"

  policy = "${data.aws_iam_policy_document.sqs_policy.json}"
}

# Add the subscription to hard delete user SNS topic
resource "aws_sns_topic_subscription" "hard_delete_user_sqs_target" {
  topic_arn = "${local.hard_delete_user_sns_arn}"
  protocol  = "sqs"
  endpoint  = "${aws_sqs_queue.hard_delete_user.arn}"
}

# Add permissions to the SQS queue
resource "aws_sqs_queue_policy" "hard_delete_user_permissions" {
  queue_url = "${aws_sqs_queue.hard_delete_user.id}"

  policy = <<POLICY
{
  "Version": "2012-10-17",
  "Id": "sqspolicy",
  "Statement": [
    {
      "Sid": "First",
      "Effect": "Allow",
      "Principal": "*",
      "Action": "SQS:SendMessage",
      "Resource": "${aws_sqs_queue.hard_delete_user.arn}",
      "Condition": {
        "ArnEquals": {
          "aws:SourceArn": "${local.hard_delete_user_sns_arn}"
        }
      }
    }
  ]
}
POLICY
}

module "service" {
  source           = "../../modules/cluster"
  environment      = "${terraform.workspace}"
  target_group_arn = "${module.alb.target_group_arn}"
  cluster_name     = "${data.terraform_remote_state.region_core.common_cluster_name}"

  min_size                  = "${local.min_count}"
  desired_count             = "${local.desired_count}"
  max_size                  = "${local.max_count}"
  service_iam_role          = "${data.terraform_remote_state.account_core.service_iam_role}"
  task_autoscale_iam_role   = "${data.terraform_remote_state.account_core.task_autoscale_iam_role}"
  profile                   = "${local.profile}"
  promote_from              = "${local.promote_from}"
  auto_promote              = "${local.auto_promote}"
  deploy_aws_role           = "${data.terraform_remote_state.account_core.build_role}"
  container_loggroup_region = "${data.terraform_remote_state.region_core.container_loggroup_region}"
  container_loggroup        = "${data.terraform_remote_state.region_core.container_loggroup}"
  aws_creds                 = "${data.terraform_remote_state.account_core.jenkins_build_secret}"
  testing_assume_role       = "${data.terraform_remote_state.account_core.build_user_arn}"
}

module "permissions" {
  source         = "../../modules/permissions"
  environment    = "${terraform.workspace}"
  iam_id         = "${module.service.iam_id}"
  sqs_policy_arn = "${aws_iam_policy.sqs_policy.arn}"
}

module "alb" {
  source            = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//alb?ref=v0.8.20"
  owner             = "${local.owner}"
  environment       = "${terraform.workspace}"
  access_log_bucket = "${local.profile}-logs"
  team              = "${local.team}"
  service           = "${local.service}"
  subnets           = "${data.terraform_remote_state.account.private_subnets}"
  security_groups   = "${data.terraform_remote_state.account.twitch_subnets_sg}"
  vpc_id            = "${data.terraform_remote_state.account.vpc_id}"
}

module alb_cloudwatch {
  source           = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//alb/cloudwatch?ref=v0.8.20"
  alb_arn          = "${module.alb.alb_arn}"
  alarm_sns_arn    = "${data.terraform_remote_state.region_core.pagerduty_alarm_arn}"
  target_group_arn = "${module.alb.target_group_arn}"
}

output target_group_arn {
  value = "${module.alb.target_group_arn}"
}

output task_role_arn {
  value = "${module.service.task_role_arn}"
}

output sqs_policy_arn {
  value = "${aws_iam_policy.sqs_policy.arn}"
}
