package main

import (
	"errors"
	"flag"
	"fmt"
	"log"
	"net"
	"strconv"
	"time"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/chat/badges/app/clients"
	"code.justin.tv/chat/badges/app/db"
	"code.justin.tv/chat/badges/app/kraken"
	"code.justin.tv/chat/badges/app/logic"
	"code.justin.tv/chat/badges/app/s3"
	"code.justin.tv/chat/badges/app/server"
	"code.justin.tv/chat/badges/config"
	"code.justin.tv/chat/golibs/clients/owlplus"
	"code.justin.tv/chat/golibs/clients/usersservice"
	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/chat/jsonconf"
	"code.justin.tv/chat/redicache"
	"code.justin.tv/chat/redis"
	tmi "code.justin.tv/chat/tmi/client"
	"code.justin.tv/common/twitchhttp"
)

const (
	oauthCacheTTL = 10 * time.Minute
)

var gometricsInterval = 5 * time.Second

func main() {
	confPath := flag.String("conf", "", "Path to config file")
	flag.Parse()

	conf, err := parseConfig(*confPath)
	if err != nil {
		log.Fatalln("error parsing config:", err)
	}

	logx.InitDefaultLogger(logx.Config{
		RollbarToken: conf.Rollbar.Token,
		RollbarEnv:   conf.Rollbar.Environment,
	})

	clients, err := setupClients(conf)
	if err != nil {
		log.Fatalln("error setting up clients:", err)
	}

	l := logic.NewLogic(conf, clients)
	s := server.NewServer(conf, clients, l)

	params := gojiplus.ServerParams{
		Port:           conf.Port,
		DebugPort:      conf.DebugPort,
		GometricsStats: clients.Stats,
	}
	if err := gojiplus.ListenAndServe(s.Handler, params); err != nil {
		log.Fatalln("Error starting server:", err)
	}
}

func setupClients(conf *config.Config) (*clients.Clients, error) {
	var stats statsd.Statter
	if conf.Stats.Disabled {
		log.Println("Stats are disabled...")
		stats, _ = statsd.NewNoopClient()
	} else {
		var err error
		statsHostPort := hostport(conf.Stats.Host, conf.Stats.Port)
		stats, err = statsd.NewBufferedClient(statsHostPort, conf.Stats.Prefix, 1*time.Second, 512)
		if err != nil {
			return nil, errx.New(err)
		}
	}

	redisClient, err := redis.Init{
		KeyPrefix:      fmt.Sprintf("badges:%s", conf.Environment),
		Address:        conf.RedisHost,
		ConnectTimeout: 1 * time.Second,
		ReadTimeout:    1 * time.Second,
		WriteTimeout:   1 * time.Second,
		MaxConns:       1000,

		XactGroup:   "redis",
		StatsPrefix: "service.redis",
		Stats:       stats,
	}.New()
	if err != nil {
		return nil, errx.New(err)
	}

	cache := redicache.NewFromRedis(redisClient)

	dynamodb := db.New(&conf.DynamoDB, cache)

	kraken, err := kraken.NewClient(twitchhttp.ClientConf{
		Host: conf.Kraken.Host,
		Transport: twitchhttp.TransportConf{
			MaxIdleConnsPerHost: conf.Kraken.MaxIdleConns,
		},
		Stats: stats,
	})
	if err != nil {
		return nil, errx.New(err)
	}

	usersservice, err := usersservice.NewClient(twitchhttp.ClientConf{
		Host: conf.UsersService.Host,
		Transport: twitchhttp.TransportConf{
			MaxIdleConnsPerHost: conf.UsersService.MaxIdleConns,
		},
		Stats: stats,
	})
	if err != nil {
		return nil, errx.New(err)
	}

	owl, err := owlplus.NewCachedClient(&owlplus.Config{
		Hostport: conf.Owl.Host,
		TransportConf: twitchhttp.TransportConf{
			MaxIdleConnsPerHost: conf.Owl.MaxIdleConns,
		},
		Cache: cache,
		TTL:   oauthCacheTTL,
		Stats: stats,
	})
	if err != nil {
		return nil, errx.New(err)
	}

	tmiClient, err := tmi.NewClient(twitchhttp.ClientConf{
		Host: conf.TMI.Host,
		Transport: twitchhttp.TransportConf{
			MaxIdleConnsPerHost: conf.TMI.MaxIdleConns,
		},
		Stats: stats,
	})
	if err != nil {
		return nil, errx.New(err)
	}

	s3Client := s3.New(&conf.S3)

	return &clients.Clients{
		Stats:        stats,
		DynamoDB:     dynamodb,
		Kraken:       kraken,
		UsersService: usersservice,
		Redis:        redisClient,
		Cache:        cache,
		Owl:          owl,
		TMI:          tmiClient,
		S3:           s3Client,
	}, nil
}

func parseConfig(path string) (*config.Config, error) {
	if path == "" {
		return nil, errors.New("Empty config path")
	}

	conf := &config.Config{}
	err := jsonconf.ReadFile(conf, path)
	if err != nil {
		return nil, err
	}
	return conf, nil
}

func hostport(host string, port int) string {
	if port == 0 {
		return host
	}
	return net.JoinHostPort(host, strconv.Itoa(port))
}
