package emoticons

import (
	"encoding/json"
	"reflect"
	"sync"
	"testing"

	"github.com/stretchr/testify/assert"

	"code.justin.tv/chat/emoticons/models"
)

var (
	testLexicon = map[int][]Emoticon{
		100: []Emoticon{
			Emoticon{101, "Kappa"},
			Emoticon{102, "foobar#"},
			Emoticon{103, ":asdf:"},
		},
		200: []Emoticon{
			// :) and :-)
			Emoticon{201, "\\:-?\\)"},
			// :B and :-B
			Emoticon{202, "\\:-?B"},
			// B) and B-)
			Emoticon{203, "B-?\\)"},
			// <3
			Emoticon{204, "\\&lt\\;3"},
		},
	}
)

func TestUpdateLexicon(t *testing.T) {
	p := NewParser()
	assert.False(t, p.HasLexicon())
	p.UpdateLexicon(testLexicon)
	assert.True(t, p.HasLexicon())
}

func TestParse(t *testing.T) {
	doTestParse(t,
		"Kappa :)",
		[]int{0},
		[]emoticonmodels.Match{})

	doTestParse(t,
		"Kappa :)",
		[]int{100},
		[]emoticonmodels.Match{
			{101, 0, 4, 100},
		})

	doTestParse(t,
		"Kappa :)",
		[]int{100, 200},
		[]emoticonmodels.Match{
			{101, 0, 4, 100},
			{201, 6, 7, 200},
		})
}

func TestParseDoesNotMatchWordBoundaries(t *testing.T) {
	doTestParse(t,
		"xKappa Kappax xKappax",
		[]int{100, 200},
		[]emoticonmodels.Match{})

	doTestParse(t,
		"x:) :)x x:)x",
		[]int{100, 200},
		[]emoticonmodels.Match{})
}

func TestParseDoesNotMatchPunctuationBoundaries(t *testing.T) {
	doTestParse(t,
		".Kappa Kappa. .Kappa.",
		[]int{100, 200},
		[]emoticonmodels.Match{})

	doTestParse(t,
		".:) :). .:).",
		[]int{100, 200},
		[]emoticonmodels.Match{})
}

func TestParseMatchesNonOverlappingEmoticons(t *testing.T) {
	doTestParse(t,
		":-B B-)",
		[]int{200},
		[]emoticonmodels.Match{
			{202, 0, 2, 200},
			{203, 4, 6, 200},
		})

	doTestParse(t,
		":-BB-)",
		[]int{200},
		[]emoticonmodels.Match{})

	doTestParse(t,
		":-B-)",
		[]int{200},
		[]emoticonmodels.Match{})
}

func TestParseMatchesCaseSensitive(t *testing.T) {
	doTestParse(t,
		"Kappa",
		[]int{100},
		[]emoticonmodels.Match{
			{101, 0, 4, 100},
		})

	doTestParse(t,
		"kappa",
		[]int{100},
		[]emoticonmodels.Match{})

	doTestParse(t,
		"KAPPA",
		[]int{100},
		[]emoticonmodels.Match{})
}

func TestParseMatchesHtmlEntity(t *testing.T) {
	doTestParse(t,
		"<3",
		[]int{200},
		[]emoticonmodels.Match{
			{204, 0, 1, 200},
		})

	doTestParse(t,
		"&amp;",
		[]int{200},
		[]emoticonmodels.Match{})
}

func TestSetByEmoteID(t *testing.T) {
	p := NewParser()
	p.UpdateLexicon(testLexicon)

	setID, err := p.SetByEmoteID(101)
	assert.Equal(t, 100, setID)
	assert.Nil(t, err)

	setID, err = p.SetByEmoteID(333)
	assert.Equal(t, -1, setID)
	assert.NotNil(t, err)
}

func TestLookupEmote(t *testing.T) {
	p := NewParser()
	p.UpdateLexicon(testLexicon)

	name, err := p.LookupEmote(101)
	assert.Equal(t, "Kappa", name)
	assert.Nil(t, err)
}

func TestLookupEmoteParallel(t *testing.T) {
	// Will fail if there is a data race with UpdateLexicon+LookupEmote
	// Test with `go test -race ./...`
	var wg sync.WaitGroup
	p := NewParser()

	wg.Add(1)
	go func() {
		defer wg.Done()
		p.UpdateLexicon(testLexicon)
	}()

	wg.Add(1)
	go func() {
		defer wg.Done()
		p.LookupEmote(101)
	}()

	wg.Wait()
}

func doTestParse(t *testing.T, str string, sets []int, expected []emoticonmodels.Match) {
	t.Run("", func(t *testing.T) {
		p := NewParser()
		p.UpdateLexicon(testLexicon)

		matches := p.Parse(str, sets)

		if len(expected) != len(matches) {
			assert.Fail(t, "uneven length of matches")
		}

		// This'll catch when things come back out of order... sometimes.
		if !reflect.DeepEqual(matches, expected) {
			errTextReal, _ := json.Marshal(matches)
			errTextExp, _ := json.Marshal(expected)
			assert.Fail(t, "result doesn't match expected", string(errTextReal), string(errTextExp))
		}

		expectedMap := make(map[emoticonmodels.Match]struct{})
		for _, e := range expected {
			expectedMap[e] = struct{}{}
		}
		for _, m := range matches {
			if _, ok := expectedMap[m]; !ok {
				errText, _ := json.Marshal(m)
				assert.Fail(t, "unexpected match:", string(errText))
			}
			delete(expectedMap, m)
		}
	})
}
