package api

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"strings"

	"code.justin.tv/chat/friendship/app/backend"
	"code.justin.tv/chat/friendship/app/backend/backendtest"
	"github.com/cactus/go-statsd-client/statsd"

	. "github.com/smartystreets/goconvey/convey"
)

func serverWithTestBackend() (*Server, *backend.Backend) {
	b := backend.NewTestBackend()
	stats, err := statsd.NewNoopClient()
	So(err, ShouldBeNil)
	s, err := NewServer(b, stats, "development")
	So(err, ShouldBeNil)
	return s, b
}

func doTestHTTP(s *Server, method, path string) *httptest.ResponseRecorder {
	strings.TrimLeft(path, "/")

	r, err := http.NewRequest(method, fmt.Sprintf("http://localhost%s", path), nil)
	So(err, ShouldBeNil)

	w := httptest.NewRecorder()
	s.ServeHTTP(w, r)
	return w
}

// Used to validate multiple calls with possible ordering of kinds (asynchUpdate/Delete/Fetch functionality)
func validateCallsWithKinds(calls []backendtest.Call, expected backendtest.Call, kinds []string) {
	for _, call := range calls {
		So(call.Name, ShouldEqual, expected.Name)
		So(call.E1, ShouldEqual, expected.E1)
		So(kinds, ShouldContain, call.Kind)
		if expected.E2 != "" {
			So(call.E2, ShouldEqual, expected.E2)
		}
		if expected.EntityKind != "" {
			So(call.EntityKind, ShouldEqual, expected.EntityKind)
		}

		for k, v := range expected.Databag {
			So(v, ShouldEqual, call.Databag[k])
		}
	}
}

// Used to validate multiple calls with different entity ordering (asynchCount functionality)
func validateCallsWithEntities(calls []backendtest.Call, expected backendtest.Call, entities []string) {
	for _, call := range calls {
		So(call.Name, ShouldEqual, expected.Name)
		So(entities, ShouldContain, call.E1)
		So(call.Kind, ShouldEqual, expected.Kind)
		So(call.EntityKind, ShouldEqual, expected.EntityKind)
	}
}

func validateCall(call, expected backendtest.Call) {
	So(call.Name, ShouldEqual, expected.Name)
	So(call.E1, ShouldEqual, expected.E1)
	So(call.Kind, ShouldEqual, expected.Kind)
	if expected.E2 != "" {
		So(call.E2, ShouldEqual, expected.E2)
	}
	if expected.EntityKind != "" {
		So(call.EntityKind, ShouldEqual, expected.EntityKind)
	}

	for k, v := range expected.Databag {
		So(v, ShouldEqual, call.Databag[k])
	}
}
