package api

import (
	"fmt"
	"net/http"
	"testing"

	"code.justin.tv/chat/friendship/app/backend"
	"code.justin.tv/chat/friendship/app/backend/backendtest"
	"code.justin.tv/chat/friendship/app/internal/cursor"

	. "github.com/smartystreets/goconvey/convey"
)

func TestGetRequestsV1(t *testing.T) {
	var server *Server
	var b *backend.Backend
	var mockFriends *backendtest.CohesionClient

	Convey("When finding friend requests", t, func() {
		server, b = serverWithTestBackend()
		mockFriends = b.Friends.(*backendtest.CohesionClient)

		Convey("should fail when the limit param is invalid", func() {
			path := fmt.Sprintf("/v1/users/%s/friends/requests?limit=0", backend.AliceID)

			w := doTestHTTP(server, "GET", path)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, http.StatusText(w.Code))
		})

		Convey("should call the appropriate Cohesion client functions for users", func() {
			path := fmt.Sprintf("/v1/users/%s/friends/requests", backend.AliceID)

			doTestHTTP(server, "GET", path)
			So(len(mockFriends.GetCalls()), ShouldEqual, 4)

			expected := []backendtest.Call{
				backendtest.Call{
					Name:       "CountAssoc",
					E1:         backend.AliceID,
					Kind:       backend.RequestedByUser,
					EntityKind: backend.User,
				},
				backendtest.Call{
					Name:       "CountAssoc",
					E1:         backend.AliceID,
					Kind:       backend.RequestedByStranger,
					EntityKind: backend.User,
				},
				backendtest.Call{
					Name:       "ListAssoc",
					E1:         backend.AliceID,
					Kind:       backend.RequestedByUser,
					EntityKind: backend.User,
				},
				backendtest.Call{
					Name:       "ListAssoc",
					E1:         backend.AliceID,
					Kind:       backend.RequestedByStranger,
					EntityKind: backend.User,
				},
			}
			for i, call := range mockFriends.GetCalls() {
				validateCall(call, expected[i])
			}
		})

		Convey("should call the appropriate Cohesion client functions for strangers", func() {
			c := cursor.New(backend.RequestedByStranger, "foobar").Encode()
			path := fmt.Sprintf("/v1/users/%s/friends/requests?cursor=%s", backend.AliceID, c)

			doTestHTTP(server, "GET", path)
			So(len(mockFriends.GetCalls()), ShouldEqual, 3)

			expected := []backendtest.Call{
				backendtest.Call{
					Name:       "CountAssoc",
					E1:         backend.AliceID,
					Kind:       backend.RequestedByUser,
					EntityKind: backend.User,
				},
				backendtest.Call{
					Name:       "CountAssoc",
					E1:         backend.AliceID,
					Kind:       backend.RequestedByStranger,
					EntityKind: backend.User,
				},
				backendtest.Call{
					Name:       "ListAssoc",
					E1:         backend.AliceID,
					Kind:       backend.RequestedByStranger,
					EntityKind: backend.User,
				},
			}
			for i, call := range mockFriends.GetCalls() {
				validateCall(call, expected[i])
			}
		})
	})
}

func TestGetRequestsV0(t *testing.T) {
	var server *Server
	var b *backend.Backend
	var mockFriends *backendtest.CohesionClient

	Convey("When finding friend requests", t, func() {
		server, b = serverWithTestBackend()
		mockFriends = b.Friends.(*backendtest.CohesionClient)

		Convey("should fail when the limit param is invalid", func() {
			path := fmt.Sprintf("/users/%s/friends/requests?limit=0", backend.AliceID)

			w := doTestHTTP(server, "GET", path)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, http.StatusText(w.Code))
		})

		Convey("should call the appropriate Cohesion client functions for users", func() {
			path := fmt.Sprintf("/users/%s/friends/requests", backend.AliceID)

			doTestHTTP(server, "GET", path)
			So(len(mockFriends.GetCalls()), ShouldEqual, 4)

			calls := mockFriends.GetCalls()

			expected := backendtest.Call{
				Name:       "CountAssoc",
				E1:         backend.AliceID,
				Kind:       backend.RequestedByUser,
				EntityKind: backend.User,
			}
			validateCall(calls[0], expected)

			expected = backendtest.Call{
				Name:       "CountAssoc",
				E1:         backend.AliceID,
				Kind:       backend.RequestedByStranger,
				EntityKind: backend.User,
			}
			validateCall(calls[1], expected)

			expected = backendtest.Call{
				Name:       "ListAssoc",
				E1:         backend.AliceID,
				Kind:       backend.RequestedByUser,
				EntityKind: backend.User,
			}
			validateCall(calls[2], expected)

			expected = backendtest.Call{
				Name:       "ListAssoc",
				E1:         backend.AliceID,
				Kind:       backend.RequestedByStranger,
				EntityKind: backend.User,
			}
			validateCall(calls[3], expected)
		})
	})
}
