package api

import (
	"fmt"
	"net/http"
	"testing"

	"code.justin.tv/chat/friendship/app/backend"
	"code.justin.tv/chat/friendship/app/backend/backendtest"

	. "github.com/smartystreets/goconvey/convey"
)

func TestListFriends(t *testing.T) {
	var server *Server
	var b *backend.Backend

	Convey("When finding friends", t, func() {
		server, b = serverWithTestBackend()
		mockFriends := b.Friends.(*backendtest.CohesionClient)

		Convey("should fail when the limit param is invalid", func() {
			path := fmt.Sprintf("/users/%s/friends?limit=0", backend.BobID)

			w := doTestHTTP(server, "GET", path)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, http.StatusText(w.Code))
		})

		Convey("should fail when the sort param is invalid", func() {
			path := fmt.Sprintf("/users/%s/friends?sort=foobar", backend.BobID)

			w := doTestHTTP(server, "GET", path)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, http.StatusText(w.Code))
		})

		Convey("should call the appropriate Cohesion client functions", func() {
			path := fmt.Sprintf("/users/%s/friends", backend.AliceID)

			doTestHTTP(server, "GET", path)
			So(len(mockFriends.GetCalls()), ShouldEqual, 2)

			calls := mockFriends.GetCalls()
			listCall := calls[0]
			expected := backendtest.Call{
				Name:       "ListAssoc",
				E1:         backend.AliceID,
				Kind:       backend.FriendsWith,
				EntityKind: backend.User,
			}
			validateCall(listCall, expected)

			countCall := calls[1]
			expected = backendtest.Call{
				Name:       "CountAssoc",
				E1:         backend.AliceID,
				Kind:       backend.FriendsWith,
				EntityKind: backend.User,
			}
			validateCall(countCall, expected)
		})
	})
}
