package friendship

import (
	"fmt"
	"net/url"
	"strconv"
	"time"

	"code.justin.tv/common/twitchhttp"
	"golang.org/x/net/context"
)

type GetIncomingRequestsResponse struct {
	Requests []IncomingRequest `json:"requests"`
	Cursor   string            `json:"cursor"`
	Total    uint              `json:"total"`
}

type IncomingRequest struct {
	UserID            string    `json:"user_id"`
	RequestedAt       time.Time `json:"requested_at"`
	IsRecommended     bool      `json:"is_recommended"`
	IsStranger        bool      `json:"is_stranger"`
	NonStrangerReason string    `json:"non_stranger_reason"`
}

type GetIncomingRequestsParams struct {
	Limit  *int
	Cursor *string
	Sort   *string
}

func (c *clientImpl) GetIncomingRequests(
	ctx context.Context, userID string, params GetIncomingRequestsParams, reqOpts *twitchhttp.ReqOpts,
) (GetIncomingRequestsResponse, error) {
	q := url.Values{}
	if params.Limit != nil {
		q.Add("limit", strconv.Itoa(*params.Limit))
	}
	if params.Cursor != nil {
		q.Add("cursor", *params.Cursor)
	}
	if params.Sort != nil {
		q.Add("sort", *params.Sort)
	}

	u := &url.URL{
		Path:     fmt.Sprintf("/v1/users/%s/friends/requests", userID),
		RawQuery: q.Encode(),
	}
	req, err := c.NewRequest("GET", u.String(), nil)
	if err != nil {
		return GetIncomingRequestsResponse{}, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.friendship.get_requests",
		StatSampleRate: defaultStatSampleRate,
	})
	ctx, cancel := context.WithTimeout(ctx, defaultCtxTimeout)
	defer cancel()

	var decoded GetIncomingRequestsResponse
	if _, err := c.DoJSON(ctx, &decoded, req, combinedReqOpts); err != nil {
		return GetIncomingRequestsResponse{}, err
	}
	return decoded, nil
}
