package friendship

import (
	"errors"
	"fmt"
	"net/url"

	"golang.org/x/net/context"

	"code.justin.tv/chat/friendship/app/api/responses"
	"code.justin.tv/common/twitchhttp"
)

const (
	// FriendCriterion checks if user and target are friends.
	FriendCriterion = "friend"
	// ModCriterion checks if target set user as mod.
	ModCriterion = "mod"
	// EditorCriterion checks if target set user as editor.
	EditorCriterion = "editor"
	// HasWhisperedCriterion checks if target has whispered user.
	HasWhisperedCriterion = "has_whispered"
	// StaffCriterion checks if user is a Twitch staff.
	StaffCriterion = "staff"
	// AdminCriterion checks if user is a Twitch admin.
	AdminCriterion = "admin"
	// GlobalModCriterion checks if user is a Twitch global mod.
	GlobalModCriterion = "global_mod"
	// FollowerCriterion checks if target is follower of user.
	FollowerCriterion = "follower"
	// PartnerCriterion checks if user is partner.
	PartnerCriterion = "partner"
	// SubscriberCriterion checks if target is a subscriber to user's channel.
	SubscriberCriterion = "subscriber"
)

func (c *clientImpl) IsFamiliar(ctx context.Context, userID, targetID string, criteria []string, reqOpts *twitchhttp.ReqOpts) (string, bool, error) {
	if len(criteria) == 0 {
		return "", false, errors.New("no criteria passed in to IsFamiliar")
	}

	q := url.Values{"criteria": criteria}
	u := &url.URL{
		Path:     fmt.Sprintf("/v1/users/%s/friends/%s/familiar", userID, targetID),
		RawQuery: q.Encode(),
	}
	req, err := c.NewRequest("GET", u.String(), nil)
	if err != nil {
		return "", false, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.friendship.is_familiar",
		StatSampleRate: defaultStatSampleRate,
	})
	ctx, cancel := context.WithTimeout(ctx, defaultCtxTimeout)
	defer cancel()

	var decoded responses.IsFamiliarResponse
	if _, err := c.DoJSON(ctx, &decoded, req, combinedReqOpts); err != nil {
		return "", false, err
	}
	return decoded.Reason, decoded.IsFamiliar, nil
}
