package leviathan

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"strconv"

	"code.justin.tv/foundation/twitchclient"
)

const (
	defaultStatSampleRate = 1.0

	createReportPath        = "/reports"
	tosSuspensionReportPath = "/suspensions"

	defaultTimingXactName = "leviathan"

	maxBodySlurpSize = int64(2 << 10)
)

var (
	// ErrMessageInvalidReasonMatrix is returned when a report is submitted for a reason
	// that is not valid for that report type
	ErrMessageInvalidReasonMatrix = errors.New("invalid report reason for content type")

	// ErrBadRequest is returned if a call to leviathan results in a failed response code
	ErrBadRequest = errors.New("invalid request submitted")

	// ErrUnknownContentType is returned if a content type is requested but doesn't exist
	ErrUnknownContentType = errors.New("No such content type")

	// ErrInvalidTargetUserID is returned when a target user id is not valid
	ErrInvalidTargetUserID = errors.New("Invalid target user id")

	// ErrInvalidFromUserID is returned when a from user id is not valid
	ErrInvalidFromUserID = errors.New("Invalid from user id")
)

// CreateReportParams contains the information necessary to create a report.
type CreateReportParams struct {
	FromUserID   string
	TargetUserID string
	Reason       string
	Description  string
	Content      string
	Origin       string
	ContentID    string
	Extra        string
}

// TosSuspensionReportParam contains the paramemters required to create a TOS
// Suspension record in Leviathan
type TosSuspensionReportParam struct {
	FromUserID     string
	TargetUserID   string
	Reason         string
	DetailedReason string
	Description    string
	Content        string
	Origin         string
	Duration       int64
	IPBan          int64
	ClearImages    bool
}

// Client facilitates sending messages to the Leviathan service.
type Client interface {
	CreateReport(ctx context.Context, createReportParams CreateReportParams, reqOpts *twitchclient.ReqOpts) error
	TosSuspensionReport(ctx context.Context, tosSuspensionReport TosSuspensionReportParam, reqOpts *twitchclient.ReqOpts) error
}

// NewClient creates a Leviathan client.
func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchclient.NewClient(conf)
	return &leviathanImpl{twitchClient: twitchClient,
		authorizationToken: "JciEQf0Z-8G764yQVhhImgxjPsLStxvA", // (Aaron C.): remove this when leviathan is not externally available.
	}, err
}

type createReportRequestBody struct {
	AuthorizationToken string       `json:"authorization_token"`
	ReportParams       reportParams `json:"report"`
}

type tosSuspensionReportRequestBody struct {
	AuthorizationToken string                    `json:"authorization_token"`
	ReportParams       tosSuspensionReportParams `json:"suspension"`
}

type reportParams struct {
	FromUserID   int64  `json:"from_user_id"`
	TargetUserID int64  `json:"target_user_id"`
	Reason       string `json:"reason"`
	Description  string `json:"description"`
	Content      string `json:"content"`
	Origin       string `json:"origin,omitempty"`
	ContentID    string `json:"content_id,omitempty"`
	Extra        string `json:"extra1,omitempty"`
}

type tosSuspensionReportParams struct {
	FromUserID     string `json:"from_user_id"`
	TargetUserID   int64  `json:"target_user_id"`
	Reason         string `json:"reason"`
	Description    string `json:"description"`
	Content        string `json:"content,omitempty"`
	Origin         string `json:"origin"`
	Duration       int64  `json:"duration"`
	IPBan          int64  `json:"ip_ban"`
	ClearImages    bool   `json:"cleared_channel_images"`
	DetailedReason string `json:"detailed_reason,omitempty"`
}

type leviathanImpl struct {
	twitchClient       twitchclient.Client
	authorizationToken string
}

// Enforce that leviathanImpl implements Client.
var _ Client = leviathanImpl{}

func (d leviathanImpl) CreateReport(ctx context.Context, createReportParams CreateReportParams, reqOpts *twitchclient.ReqOpts) (retErr error) {
	reportParams, err := d.parseCreateReportParams(createReportParams)
	if err != nil {
		return err
	}

	requestBody := createReportRequestBody{
		AuthorizationToken: d.authorizationToken,
		ReportParams:       *reportParams,
	}

	body, err := json.Marshal(requestBody)
	if err != nil {
		return err
	}

	req, err := d.twitchClient.NewRequest("POST", createReportPath, bytes.NewBuffer(body))
	if err != nil {
		return err
	}
	req.Header.Add("Content-Type", "application/json")

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.leviathan.create_report",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := d.twitchClient.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer cleanupResponseBody(resp.Body, &retErr)

	if resp.StatusCode >= 400 && resp.StatusCode < 500 {
		return ErrBadRequest
	}

	if resp.StatusCode != http.StatusOK && resp.StatusCode != http.StatusCreated {
		errResp := &errResponse{
			code: resp.StatusCode,
		}

		_, err2 := io.CopyN(&errResp.body, resp.Body, maxBodySlurpSize)
		if err2 != nil && err2 != io.EOF {
			return err2
		}

		return errResp
	}

	return nil
}

func (d leviathanImpl) parseCreateReportParams(params CreateReportParams) (*reportParams, error) {
	targetUserID, err := strconv.ParseInt(params.TargetUserID, 10, 64)
	if err != nil {
		return nil, ErrInvalidTargetUserID
	}

	fromUserID, err := strconv.ParseInt(params.FromUserID, 10, 64)
	if err != nil {
		return nil, ErrInvalidFromUserID
	}

	ok := false
	for _, reason := range reportReasonMatrix[params.Content] {
		if reason == params.Reason {
			ok = true
			break
		}
	}

	if !ok {
		return nil, ErrMessageInvalidReasonMatrix
	}

	return &reportParams{
		FromUserID:   fromUserID,
		TargetUserID: targetUserID,
		Reason:       params.Reason,
		Description:  params.Description,
		Content:      params.Content,
		Origin:       params.Origin,
		ContentID:    params.ContentID,
		Extra:        params.Extra,
	}, nil
}

func (d leviathanImpl) TosSuspensionReport(ctx context.Context, tosSuspensionReport TosSuspensionReportParam, reqOpts *twitchclient.ReqOpts) (retErr error) {
	tosSuspensionReportParams, err := d.parseTosSuspensionReportParams(tosSuspensionReport)
	if err != nil {
		return err
	}

	requestBody := tosSuspensionReportRequestBody{
		AuthorizationToken: d.authorizationToken,
		ReportParams:       *tosSuspensionReportParams,
	}

	body, err := json.Marshal(requestBody)
	if err != nil {
		return err
	}

	req, err := d.twitchClient.NewRequest("POST", tosSuspensionReportPath, bytes.NewBuffer(body))
	if err != nil {
		return err
	}
	req.Header.Add("Content-Type", "application/json")

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.leviathan.tos_suspension_report",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := d.twitchClient.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer cleanupResponseBody(resp.Body, &retErr)

	if resp.StatusCode != http.StatusOK && resp.StatusCode != http.StatusCreated {
		errResp := &errResponse{
			code: resp.StatusCode,
		}

		_, err2 := io.CopyN(&errResp.body, resp.Body, maxBodySlurpSize)
		if err2 != nil && err2 != io.EOF {
			return err2
		}

		return errResp
	}

	return nil
}

func (d leviathanImpl) parseTosSuspensionReportParams(params TosSuspensionReportParam) (*tosSuspensionReportParams, error) {
	targetUserID, err := strconv.ParseInt(params.TargetUserID, 10, 64)
	if err != nil {
		return nil, err
	}

	return &tosSuspensionReportParams{
		FromUserID:     params.FromUserID,
		TargetUserID:   targetUserID,
		Reason:         params.Reason,
		Description:    params.Description,
		Content:        params.Content,
		Origin:         params.Origin,
		DetailedReason: params.DetailedReason,
		Duration:       params.Duration,
		IPBan:          params.IPBan,
		ClearImages:    params.ClearImages,
	}, nil
}

// Reason represents a valid reason ID and the display strings for the user. User side clients
// can either hardcode their own copy of the Display strings for locale reasons or extract them
// from the Reason.
//
// Note: Currently only "en" locale is populated
type Reason struct {
	ID      string            `json:"id"`      // The identifier of the reason
	Display map[string]string `json:"display"` // Map of locale to localized version of the reason's string to display to the user
}

// ReasonsForReportType returns the list of valid reasons for a report type
func ReasonsForReportType(reportType string) ([]Reason, error) {
	ids, ok := reportReasonMatrix[reportType]
	if !ok {
		return nil, ErrUnknownContentType
	}
	reasons := make([]Reason, 0, len(ids))
	for _, id := range ids {
		reasons = append(reasons, Reason{
			ID: id,
			Display: map[string]string{
				"en": englishReasons[id],
			},
		})
	}
	return reasons, nil
}

func cleanupResponseBody(body io.ReadCloser, retErr *error) {
	// Read the body so the underlying TCP connection will be re-used.twitchClient.
	_, err := io.CopyN(ioutil.Discard, body, maxBodySlurpSize)
	if err != io.EOF && err != nil && *retErr == nil {
		*retErr = err
	}

	err = body.Close()
	if err != nil && *retErr == nil {
		*retErr = err
	}
}

type errResponse struct {
	code int
	body bytes.Buffer
}

func (e *errResponse) Error() string {
	return fmt.Sprintf("invalid status code: %d: %s", e.code, e.body.String())
}

func (e *errResponse) HTTPCode() int {
	return e.code
}

// NoopClient is an implementation of Client that does nothing.
type NoopClient struct{}

// Enforce that NoopClient implements Client.
var _ Client = NoopClient{}

// CreateReport is a No-op implementation of the CreateReport method
func (n NoopClient) CreateReport(context.Context, CreateReportParams, *twitchclient.ReqOpts) error {
	return nil
}

// TosSuspensionReport is a No-op implementation of the TosSuspensionReport method
func (n NoopClient) TosSuspensionReport(context.Context, TosSuspensionReportParam, *twitchclient.ReqOpts) error {
	return nil
}
