package rails

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/url"
	"strconv"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/common/twitchhttp"
)

const (
	defaultStatSampleRate = 0.1
	defaultTimingXactName = "rails"
)

var emptyBody = map[string]interface{}{}

type Rails interface {
	AllEmoticons(ctx context.Context, reqOpts *twitchhttp.ReqOpts) (*EmoticonsResp, error)
	AreStreamUpEmailsDisabled(ctx context.Context, userID int, reqOpts *twitchhttp.ReqOpts) (bool, error)
	FacebookConnected(ctx context.Context, userName string, reqOpts *twitchhttp.ReqOpts) (bool, error)
	IsFamiliar(ctx context.Context, fromUserId, toUserId int, reqOpts *twitchhttp.ReqOpts) (bool, error)
	TriggerCommercial(ctx context.Context, channelName string, length time.Duration, reqOpts *twitchhttp.ReqOpts) (time.Duration, error)
}

type railsImpl struct {
	cl     twitchhttp.Client
	apiKey string
}

func New(conf twitchhttp.ClientConf, apiKey string) (Rails, error) {
	confWithXactName := addXactNameIfNoneExists(conf)
	twitchClient, err := twitchhttp.NewClient(confWithXactName)
	if err != nil {
		return nil, err
	}
	return &railsImpl{
		cl:     twitchClient,
		apiKey: apiKey,
	}, nil
}

func (r *railsImpl) do(ctx context.Context, req *http.Request, body interface{}, reqOpts twitchhttp.ReqOpts) (*http.Response, error) {
	resp, err := r.cl.Do(ctx, req, reqOpts)
	if err != nil {
		return nil, err
	}

	defer func() {
		err = resp.Body.Close()
		if err != nil {
		}
	}()

	if body == nil {
		return resp, nil
	}

	decoder := json.NewDecoder(resp.Body)
	err = decoder.Decode(body)
	if err != nil {
		return resp, err
	}

	return resp, nil
}

func (r *railsImpl) newRequest(method, path string, values url.Values) (*http.Request, error) {
	url := url.URL{
		Path:     path,
		RawQuery: values.Encode(),
	}

	switch method {
	case "GET", "DELETE":
		req, err := r.cl.NewRequest(method, url.String(), nil)
		if err != nil {
			return nil, err
		}
		req.Header.Add("Client-ID", r.apiKey)
		return req, nil
	case "POST", "PUT":
		bodyBytes, err := json.Marshal(emptyBody)
		if err != nil {
			return nil, err
		}
		req, err := r.cl.NewRequest(method, url.String(), bytes.NewReader(bodyBytes))
		if err != nil {
			return nil, err
		}
		req.Header.Add("Client-ID", r.apiKey)
		return req, nil
	default:
		return nil, errors.New("Rails: unrecognized http method")
	}
}

type triggerCommercialResp struct {
	Length int `json:"length"`
}

func (r *railsImpl) TriggerCommercial(ctx context.Context, channelName string, length time.Duration, reqOpts *twitchhttp.ReqOpts) (time.Duration, error) {
	path := fmt.Sprintf("/api/internal/users/%s/commercial", channelName)
	values := url.Values{"length": {strconv.Itoa(int(length.Seconds()))}}

	req, err := r.newRequest("POST", path, values)
	if err != nil {
		return 0, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "rails.internal.commercial",
		StatSampleRate: defaultStatSampleRate,
	})

	responseBody := &triggerCommercialResp{}

	resp, err := r.do(ctx, req, responseBody, combinedReqOpts)
	if err != nil {
		return 0, err
	}

	if resp.StatusCode != http.StatusOK {
		return 0, fmt.Errorf("rails.TriggerCommercial: bad http status %d", resp.StatusCode)
	}

	return time.Duration(responseBody.Length) * time.Second, nil
}

type EmoticonsResp struct {
	EmoticonSets map[string][]Emoticon `json:"emoticon_sets"`
}

type Emoticon struct {
	Id      int    `json:"id"`
	Pattern string `json:"code"`
}

func (r *railsImpl) AllEmoticons(ctx context.Context, reqOpts *twitchhttp.ReqOpts) (*EmoticonsResp, error) {
	path := "kraken/chat/emoticon_images"
	values := url.Values{"emotesets": {"all"}}

	req, err := r.newRequest("GET", path, values)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "rails.kraken.emoticon_images",
		StatSampleRate: defaultStatSampleRate,
	})

	responseBody := &EmoticonsResp{}

	resp, err := r.do(ctx, req, responseBody, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("rails.AllEmoticons: bad http status %d", resp.StatusCode)
	}

	return responseBody, nil
}

type facebookConnectedResp struct {
	Id                int    `json:"id"`
	Login             string `json:"login"`
	FacebookConnected bool   `json:"facebook_connected?"`
}

func (r *railsImpl) FacebookConnected(ctx context.Context, userName string, reqOpts *twitchhttp.ReqOpts) (bool, error) {
	path := "/api/internal/user/properties"
	values := url.Values{
		"logins":     {userName},
		"properties": {"facebook_connected?"},
	}

	req, err := r.newRequest("GET", path, values)
	if err != nil {
		return false, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "rails.internal.booluserprop",
		StatSampleRate: defaultStatSampleRate,
	})

	var responseBody []facebookConnectedResp

	resp, err := r.do(ctx, req, &responseBody, combinedReqOpts)
	if err != nil {
		return false, err
	}

	if resp.StatusCode != http.StatusOK {
		return false, fmt.Errorf("rails.FacebookConnected: bad http status %d", resp.StatusCode)
	}

	return responseBody[0].FacebookConnected, nil
}

type isStrangerResponse struct {
	IsStranger bool `json:"is_stranger"`
}

// Returns true if fromUserId is known to toUserId
// ex: toUserId subscribes to fromUserId, or fromUserId is an editor of toUserId's channel
func (r *railsImpl) IsFamiliar(ctx context.Context, fromUserId, toUserId int, reqOpts *twitchhttp.ReqOpts) (bool, error) {
	path := "/api/internal/friendships/stranger"
	values := url.Values{
		"from_user_id": {strconv.Itoa(fromUserId)},
		"to_user_id":   {strconv.Itoa(toUserId)},
	}

	req, err := r.newRequest("GET", path, values)
	if err != nil {
		return false, err
	}

	req.Host = "api.internal.twitch.tv"

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "rails.internal.is_stranger",
		StatSampleRate: defaultStatSampleRate,
	})

	responseBody := &isStrangerResponse{}

	resp, err := r.do(ctx, req, responseBody, combinedReqOpts)
	if err != nil {
		return false, err
	}

	if resp.StatusCode != http.StatusOK {
		return false, fmt.Errorf("rails.IsFamiliar: bad http status %d", resp.StatusCode)
	}

	return !responseBody.IsStranger, nil
}

type streamUpEmailsDisabledResponse struct {
	ID                        int    `json:"id"`
	Login                     string `json:"login"`
	AreStreamUpEmailsDisabled bool   `json:"stream_up_emails_disabled"`
}

// Returns true if the user has disabled "stream up" email notifications
// Also returns true if the user has disabled all email notifications
func (r *railsImpl) AreStreamUpEmailsDisabled(ctx context.Context, userID int, reqOpts *twitchhttp.ReqOpts) (bool, error) {
	values := url.Values{
		"ids":        {strconv.Itoa(userID)},
		"properties": {"stream_up_emails_disabled"},
	}
	req, err := r.newRequest("GET", "/api/internal/user/properties", values)
	if err != nil {
		return false, err
	}
	req.Host = "api.internal.twitch.tv"
	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "rails.internal.stream_up_emails_disabled",
		StatSampleRate: defaultStatSampleRate,
	})
	var responseBody []streamUpEmailsDisabledResponse
	resp, err := r.do(ctx, req, &responseBody, combinedReqOpts)
	if err != nil {
		return false, err
	}
	if resp.StatusCode != http.StatusOK {
		return false, fmt.Errorf("rails.AreStreamUpEmailsDisabled: bad http status %d", resp.StatusCode)
	}
	return responseBody[0].AreStreamUpEmailsDisabled, nil
}

func addXactNameIfNoneExists(conf twitchhttp.ClientConf) twitchhttp.ClientConf {
	if conf.TimingXactName != "" {
		return conf
	}
	conf.TimingXactName = defaultTimingXactName
	return conf
}
