package tmi

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"strconv"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/foundation/twitchclient"
)

// Responses
type ChannelBannedUsersResponse struct {
	TotalCount  int                 `json:"_total"`
	BannedUsers []ChannelBannedUser `json:"banned_users"`
}

type ChannelUserBannedStatusResponse struct {
	BannedUser ChannelBannedUser `json:"banned_user"`
}

type ChannelBannedUser struct {
	BannedUserID    int        `json:"banned_user_id"`
	RequesterUserID int        `json:"requester_user_id"`
	BannedAt        time.Time  `json:"banned_at"`
	ExpiresAt       *time.Time `json:"expires_at"`
}

type BanChannelUserResponse struct {
	ResponseCode string `json:"response_code"`
}

type UnbanChannelUserResponse struct {
	ResponseCode string `json:"response_code"`
}

// Requests
type BanChannelUserRequest struct {
	BannedUser BanChannelUserParams `json:"banned_user"`
}

type BanChannelUserParams struct {
	BannedUserID    int     `json:"banned_user_id"`
	RequesterUserID int     `json:"requester_user_id"`
	ExpiresIn       *string `json:"expires_in"`
}

type UnbanChannelUserRequest struct {
	UnbannedUser UnbanChannelUserParams `json:"unbanned_user"`
}

type UnbanChannelUserParams struct {
	RequesterUserID int `json:"requester_user_id"`
}

const (
	badBanCode = "bad_ban_error"
)

func (c *clientImpl) BanUser(ctx context.Context, channelID, bannedUserID, requesterUserID string, expiresIn *string, reqOpts *twitchclient.ReqOpts) (string, error) {

	path := fmt.Sprintf("/rooms/%s/bans", channelID)

	intBannedUserID, err := strconv.Atoi(bannedUserID)
	if err != nil {
		return badBanCode, err
	}

	intRequesterUserID, err := strconv.Atoi(requesterUserID)
	if err != nil {
		return badBanCode, err
	}

	bodyBytes, err := json.Marshal(
		BanChannelUserRequest{
			BannedUser: BanChannelUserParams{
				BannedUserID:    intBannedUserID,
				RequesterUserID: intRequesterUserID,
				ExpiresIn:       expiresIn,
			},
		},
	)

	req, err := c.NewRequest("PUT", path, bytes.NewReader(bodyBytes))
	if err != nil {
		return badBanCode, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.clue.ban_user",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return badBanCode, err
	}
	defer resp.Body.Close()

	decoded := BanChannelUserResponse{}
	if err := json.NewDecoder(resp.Body).Decode(&decoded); err != nil {
		return badBanCode, err
	}

	if resp.StatusCode != http.StatusOK {
		return decoded.ResponseCode, twitchclient.HandleFailedResponse(resp)
	}

	return decoded.ResponseCode, err
}

func (c *clientImpl) UnbanUser(ctx context.Context, channelID, userID, requesterID string, reqOpts *twitchclient.ReqOpts) (string, error) {
	badUnbanCode := "bad_unban_error"
	path := fmt.Sprintf("/rooms/%s/bans/%s", channelID, userID)

	intRequesterUserID, err := strconv.Atoi(requesterID)
	if err != nil {
		return badUnbanCode, err
	}

	bodyBytes, err := json.Marshal(
		UnbanChannelUserRequest{
			UnbannedUser: UnbanChannelUserParams{
				RequesterUserID: intRequesterUserID,
			},
		},
	)

	req, err := c.NewRequest("DELETE", path, bytes.NewReader(bodyBytes))
	if err != nil {
		return badUnbanCode, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.clue.unban_user",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return badUnbanCode, err
	}
	defer resp.Body.Close()

	decoded := UnbanChannelUserResponse{}
	if err := json.NewDecoder(resp.Body).Decode(&decoded); err != nil {
		return badUnbanCode, err
	}

	if resp.StatusCode != http.StatusOK {
		return decoded.ResponseCode, twitchclient.HandleFailedResponse(resp)
	}

	return decoded.ResponseCode, err
}

// IsBanned returns true if the given user is banned from the given channel, or false otherwise
func (c *clientImpl) GetBanStatus(ctx context.Context, channelID, userID string, reqOpts *twitchclient.ReqOpts) (ChannelBannedUser, bool, error) {
	path := fmt.Sprintf("/rooms/%s/bans/%s", channelID, userID)
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return ChannelBannedUser{}, false, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.clue.is_banned",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return ChannelBannedUser{}, false, err
	}
	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()

	if resp.StatusCode == http.StatusNotFound {
		// endpoint returns 404 if a user is not banned in room
		return ChannelBannedUser{}, false, nil
	} else if resp.StatusCode != http.StatusOK {
		// unexpected status code
		return ChannelBannedUser{}, false, httpErrorImpl{
			statusCode: resp.StatusCode,
		}
	}

	decoded := ChannelUserBannedStatusResponse{}
	if err := json.NewDecoder(resp.Body).Decode(&decoded); err != nil {
		return ChannelBannedUser{}, false, err
	}
	return decoded.BannedUser, true, nil
}
