package tmi

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"strconv"
	"time"

	"code.justin.tv/foundation/twitchclient"

	"golang.org/x/net/context"
)

type GlobalBannedWordsMessage struct {
	GlobalBannedWords []GlobalBannedWord `json:"global_banned_words"`
}

type ChannelBannedWordsMessage struct {
	ChannelBannedWords []ChannelBannedWord `json:"channel_banned_words"`
}

type ChannelPermittedWordsMessage struct {
	ChannelPermittedWords []ChannelPermittedWord `json:"channel_permitted_words"`
}

type Phrases []string

type GlobalBannedWord struct {
	Word      string `json:"word"`
	CanOptOut bool   `json:"can_optout"`
}

type ChannelModeratedWord struct {
	Category  string     `json:"-"`
	Phrases   []string   `json:"phrases"`
	CreatedAt *time.Time `json:"created_at"`
	UpdatedAt *time.Time `json:"updated_at"`
	ExpiresAt *time.Time `json:"expires_at"`
	ExpiresIn *string    `json:"expires_in"`
	Source    *string    `json:"-"`
}

type ChannelBannedWord ChannelModeratedWord

type ChannelPermittedWord ChannelModeratedWord

func (c *clientImpl) GlobalBannedWords(ctx context.Context, reqOpts *twitchclient.ReqOpts) (*GlobalBannedWordsMessage, error) {
	path := "/global/banned_words"
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.clue.global_banned_words",
		StatSampleRate: defaultStatSampleRate,
	})

	var decoded GlobalBannedWordsMessage
	if _, err := c.DoJSON(ctx, &decoded, req, combinedReqOpts); err != nil {
		return nil, err
	}
	return &decoded, nil
}

func (c *clientImpl) ChannelBannedWords(ctx context.Context, channelID int, reqOpts *twitchclient.ReqOpts) (*ChannelBannedWordsMessage, error) {
	path := fmt.Sprintf("/rooms/%s/banned_words", url.QueryEscape(strconv.Itoa(channelID)))
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.tmi.get_channel_banned_words",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()
	if resp.StatusCode != http.StatusOK {
		return nil, httpErrorImpl{
			statusCode: resp.StatusCode,
		}
	}

	decoded := &ChannelBannedWordsMessage{}
	if err := json.NewDecoder(resp.Body).Decode(decoded); err != nil {
		return nil, err
	}
	return decoded, nil
}

func (c *clientImpl) SetChannelBannedWords(ctx context.Context, channelID int, msg ChannelBannedWordsMessage, reqOpts *twitchclient.ReqOpts) error {
	path := fmt.Sprintf("/rooms/%s/banned_words", url.QueryEscape(strconv.Itoa(channelID)))
	bodyBytes, err := json.Marshal(msg)
	if err != nil {
		return err
	}
	req, err := c.NewRequest("PUT", path, bytes.NewReader(bodyBytes))
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.clue.set_channel_banned_words",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer resp.Body.Close()

	if resp.StatusCode != http.StatusOK {
		return twitchclient.HandleFailedResponse(resp)
	}
	return nil
}

func (c *clientImpl) ChannelPermittedWords(ctx context.Context, channelID int, reqOpts *twitchclient.ReqOpts) (*ChannelPermittedWordsMessage, error) {
	path := fmt.Sprintf("/rooms/%s/permitted_words", url.QueryEscape(strconv.Itoa(channelID)))
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.tmi.get_channel_permitted_words",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()
	if resp.StatusCode != http.StatusOK {
		return nil, httpErrorImpl{
			statusCode: resp.StatusCode,
		}
	}

	decoded := &ChannelPermittedWordsMessage{}
	if err := json.NewDecoder(resp.Body).Decode(decoded); err != nil {
		return nil, err
	}
	return decoded, nil
}

func (c *clientImpl) SetChannelPermittedWords(ctx context.Context, channelID int, msg ChannelPermittedWordsMessage, reqOpts *twitchclient.ReqOpts) error {
	path := fmt.Sprintf("/rooms/%s/permitted_words", url.QueryEscape(strconv.Itoa(channelID)))
	bodyBytes, err := json.Marshal(msg)
	if err != nil {
		return err
	}
	req, err := c.NewRequest("PUT", path, bytes.NewReader(bodyBytes))
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.clue.set_channel_permitted_words",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer resp.Body.Close()

	if resp.StatusCode != http.StatusOK {
		return twitchclient.HandleFailedResponse(resp)
	}
	return nil
}
