package tmi

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"

	"golang.org/x/net/context"

	"code.justin.tv/chat/tmi/clue/api"
	"code.justin.tv/foundation/twitchclient"
)

const (
	defaultStatSampleRate = 1.0
	defaultTimingXactName = "clue"
)

var (
	// ErrEntityNotFound should be returned by Clue methods that receive an ID that
	// does not resolve to a valid entity (user, channel, etc.).
	ErrEntityNotFound = errors.New("entity not found by ID")
)

// Client is a client into TMI's Clue service.
type Client interface {
	// IsModerator returns true if the given user is a moderator of the given channel, or false otherwise.
	IsModerator(ctx context.Context, userID, channelID string, reqOpts *twitchclient.ReqOpts) (bool, error)

	// IsSubscriber returns true if the given user is a subscriber to the given channel, or false otherwise.
	IsSubscriber(ctx context.Context, userID, channelID string, reqOpts *twitchclient.ReqOpts) (bool, error)

	// GetUser returns the chat properties of a user, like chat color and turbo badge show/hide status.
	GetUser(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (*User, error)

	// GetUserEmoteSets returns a user's emote sets (for message parsing)
	GetUserEmoteSets(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (*UserEmoteSetsResponse, error)

	// GetTurboStatus returns a user's hide_turbo and is_turbo status (for badges service)
	GetTurboStatus(ctx context.Context, userID int64, reqOpts *twitchclient.ReqOpts) (*TurboStatusResponse, error)

	// GetUserType returns a user's staff, admin, or global_mod status (for badges service)
	GetUserType(ctx context.Context, userID int64, reqOpts *twitchclient.ReqOpts) (*UserTypeResponse, error)

	// IsVerifiedBot returns whether a user is a verified bot
	IsVerifiedBot(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (bool, error)

	// GetVerifiedBots returns a list of user IDs of verified bots.
	GetVerifiedBots(ctx context.Context, reqOpts *twitchclient.ReqOpts) (GetVerifiedBotsResponse, error)

	// BotProperties returns information about this user's bot status
	BotProperties(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (BotPropertiesResponse, error)

	// GetBots returns a list of user IDs of verified and known bots.
	GetBots(ctx context.Context, reqOpts *twitchclient.ReqOpts) (GetBotsResponse, error)

	// IsIgnoring returns true if the given user is ignoring another given user
	IsIgnoring(ctx context.Context, userID, targetUserID string, reqOpts *twitchclient.ReqOpts) (bool, error)

	// GetBanStatus returns the details of a ban, if it exists, and a boolean indicating whether the ban exists
	GetBanStatus(ctx context.Context, channelID, userID string, reqOpts *twitchclient.ReqOpts) (ChannelBannedUser, bool, error)

	// BanUser sets a temporary or permanent ban on a user from a channel
	BanUser(ctx context.Context, channelID, bannedUserID, modUserID string, expiresIn *string, reqOpts *twitchclient.ReqOpts) (string, error)

	// UnbanUser removes a temporary or permanent ban on a user from a channel
	UnbanUser(ctx context.Context, channelID, userID, modID string, reqOpts *twitchclient.ReqOpts) (string, error)

	// GetInternal returns a json blob of room properties given an ID of the room
	GetInternalRoomProperties(ctx context.Context, roomID string, reqOpts *twitchclient.ReqOpts) (*InternalRoomPropertiesResponse, error)

	// GetPublicRoomProperties returns a json blob of room properties given an ID of the room that public users are allowed to see
	GetPublicRoomProperties(ctx context.Context, roomID, requesterUserID string, reqOpts *twitchclient.ReqOpts) (*PublicRoomPropertiesResponse, error)

	// GetHostTargets returns the channels that a set of channels are hosting, if any.
	GetHostTargets(ctx context.Context, channelIDs []string, reqOpts *twitchclient.ReqOpts) (*HostTargetsResponse, error)

	// GetHosters returns the channels hosting target channel, if any.
	GetHosters(ctx context.Context, channelID string, reqOpts *twitchclient.ReqOpts) (*HostersResponse, error)

	// Gets the entity representing a message rejected by Automod
	GetAutoModRejectedMessage(ctx context.Context, msgID string, reqOpts *twitchclient.ReqOpts) (api.AutoModRejectedMessage, bool, error)

	// Approves a message already rejected by Automod
	ApproveAutoModRejected(ctx context.Context, msgID, requesterUserID string, reqOpts *twitchclient.ReqOpts) error

	// Denies a message already rejected by automod
	DenyAutoModRejected(ctx context.Context, msgID, requesterUserID string, reqOpts *twitchclient.ReqOpts) error

	// only a single param of target user_id or a target_channel_id can be provided,
	AutoModCheckMessage(ctx context.Context, params api.AutoModCheckMessage, reqOpts *twitchclient.ReqOpts) (api.AutoModCheckMessageResult, error)

	// GlobalBannedWords returns all global banned words, indicating whether they are able to be opt out.
	GlobalBannedWords(ctx context.Context, reqOpts *twitchclient.ReqOpts) (*GlobalBannedWordsMessage, error)

	// ChannelBannedWords returns all channel banned words (not including global banned words).
	ChannelBannedWords(ctx context.Context, channelID int, reqOpts *twitchclient.ReqOpts) (*ChannelBannedWordsMessage, error)

	// ChannelPermittedWords returns all channel permitted words.
	ChannelPermittedWords(ctx context.Context, channelID int, reqOpts *twitchclient.ReqOpts) (*ChannelPermittedWordsMessage, error)

	// SetChannelBannedWords updates the set of banned words for a channel, anything not set in here will be deleted.
	SetChannelBannedWords(ctx context.Context, channelID int, msg ChannelBannedWordsMessage, reqOpts *twitchclient.ReqOpts) error

	// SetChannelPermittedWords updates the set of permitted words for a channel, anything not set in here will be deleted.
	SetChannelPermittedWords(ctx context.Context, channelID int, msg ChannelPermittedWordsMessage, reqOpts *twitchclient.ReqOpts) error

	// SendMessage sends a chat message or command to a channel.
	// For chat messages, it returns the body and tag information of the processed and delivered message
	// For commands, it returns the output and result of the command
	// For errors, it returns an error response code and detailed error message
	SendMessage(ctx context.Context, params SendMessageParams, reqOpts *twitchclient.ReqOpts) (SendMessageResponse, error)

	// SendWhisper sends a whisper message via IRC.
	SendWhisper(ctx context.Context, params SendWhisperParams, reqOpts *twitchclient.ReqOpts) error

	// SendUserNotice sends a UserNotice system message.
	SendUserNotice(ctx context.Context, params SendUserNoticeParams, reqOpts *twitchclient.ReqOpts) error

	// TallyEmotes records a count of emotes to be announced periodically. (project mercury)
	TallyEmotes(ctx context.Context, params AddEmotesParams, reqOpts *twitchclient.ReqOpts) error
}

// NewClient creates and returns a client that can interact with Clue.
func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	c, err := twitchclient.NewClient(conf)
	if err != nil {
		return nil, err
	}
	return &clientImpl{Client: c}, nil
}

type clientImpl struct {
	twitchclient.Client
}

// IsModeratorResponse is meant only for an IsModerator response from Clue to be unmarshaled into.
type IsModeratorResponse struct {
	IsMod bool `json:"is_mod"`
}

// IsSubscriberResponse is meant only for an IsSubscriber response from Clue to be unmarshaled into.
type IsSubscriberResponse struct {
	IsSub bool `json:"is_sub"`
}

// UserResponse is meant only for a GetUser response from Clue to be unmarshaled into.
type UserResponse struct {
	User EmbeddedUserResponse `json:"user"`
}

// EmbeddedUserResponse is meant only to assist with UserResponse when unmarshaling a Clue GetUser call.
type EmbeddedUserResponse struct {
	ID             int    `json:"user_id"`
	HideTurboBadge bool   `json:"hide_turbo_badge"`
	Color          string `json:"chat_color"`
}

// User holds the chat properties of a user, such as chat color and turbo badge show/hide status.
type User struct {
	ID             int
	HideTurboBadge bool
	Color          string
}

// TurboStatusResponse is meant for a TurboStatus response from Clue to be unmarshaled into.
type TurboStatusResponse struct {
	IsTurbo   bool `json:"is_turbo"`
	HideTurbo bool `json:"hide_turbo"`
}

// UserTypeResponse is meant for a UserType response from Clue to be unmarshaled into.
type UserTypeResponse struct {
	IsStaff     bool `json:"is_staff"`
	IsAdmin     bool `json:"is_admin"`
	IsGlobalMod bool `json:"is_global_mod"`
}

// IsVerifiedBotResponse is meant only for an IsVerifiedBot response from Clue to be unmarshaled into.
type IsVerifiedBotResponse struct {
	IsVerifiedBot bool `json:"is_verified_bot"`
}

// BotPropertiesResponse is gives us back whether or not our bot is known or
type BotPropertiesResponse struct {
	IsVerifiedBot bool `json:"is_verified_bot"`
	IsKnownBot    bool `json:"is_known_bot"`
}

// IsIgnoringResponse is meant only for an IsIgnoring response from Clue to be unmarshaled into.
type IsIgnoringResponse struct {
	IsIgnoring bool `json:"is_ignoring"`
}

// IsModerator returns true if the given user is a moderator of the given channel, or false otherwise.
func (c *clientImpl) IsModerator(ctx context.Context, userID, channelID string, reqOpts *twitchclient.ReqOpts) (bool, error) {
	path := fmt.Sprintf("/rooms/%s/mods/%s", channelID, userID)
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return false, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.clue.is_moderator",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return false, err
	}
	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()
	if resp.StatusCode == http.StatusNotFound {
		return false, nil
	} else if resp.StatusCode != http.StatusOK {
		return false, httpErrorImpl{
			statusCode: resp.StatusCode,
		}
	}

	decoded := &IsModeratorResponse{}
	if err := json.NewDecoder(resp.Body).Decode(decoded); err != nil {
		return true, err
	}
	return decoded.IsMod, nil
}

// IsSubscriber returns true if the given user is a subscriber to the given channel, or false otherwise.
func (c *clientImpl) IsSubscriber(ctx context.Context, userID, channelID string, reqOpts *twitchclient.ReqOpts) (bool, error) {
	path := fmt.Sprintf("/rooms/%s/subs/%s", channelID, userID)
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return false, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.clue.is_subscriber",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return false, err
	}
	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()
	if resp.StatusCode == http.StatusNotFound {
		return false, nil
	} else if resp.StatusCode != http.StatusOK {
		return false, httpErrorImpl{
			statusCode: resp.StatusCode,
		}
	}

	decoded := &IsSubscriberResponse{}
	if err := json.NewDecoder(resp.Body).Decode(decoded); err != nil {
		return true, err
	}
	return decoded.IsSub, nil
}

// GetUser returns the chat properties of a user, like chat color and turbo badge show/hide status.
func (c *clientImpl) GetUser(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (*User, error) {
	path := fmt.Sprintf("/users/%s", userID)
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.clue.user_properties",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()
	if resp.StatusCode != http.StatusOK {
		return nil, httpErrorImpl{
			statusCode: resp.StatusCode,
		}
	}

	var decoded UserResponse
	if err := json.NewDecoder(resp.Body).Decode(&decoded); err != nil {
		return nil, err
	}
	return &User{
		ID:             decoded.User.ID,
		HideTurboBadge: decoded.User.HideTurboBadge,
		Color:          decoded.User.Color,
	}, nil
}

// GetTurboStatus fetches a user's is_turbo and hide_turbo
func (c *clientImpl) GetTurboStatus(ctx context.Context, userID int64, reqOpts *twitchclient.ReqOpts) (*TurboStatusResponse, error) {
	path := fmt.Sprintf("/users/%d/turbo_status", int(userID))
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.tmi.get_turbo_status",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()
	if resp.StatusCode != http.StatusOK {
		return nil, httpErrorImpl{
			statusCode: resp.StatusCode,
		}
	}

	decoded := &TurboStatusResponse{}
	if err := json.NewDecoder(resp.Body).Decode(decoded); err != nil {
		return nil, err
	}
	return decoded, nil
}

// GetUserType returns a user's staff, admin, or global_mod status
func (c *clientImpl) GetUserType(ctx context.Context, userID int64, reqOpts *twitchclient.ReqOpts) (*UserTypeResponse, error) {
	path := fmt.Sprintf("/users/%d/user_type", int(userID))
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.tmi.get_user_type",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()
	if resp.StatusCode != http.StatusOK {
		return nil, httpErrorImpl{
			statusCode: resp.StatusCode,
		}
	}

	decoded := &UserTypeResponse{}
	if err := json.NewDecoder(resp.Body).Decode(decoded); err != nil {
		return nil, err
	}
	return decoded, nil
}

// IsVerifiedBot returns true if the user is a verified bot.
func (c *clientImpl) IsVerifiedBot(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (bool, error) {
	path := fmt.Sprintf("/users/%s/bot_status", userID)
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return false, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.clue.is_verified_bot",
		StatSampleRate: defaultStatSampleRate,
	})
	var decoded IsVerifiedBotResponse
	if _, err := c.DoJSON(ctx, &decoded, req, combinedReqOpts); err != nil {
		return false, err
	}
	return decoded.IsVerifiedBot, nil
}

// BotProperties returns status information about what sort of bot this user is (if it is any)
func (c *clientImpl) BotProperties(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (BotPropertiesResponse, error) {
	path := fmt.Sprintf("/v2/users/%s/bot_status", userID)
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return BotPropertiesResponse{}, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.clue.is_bot_v2",
		StatSampleRate: defaultStatSampleRate,
	})
	var decoded BotPropertiesResponse
	if _, err := c.DoJSON(ctx, &decoded, req, combinedReqOpts); err != nil {
		return BotPropertiesResponse{}, err
	}
	return decoded, nil
}

// IsIgnoring returns true if the given user is ignoring another given user.
// Returns ErrEntityNotFound if user or targer user is not found.
func (c *clientImpl) IsIgnoring(ctx context.Context, userID, targetUserID string, reqOpts *twitchclient.ReqOpts) (bool, error) {
	path := fmt.Sprintf("/users/%s/ignoring/%s", userID, targetUserID)
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return false, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.clue.is_ignoring",
		StatSampleRate: defaultStatSampleRate,
	})
	var decoded IsIgnoringResponse
	if _, err := c.DoJSON(ctx, &decoded, req, combinedReqOpts); err != nil {
		if twitchclientErr, ok := err.(*twitchclient.Error); ok {
			if twitchclientErr.StatusCode == http.StatusNotFound {
				return false, ErrEntityNotFound
			}
		}
		return false, err
	}
	return decoded.IsIgnoring, nil
}

type InternalRoomPropertiesResponse struct {
	Room *InternalRoomProperties `json:"room"`
}

type InternalRoomProperties struct {
	ChannelId                  int      `json:"id"`
	FacebookConnectModerated   bool     `json:"facebook_connect_moderated"`
	GlobalBannedWordsOptout    bool     `json:"global_banned_words_optout"`
	R9kOnlyChat                bool     `json:"r9k_only_chat"`
	ChatFastsubs               bool     `json:"chat_fastsubs"`
	ChatRequireVerifiedAccount bool     `json:"chat_require_verified_account"`
	SubscribersOnlyChat        bool     `json:"subscribers_only_chat"`
	HideChatLinks              bool     `json:"hide_chat_links"`
	Cluster                    string   `json:"cluster"`
	BroadcasterLanguageMode    bool     `json:"broadcaster_language_enabled"`
	ChatDelayDuration          int      `json:"chat_delay_duration"`
	TwitchBotRuleID            int      `json:"twitchbot_rule_id"`
	AutoModRuleID              int      `json:"automod_rule_id"`
	ChatRules                  []string `json:"chat_rules"`
	FollowersOnlyDuration      int      `json:"followers_only_duration"`
	ChatEmoteOnly              bool     `json:"chat_emote_only"`
}

// GetInternalRoomProperties returns a json blob of room properties given an ID of the room
func (c *clientImpl) GetInternalRoomProperties(ctx context.Context, roomID string, reqOpts *twitchclient.ReqOpts) (*InternalRoomPropertiesResponse, error) {
	path := fmt.Sprintf("/rooms/%s", roomID)
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.clue.get_internal_room_properties",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()

	if resp.StatusCode == http.StatusNotFound {
		return nil, ErrChannelNotFound
	}

	decoded := &InternalRoomPropertiesResponse{}
	if err := json.NewDecoder(resp.Body).Decode(decoded); err != nil {
		return nil, err
	}
	return decoded, nil
}

type PublicRoomPropertiesResponse struct {
	ChannelId                  int      `json:"id"`
	FacebookConnectModerated   bool     `json:"facebook_connect_moderated"`
	GlobalBannedWordsOptout    bool     `json:"global_banned_words_optout"`
	R9kOnlyChat                bool     `json:"r9k_only_chat"`
	ChatFastsubs               bool     `json:"chat_fastsubs"`
	ChatRequireVerifiedAccount bool     `json:"chat_require_verified_account"`
	SubscribersOnlyChat        bool     `json:"subscribers_only_chat"`
	HideChatLinks              bool     `json:"hide_chat_links"`
	BroadcasterLanguageMode    bool     `json:"broadcaster_language_enabled"`
	ChatDelayDuration          int      `json:"chat_delay_duration"`
	TwitchBotRuleID            int      `json:"twitchbot_rule_id"`
	AutoModRuleID              int      `json:"automod_rule_id"`
	ChatRules                  []string `json:"chat_rules"`
}

// GetExternalRoomProperties returns a json blob of room properties given an ID of the room with fields public users can see
func (c *clientImpl) GetPublicRoomProperties(ctx context.Context, roomID, requesterUserID string, reqOpts *twitchclient.ReqOpts) (*PublicRoomPropertiesResponse, error) {
	path := fmt.Sprintf("/rooms/%s?requester_user_id=%s", roomID, requesterUserID)
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.clue.get_public_room_properties",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()

	decoded := &PublicRoomPropertiesResponse{}
	if err := json.NewDecoder(resp.Body).Decode(decoded); err != nil {
		return nil, err
	}
	return decoded, nil
}

var ErrChannelNotFound = errors.New("channel not found")

func (c *clientImpl) SendMessage(ctx context.Context, params SendMessageParams, reqOpts *twitchclient.ReqOpts) (SendMessageResponse, error) {
	url := "/internal/send_message"
	bodyBytes, err := json.Marshal(params)
	if err != nil {
		return SendMessageResponse{}, err
	}
	req, err := c.NewRequest("POST", url, bytes.NewReader(bodyBytes))
	if err != nil {
		return SendMessageResponse{}, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.clue.send_message",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return SendMessageResponse{}, err
	}
	defer resp.Body.Close()

	if resp.StatusCode == http.StatusNotFound {
		return SendMessageResponse{}, ErrChannelNotFound
	} else if resp.StatusCode != http.StatusOK {
		return SendMessageResponse{}, fmt.Errorf("unexpected response code %d during call to SendMessage", resp.StatusCode)
	}

	var decoded SendMessageResponse
	if err := json.NewDecoder(resp.Body).Decode(&decoded); err != nil {
		return SendMessageResponse{}, err
	}
	return decoded, nil
}

func (c *clientImpl) SendUserNotice(ctx context.Context, params SendUserNoticeParams, reqOpts *twitchclient.ReqOpts) error {
	url := "/internal/usernotice"
	bodyBytes, err := json.Marshal(params)
	if err != nil {
		return err
	}
	req, err := c.NewRequest("POST", url, bytes.NewReader(bodyBytes))
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.clue.send_usernotice",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer resp.Body.Close()

	if resp.StatusCode != http.StatusOK {
		return fmt.Errorf("unexpected response code %d during call to SendUserNotice", resp.StatusCode)
	}
	return nil
}

func (c *clientImpl) TallyEmotes(ctx context.Context, params AddEmotesParams, reqOpts *twitchclient.ReqOpts) error {
	url := "/internal/tally_emotes"
	bodyBytes, err := json.Marshal(params)
	if err != nil {
		return err
	}
	req, err := c.NewRequest("POST", url, bytes.NewReader(bodyBytes))
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.clue.tally_emotes",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer resp.Body.Close()

	if resp.StatusCode == http.StatusBadRequest {
		return httpErrorImpl{
			statusCode: resp.StatusCode,
		}
	} else if resp.StatusCode != http.StatusOK {
		return fmt.Errorf("unexpected response code %d during call to TallyEmotes", resp.StatusCode)
	}
	return nil
}
