package main

import (
	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"net"
	"net/http"
	"net/url"
	"os"
	"runtime"
	"strconv"
	"time"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/zenazn/goji/graceful"
	"golang.org/x/net/context"

	"code.justin.tv/chat/emoticons"
	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/chat/jsonconf"
	"code.justin.tv/chat/tmi/clients"
	"code.justin.tv/chat/tmi/clients/tracking"
	"code.justin.tv/chat/tmi/clue/config"
	"code.justin.tv/chat/tmi/clue/data"
	"code.justin.tv/chat/tmi/clue/logic"
	"code.justin.tv/chat/tmi/clue/models"
	"code.justin.tv/chat/tmi/clue/server"
	"code.justin.tv/common/chitin"
	"code.justin.tv/common/meh-restart/restart"
	"code.justin.tv/common/spade-client-go/spade"
	"code.justin.tv/foundation/xray"
)

const (
	defaultStatSampleRate = 0.1
	yearTTL               = 24 * time.Hour * 365
)

func configureStats(ctx context.Context, conf config.StatsConfig) (stats statsd.Statter) {
	log.Printf("Configuring stats client with config: %v", conf)

	addr := net.JoinHostPort(conf.Host, strconv.Itoa(conf.Port))
	s, err := statsd.NewBufferedClient(addr, conf.Prefix, 1*time.Second, 512)
	if err != nil {
		logx.Error(ctx, err, logx.Fields{
			"note": "creating buffered client",
		})
		s, err = statsd.NewNoopClient()
		if err != nil {
			logx.Error(ctx, err, logx.Fields{
				"note": "creating buffered noop client",
			})
		}
	}
	return s

}

func configureTracking(spadeHost string, mixpanelToken string, statsHook func(name string, httpStatus int, dur time.Duration)) (tracking.Client, error) {
	initOpts := []tracking.InitFunc{
		tracking.InitMixpanel(tracking.InitMixpanelToken(mixpanelToken)),
		tracking.InitMaxConcurrency(1000),
	}

	if spadeHost != "" {
		initOpts = append(initOpts, tracking.InitSpade(
			spade.InitBaseURL(url.URL{
				Scheme: "https",
				Host:   spadeHost,
			}),
			spade.InitStatHook(statsHook),
		))
	}

	return tracking.NewClient(initOpts...)
}

func main() {
	var err error
	ctx := context.Background()

	validateConf := flag.Bool("validate_conf", false, "Only check validity of Clue configuration file")
	confFile := flag.String("conf", "", "Path to Clue configuration file")
	flag.Parse()

	if *confFile == "" {
		logx.Fatal(ctx, "missing 'conf' argument")
	}

	var conf config.ClueConfig
	if err := jsonconf.ReadFile(&conf, *confFile); err != nil {
		logx.Fatal(ctx, err, logx.Fields{"note": "parsing conf"})
	}

	if *validateConf {
		os.Exit(0)
	}

	logx.InitDefaultLogger(logx.Config{
		RollbarToken: conf.Rollbar.Token,
		RollbarEnv:   conf.Rollbar.Environment,
	})

	defer logx.RecoverAndLog()

	stats := configureStats(ctx, conf.Stats)
	ds, err := data.NewDataSource(conf, stats)
	if err != nil {
		logx.Fatal(ctx, err, logx.Fields{"note": "initializing data sources"})
	}

	tracking, err := configureTracking(conf.SpadeHost, conf.MixpanelToken, func(name string, httpStatus int, dur time.Duration) {
		stats.TimingDuration(fmt.Sprintf("spade.%s.%d", name, httpStatus), dur, defaultStatSampleRate)
	})
	if err != nil {
		logx.Fatal(ctx, err, logx.Fields{"note": "initializing tracking"})
	}

	// Hystrix
	// TODO: hard-code hystrix configs in hystrix.go instead of in conf
	for cmd, cmdConf := range conf.Hystrix.Commands {
		// Some hystrix configs are hard-coded and also defined in the JSON conf.
		// Ignore the JSON conf ones if there's a conflict.
		if server.HasHystrixCommand(cmd) {
			continue
		}
		hystrix.ConfigureCommand(cmd, cmdConf)
	}
	statsHost := net.JoinHostPort(conf.Stats.Host, strconv.Itoa(conf.Stats.Port))
	if err := server.InitHystrix(statsHost, conf.Stats.Prefix); err != nil {
		logx.Fatal(ctx, err, logx.Fields{"note": "initializing hystrix"})
	}
	
	// Configure xray
	if err := xray.Configure(xray.Config{Name: "tmi", Sampling: -1}) ; err != nil {
		log.Fatal(err)
	}

	hystrixStreamHandler := hystrix.NewStreamHandler()
	hystrixStreamHandler.Start()
	hystrixPort := strconv.Itoa(conf.Hystrix.MetricsPort)
	log.Printf("starting hystrix metrics listener on port %v", hystrixPort)
	go http.ListenAndServe(net.JoinHostPort("", hystrixPort), hystrixStreamHandler)

	// Debug server
	debugAddr := net.JoinHostPort(conf.DebugHost, strconv.Itoa(conf.DebugPort))
	debugSrv := server.NewDebugSrv(debugAddr)
	go func() {
		log.Println("Starting debug server on", debugAddr)
		err := debugSrv.ListenAndServe()
		if err != nil {
			log.Println("Unable to start debug server:", err)
		}
	}()
	go memoryStatsRecorder(stats)

	ctx, err = chitin.ExperimentalTraceContext(ctx)
	if err != nil {
		logx.Fatal(ctx, err, logx.Fields{"note": "initializing chitin"})
	}
	mux := server.NewMux(ctx, conf, stats, ds)
	mux.Logic = initLogic(ctx, stats, tracking, ds, conf)

	addr := net.JoinHostPort(conf.Host, strconv.Itoa(conf.Port))
	log.Printf("Listening on %s", addr)
	listener, err := net.Listen("tcp", addr)
	if err != nil {
		logx.Fatal(ctx, err, logx.Fields{"note": "listening to addr" + addr})
	}

	restart.OnShutdown(func() {
		graceful.Shutdown()
	})
	err = graceful.Serve(listener, mux)
	if err != nil {
		logx.Fatal(ctx, err, logx.Fields{"note": "running HTTP server"})
	}
	graceful.Wait()
}

// record GC pause times periodically
func memoryStatsRecorder(stats statsd.Statter) {
	var lastPauseTime uint64
	ticker := time.NewTicker(time.Second * 30)
	defer ticker.Stop()
	for range ticker.C {
		memStats := &runtime.MemStats{}
		runtime.ReadMemStats(memStats)
		stats.TimingDuration("process.garbage_collect_duration", time.Duration(memStats.PauseTotalNs-uint64(lastPauseTime)), defaultStatSampleRate)
		lastPauseTime = memStats.PauseTotalNs

	}
}

func createWhispersLogger(ctx context.Context, path string) *log.Logger {
	f, err := os.OpenFile(path, os.O_WRONLY|os.O_CREATE|os.O_APPEND, 0644)
	if err != nil {
		logx.Warn(ctx, err, logx.Fields{"note": "opening whispers log file"})
		return log.New(ioutil.Discard, "", log.LstdFlags)
	}

	return log.New(f, "", log.LstdFlags)
}

func initLogic(ctx context.Context, stats statsd.Statter, tracking tracking.Client, ds *data.DataSource, conf config.ClueConfig) logic.ClueLogic {
	pub := clients.NewPubClient(clients.PubConfig{
		Host:           conf.Pub.Host,
		RequestTimeout: time.Duration(conf.Pub.RequestTimeout),
		StatsPrefix:    "pub",
	}, stats)

	fetcher := emoticons.NewFetcherFromRails(ds.Rails)
	emoticonsParser, err := emoticons.NewParserClient(ctx, fetcher, 18*time.Minute, stats)
	if err != nil {
		logx.Fatal(ctx, errx.Wrap(err, "instantiating emoticons parser client"))
	}
	emoticonsParser.InitPeriodicLexiconUpdates(ctx)

	var whispersLogger *log.Logger
	if !conf.WhispersDisabled {
		whispersLogger = createWhispersLogger(ctx, conf.WhispersLogPath)
	}

	l := &logic.ClueLogicImpl{
		Config:            conf,
		Stats:             stats,
		Tracking:          tracking,
		RateTracker:       models.NewRateTracker(ds),
		ChatProperties:    models.NewChatProperties(ds),
		MessageProperties: models.NewMessageProperties(ds),
		IPProperties:      models.NewIPProperties(ds),
		BatchUserData:     models.NewBatchUserData(ds),
		Repository:        logic.NewRepository(models.NewDataRepository(ds)),
		EmoticonsParser:   emoticonsParser,
		WhispersLogger:    whispersLogger,
		Pub:               pub,
		Pubsub:            ds.Pubsub,
		IMStore:           ds.IMStore,
		Badges:            ds.Badges,
		Zuma:              ds.Zuma,
		Redis:             ds.Redis,
		MercuryRedis:      ds.MercuryRedis,
		UsersService:      ds.UsersService,
		Pushy:             ds.Pushy,
		Autohost:          ds.Autohost,
	}
	l.Init(ctx)

	if err := l.InitCurseIPLoader(); err != nil {
		logx.Error(ctx, err, logx.Fields{"note": "initializing curse IP loader"})
	}

	return l
}
