package api

import (
	"errors"
	"time"

	"code.justin.tv/feeds/feeds-common/entity"
)

const (
	// Enum values for container property "require_review":

	// RequireReviewNone indicates that messages created for this container are
	// immediately published.
	RequireReviewNone = "none"
	// RequireReviewAll indicates that messages created for this container are
	// intially pending review.
	RequireReviewAll = "all"

	// Enum values for message change property "action_type":
	// ActionTypeEdit is the message change action when a message is edited.
	ActionTypeEdit = "edit"
	// ActionTypeDelete is the message change action when a message is deleted.
	ActionTypeDelete = "delete"
)

var (
	ErrLikelySpam                 = errors.New("The message is determined to likely be spam")
	ErrContainsBannedWord         = errors.New("The message contains one or more banned words")
	ErrSenderIsSuspendedOrDeleted = errors.New("The message sender is suspended or deleted")
	ErrSenderHasUnverifiedEmail   = errors.New("The message sender does not have a verified email")
	ErrSenderIsChannelBanned      = errors.New("The message sender is banned in the container owner's channel")
	ErrSenderIsIgnored            = errors.New("The message sender is ignored by the container owner")
)

// Databag is a map that contains arbitrary keys and values defined by
// the feature service. It is intended to be used as metadata storage and it's
// properties should not affect business logic within the messaging platform.
type Databag map[string]interface{}

// MessageRisk contains metadata around the likelihood the message is intended
// to be abusive or spammy.
type MessageRisk struct {
	// SpamLikelihood is the likelihood the message is spam.
	SpamLikelihood float64 `json:"spam_likelihood"`
	// IsAutomodClean indicates whether automod has flagged the message as clean.
	IsAutomodClean bool `json:"is_automod_clean"`
	// RuleResponses gives the individual responses from each rule
	RuleResponses map[string]string `json:"rule_responses"`
	// OverallResponse gives the aggregate response of all the rules
	OverallResponse string `json:"overall_response"`
}

// MessageContent contains metadata around the message text.
type MessageContent struct {
	Text      string                   `json:"text"`
	Emoticons []MessageContentEmoticon `json:"emoticons"`
}

// MessageContentEmoticon represents an emoticon found within message text.
// Ranges are 0-indexed and inclusive: [start,end]
type MessageContentEmoticon struct {
	ID    string `json:"id"`
	SetID string `json:"set_id"`
	Start int    `json:"start"`
	End   int    `json:"end"`
}

// MessageSender contains metadata around the user that sends a message.
type MessageSender struct {
	UserID      string      `json:"user_id"`
	Login       string      `json:"login"`
	DisplayName string      `json:"display_name"`
	ChatColor   string      `json:"chat_color"`
	Badges      []UserBadge `json:"badges"`
}

// UserBadge represents a chat badge for a user.
type UserBadge struct {
	ID      string `json:"id"`
	Version string `json:"version"`
}

// Message defines a user-generated message object.
type Message struct {
	ID              string         `json:"id"`
	ContainerEntity entity.Entity  `json:"container_id"`
	Content         MessageContent `json:"content"`
	Sender          MessageSender  `json:"sender"`

	// SentAt is the time the message was originally received by the server.
	SentAt time.Time `json:"sent_at"`
	// EditedAt is the time the message was last edited.
	// Nil if the message was never edited.
	EditedAt *time.Time `json:"edited_at,omitempty"`
	// DeletedAt is the time the message was deleted.
	// Nil if the message is not deleted.
	DeletedAt *time.Time `json:"deleted_at,omitempty"`
	// IsPublished indicates whether a message is publicly viewable.
	IsPublished bool `json:"is_published"`
	// IsPendingReview indicates whether a message is awaiting moderation review.
	IsPendingReview bool `json:"is_pending_review"`
	// Databag is arbitrary metadata defined by the feature service.
	Databag Databag `json:"databag,omitempty"`
}

// Container defines a parent object of one or more messages.
type Container struct {
	ContainerEntity entity.Entity `json:"container_id"`

	// Owner is the object (likely a User) that owns this container. The owner
	// is relevant for applying chat rules like banned words.
	Owner *entity.Entity `json:"owner_id,omitempty"`
	// RequireReview is an enum that represents whether messages in this container
	// should be published by default.
	RequireReview string `json:"require_review"`
	// MemberCount is the number of members in this container.
	MemberCount int `json:"member_count"`
	// LastMessage is the last message sent to this container. Nil if no message
	// has been sent or the last message is set on the container view object.
	LastMessage *Message `json:"last_message,omitempty"`
	// Databag is arbitrary metadata defined by the feature service.
	Databag Databag `json:"databag,omitempty"`
}

// ContainerView defines user-specific properties of a container.
type ContainerView struct {
	UserID string `json:"user_id"`

	// IsArchived indicates whether this user wants to hide this container
	// until a new message appears.
	IsArchived bool `json:"is_archived"`
	// IsMuted indicates whether this user wants to ignore all notifications for
	// new messages to this container.
	IsMuted bool `json:"is_muted"`
	// LastReadAt is the time thie user has last acknowledged this container.
	// This time may or may not line up with a message SentAt time.
	LastReadAt time.Time `json:"last_read_at"`
	// UserSendCount is the number of times this user has sent a message to this
	// container.
	UserSendCount int `json:"user_send_count"`
	// UserLastSentAt is the time this user has last sent a message to this
	// container.
	UserLastSentAt time.Time `json:"user_last_sent_at"`
	// LastMessage is the last message sent to this container. Nil if no message
	// has been sent or the last message is set on the container object.
	LastMessage *Message `json:"last_message,omitempty"`
	// JoinedAt is the time this user joined this container.
	JoinedAt time.Time `json:"joined_at"`
	// HasUnreadMessages indicates whether this container has messages this user
	// has not read.
	HasUnreadMessages bool `json:"has_unread_messages"`
	// UnreadNoticeCount is the number of messages that require the user's attention.
	UnreadNoticeCount int `json:"unread_notice_count"`
}

// MemberUser is a user that is a member of a container.
type MemberUser struct {
	UserID string `json:"user_id"`

	// JoinedAt is the time the user last joined this container.
	JoinedAt time.Time `json:"joined_at"`
}
