package app

import (
	"net/http"

	"github.com/cactus/go-statsd-client/statsd"
	goji "goji.io"
	"goji.io/pat"
	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/golibs/gojiplus/decorators"
	"code.justin.tv/chat/golibs/gojiplus/middleware"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/app/config"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/common/goauthorization"
)

// Params defines parameters to initialize the Zuma API handler
type Params struct {
	Conf          config.Config
	Stats         statsd.Statter
	Backend       backend.Backender
	Authorization goauthorization.IDecoder
}

// New returns a HTTP Handler that handles Zuma API requests
func New(p Params) (http.Handler, error) {
	mux := goji.NewMux()
	mux.Handle(pat.Get("/debug/running"), gojiplus.HealthCheck())
	mux.Use(middleware.EnhancedResponseWriter)
	mux.UseC(middleware.TwitchRepository)

	decorate := func(h goji.HandlerFunc, hp handlerParams) goji.HandlerFunc {
		decorated := decorators.Join(
			decorators.TimingFn(p.Stats, "endpoint."+hp.EndpointName),
		)(h)

		return func(ctx context.Context, w http.ResponseWriter, r *http.Request) {
			defer logx.RecoverAndLog()
			ctx = logx.WithFields(ctx, logx.Fields{
				"method": r.Method,
				"ip":     gojiplus.GetIPFromRequest(r),
				"url":    r.URL.Path,
			})
			decorated(ctx, w, r)
		}
	}

	h := &handlers{
		Conf:          p.Conf,
		Stats:         p.Stats,
		Backend:       p.Backend,
		Authorization: p.Authorization,
	}

	modsV1 := goji.SubMux()
	mux.HandleC(pat.New("/v1/mods/*"), modsV1)
	modsV1.HandleFuncC(pat.Post("/get"), decorate(h.GetMod, handlerParams{
		EndpointName: "get_mod",
	}))
	modsV1.HandleFuncC(pat.Post("/list"), decorate(h.ListMods, handlerParams{
		EndpointName: "list_mods",
	}))
	modsV1.HandleFuncC(pat.Post("/add"), decorate(h.AddMod, handlerParams{
		EndpointName: "add_mod",
	}))
	modsV1.HandleFuncC(pat.Post("/remove"), decorate(h.RemoveMod, handlerParams{
		EndpointName: "remove_mod",
	}))

	// communities
	communitiesV1 := goji.SubMux()
	mux.HandleC(pat.New("/v1/communities/*"), communitiesV1)
	communitiesV1.HandleFuncC(pat.Post("/create"), decorate(h.CreateCommunity, handlerParams{
		EndpointName: "create_community",
	}))
	communitiesV1.HandleFuncC(pat.Post("/get_by_name"), decorate(h.GetCommunityByName, handlerParams{
		EndpointName: "get_community_by_name",
	}))

	// top communities
	communitiesV1.HandleFuncC(pat.Post("/top"), decorate(h.TopCommunities, handlerParams{
		EndpointName: "top_communities",
	}))

	// featured communities
	communitiesV1.HandleFuncC(pat.Post("/featured/list"), decorate(h.ListFeaturedCommunities, handlerParams{
		EndpointName: "list_featured_communities",
	}))
	communitiesV1.HandleFuncC(pat.Post("/featured/add"), decorate(h.AddFeaturedCommunity, handlerParams{
		EndpointName: "add_featured_community",
	}))
	communitiesV1.HandleFuncC(pat.Post("/featured/remove"), decorate(h.RemoveFeaturedCommunity, handlerParams{
		EndpointName: "remove_featured_community",
	}))

	// reporting and banning communities
	communitiesV1.HandleFuncC(pat.Post("/report"), decorate(h.ReportCommunity, handlerParams{
		EndpointName: "report_community",
	}))
	communitiesV1.HandleFuncC(pat.Post("/tos_ban"), decorate(h.TOSBanCommunity, handlerParams{
		EndpointName: "tos_ban_community",
	}))

	// settings
	communitiesV1.HandleFuncC(pat.Post("/settings/get"), decorate(h.GetCommunitySettings, handlerParams{
		EndpointName: "get_community_settings",
	}))
	communitiesV1.HandleFuncC(pat.Post("/settings/internal_get"), decorate(h.InternalGetCommunitySettings, handlerParams{
		EndpointName: "internal_get_community_settings",
	}))
	communitiesV1.HandleFuncC(pat.Post("/settings/set"), decorate(h.SetCommunitySettings, handlerParams{
		EndpointName: "set_community_settings",
	}))

	// images
	communitiesV1.HandleFuncC(pat.Post("/images/upload"), decorate(h.UploadCommunityImages, handlerParams{
		EndpointName: "upload_community_images",
	}))
	communitiesV1.HandleFuncC(pat.Post("/images/remove"), decorate(h.RemoveCommunityImages, handlerParams{
		EndpointName: "remove_community_images",
	}))

	// mods
	communitiesV1.HandleFuncC(pat.Post("/mods/get"), decorate(h.GetCommunityMod, handlerParams{
		EndpointName: "get_community_mod",
	}))
	communitiesV1.HandleFuncC(pat.Post("/mods/list"), decorate(h.ListCommunityMods, handlerParams{
		EndpointName: "list_community_mods",
	}))
	communitiesV1.HandleFuncC(pat.Post("/mods/add"), decorate(h.AddCommunityMod, handlerParams{
		EndpointName: "add_community_mod",
	}))
	communitiesV1.HandleFuncC(pat.Post("/mods/remove"), decorate(h.RemoveCommunityMod, handlerParams{
		EndpointName: "remove_community_mod",
	}))

	// bans
	communitiesV1.HandleFuncC(pat.Post("/bans/get"), decorate(h.GetCommunityBan, handlerParams{
		EndpointName: "get_community_ban",
	}))
	communitiesV1.HandleFuncC(pat.Post("/bans/list"), decorate(h.ListCommunityBans, handlerParams{
		EndpointName: "list_community_bans",
	}))
	communitiesV1.HandleFuncC(pat.Post("/bans/add"), decorate(h.AddCommunityBan, handlerParams{
		EndpointName: "add_community_ban",
	}))
	communitiesV1.HandleFuncC(pat.Post("/bans/remove"), decorate(h.RemoveCommunityBan, handlerParams{
		EndpointName: "remove_community_ban",
	}))

	// timeouts
	communitiesV1.HandleFuncC(pat.Post("/timeouts/get"), decorate(h.GetCommunityTimeout, handlerParams{
		EndpointName: "get_community_timeout",
	}))
	communitiesV1.HandleFuncC(pat.Post("/timeouts/list"), decorate(h.ListCommunityTimeouts, handlerParams{
		EndpointName: "list_community_timeouts",
	}))
	communitiesV1.HandleFuncC(pat.Post("/timeouts/add"), decorate(h.AddCommunityTimeout, handlerParams{
		EndpointName: "add_community_timeout",
	}))
	communitiesV1.HandleFuncC(pat.Post("/timeouts/remove"), decorate(h.RemoveCommunityTimeout, handlerParams{
		EndpointName: "remove_community_timeout",
	}))

	// moderation logs
	communitiesV1.HandleFuncC(pat.Post("/moderation_logs/list"), decorate(h.GetCommunityModerationActions, handlerParams{
		EndpointName: "get_moderation_logs",
	}))

	// user permissions
	communitiesV1.HandleFuncC(pat.Post("/permissions"), decorate(h.GetCommunityPermissions, handlerParams{
		EndpointName: "get_community_permissions",
	}))

	// reserved names
	communitiesV1.HandleFuncC(pat.Post("/reserved_names/list"), decorate(h.ListCommunityReservedNames, handlerParams{
		EndpointName: "list_community_reserved_names",
	}))
	communitiesV1.HandleFuncC(pat.Post("/reserved_names/create"), decorate(h.CreateCommunityReservedName, handlerParams{
		EndpointName: "create_community_reserved_name",
	}))
	communitiesV1.HandleFuncC(pat.Post("/reserved_names/delete"), decorate(h.DeleteCommunityReservedName, handlerParams{
		EndpointName: "delete_community_reserved_name",
	}))

	// followers
	communitiesV1.HandleFuncC(pat.Post("/followers/list"), decorate(h.ListCommunityFollowers, handlerParams{
		EndpointName: "list_community_followers",
	}))
	communitiesV1.HandleFuncC(pat.Post("/followers/count"), decorate(h.CountCommunityFollowers, handlerParams{
		EndpointName: "count_community_followers",
	}))
	communitiesV1.HandleFuncC(pat.Post("/followers/get"), decorate(h.GetCommunityFollower, handlerParams{
		EndpointName: "get_community_follower",
	}))

	// channels
	channelsV1 := goji.SubMux()
	mux.HandleC(pat.New("/v1/channels/*"), channelsV1)
	channelsV1.HandleFuncC(pat.Post("/communities/set"), decorate(h.SetChannelCommunity, handlerParams{
		EndpointName: "set_channel_community",
	}))
	channelsV1.HandleFuncC(pat.Post("/communities/set_multiple"), decorate(h.SetChannelCommunities, handlerParams{
		EndpointName: "set_channel_communities",
	}))
	channelsV1.HandleFuncC(pat.Post("/communities/unset"), decorate(h.UnsetChannelCommunity, handlerParams{
		EndpointName: "unset_channel_community",
	}))
	channelsV1.HandleFuncC(pat.Post("/communities/get"), decorate(h.GetChannelCommunity, handlerParams{
		EndpointName: "get_channel_community",
	}))
	channelsV1.HandleFuncC(pat.Post("/communities/bulk_get_by_login"), decorate(h.BulkGetChannelCommunities, handlerParams{
		EndpointName: "bulk_get_channel_communities",
	}))
	channelsV1.HandleFuncC(pat.Post("/communities/report"), decorate(h.ReportChannelCommunity, handlerParams{
		EndpointName: "report_channel_community",
	}))

	// users
	usersV1 := goji.SubMux()
	mux.HandleC(pat.New("/v1/users/*"), usersV1)

	// user blocks
	usersV1.HandleFuncC(pat.Post("/blocks/get"), decorate(h.ListUserBlocks, handlerParams{
		EndpointName: "list_user_blocks",
	}))
	usersV1.HandleFuncC(pat.Post("/blocks/add"), decorate(h.AddUserBlock, handlerParams{
		EndpointName: "add_user_block",
	}))
	usersV1.HandleFuncC(pat.Post("/blocks/remove"), decorate(h.RemoveUserBlock, handlerParams{
		EndpointName: "remove_user_block",
	}))

	// user followed communities
	usersV1.HandleFuncC(pat.Post("/follows/communities/list"), decorate(h.ListUserFollowedCommunities, handlerParams{
		EndpointName: "list_user_followed_communities",
	}))
	usersV1.HandleFuncC(pat.Post("/follows/communities/top"), decorate(h.TopUserFollowedCommunities, handlerParams{
		EndpointName: "top_user_followed_communities",
	}))
	usersV1.HandleFuncC(pat.Post("/follows/communities/add"), decorate(h.AddUserFollowedCommunity, handlerParams{
		EndpointName: "remove_user_followed_community",
	}))
	usersV1.HandleFuncC(pat.Post("/follows/communities/remove"), decorate(h.RemoveUserFollowedCommunity, handlerParams{
		EndpointName: "remove_user_followed_community",
	}))

	usersV1.HandleFuncC(pat.Post("/emoticons/check_access"), decorate(h.CheckEmoticonsAccess, handlerParams{
		EndpointName: "check_emoticons_access_v1",
	}))

	// Deprecated: Use v1/maap components.
	messagesV1 := goji.SubMux()
	mux.HandleC(pat.New("/v1/messages/*"), messagesV1)
	messagesV1.HandleFuncC(pat.Post("/parse"), decorate(h.ParseMessage, handlerParams{
		EndpointName: "parse_message",
	}))

	// Messaging as a Platform (MaaP) components
	maapV1 := goji.SubMux()
	mux.HandleC(pat.New("/v1/maap/*"), maapV1)
	maapV1.HandleFuncC(pat.Post("/extraction/extract"), decorate(h.ExtractMessage, handlerParams{
		EndpointName: "extract_message_v1",
	}))

	maapV1.HandleFuncC(pat.Post("/messages/get"), decorate(h.GetMessage, handlerParams{
		EndpointName: "get_message_v1",
	}))
	maapV1.HandleFuncC(pat.Post("/messages/list"), decorate(h.ListMessages, handlerParams{
		EndpointName: "list_messages_v1",
	}))
	maapV1.HandleFuncC(pat.Post("/messages/create"), decorate(h.CreateMessage, handlerParams{
		EndpointName: "create_message_v1",
	}))
	maapV1.HandleFuncC(pat.Post("/messages/edit"), decorate(h.EditMessage, handlerParams{
		EndpointName: "edit_message_v1",
	}))
	maapV1.HandleFuncC(pat.Post("/messages/delete"), decorate(h.DeleteMessage, handlerParams{
		EndpointName: "delete_message_v1",
	}))
	maapV1.HandleFuncC(pat.Post("/messages/approve"), decorate(h.ApproveMessage, handlerParams{
		EndpointName: "approve_message_v1",
	}))
	maapV1.HandleFuncC(pat.Post("/messages/reject"), decorate(h.RejectMessage, handlerParams{
		EndpointName: "reject_message_v1",
	}))

	maapV1.HandleFuncC(pat.Post("/containers/list_by_owner"), decorate(h.ListContainersByOwner, handlerParams{
		EndpointName: "list_containers_by_owner_v1",
	}))
	maapV1.HandleFuncC(pat.Post("/containers/get"), decorate(h.GetContainer, handlerParams{
		EndpointName: "get_container_v1",
	}))
	maapV1.HandleFuncC(pat.Post("/containers/create"), decorate(h.CreateContainer, handlerParams{
		EndpointName: "create_container_v1",
	}))
	maapV1.HandleFuncC(pat.Post("/containers/update"), decorate(h.UpdateContainer, handlerParams{
		EndpointName: "update_container_v1",
	}))

	maapV1.HandleFuncC(pat.Post("/container_views/list_by_user"), decorate(h.ListContainerViewsByUser, handlerParams{
		EndpointName: "list_container_views_by_user_v1",
	}))
	maapV1.HandleFuncC(pat.Post("/container_views/get"), decorate(h.GetContainerView, handlerParams{
		EndpointName: "get_container_view_v1",
	}))
	maapV1.HandleFuncC(pat.Post("/container_views/update"), decorate(h.UpdateContainerView, handlerParams{
		EndpointName: "update_container_view_v1",
	}))

	maapV1.HandleFuncC(pat.Post("/members/list"), decorate(h.ListMembers, handlerParams{
		EndpointName: "list_members_v1",
	}))
	maapV1.HandleFuncC(pat.Post("/members/get"), decorate(h.GetMember, handlerParams{
		EndpointName: "get_member_v1",
	}))
	maapV1.HandleFuncC(pat.Post("/members/add"), decorate(h.AddMember, handlerParams{
		EndpointName: "add_member_v1",
	}))
	maapV1.HandleFuncC(pat.Post("/members/remove"), decorate(h.RemoveMember, handlerParams{
		EndpointName: "remove_member_v1",
	}))

	return mux, nil
}

// serveError writes an error status code, but also writes a detailed JSON
// error response to callers. It also doesn't log errors to stdout or rollbar,
// because the only errors it should be used for are user errors
func serveError(rw http.ResponseWriter, req *http.Request, err string, status int) {
	rw.WriteHeader(status)
	gojiplus.ServeJSON(rw, req, &api.ErrorResponse{
		Err:    err,
		Status: status,
	})
}

type handlerParams struct {
	EndpointName string
}

type handlers struct {
	Conf          config.Config
	Stats         statsd.Statter
	Backend       backend.Backender
	Authorization goauthorization.IDecoder
}
