package app

import (
	"errors"
	"net/http"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"

	"golang.org/x/net/context"
)

func (h *handlers) BulkGetChannelCommunities(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.BulkGetChannelCommunitiesRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateBulkGetChannelCommunitiesParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	usersMap, err := h.Backend.BulkGetSiteUsersFromLogins(ctx, params.ChannelLogins)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	channelsMap := map[string]string{}
	channelIDs := []string{}
	for _, user := range usersMap {
		channelIDs = append(channelIDs, user.UserID)
		channelsMap[user.UserID] = user.Login
	}

	communities, err := h.Backend.BulkGetChannelCommunities(ctx, channelIDs)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	results := []api.BulkGetChannelCommunitiesResult{}
	for channelID, communityIDs := range communities {
		if login, ok := channelsMap[channelID]; ok {
			result := api.BulkGetChannelCommunitiesResult{
				ChannelID:    channelID,
				ChannelLogin: login,
				CommunityID:  "",
				CommunityIDs: communityIDs,
			}
			if len(communityIDs) > 0 {
				result.CommunityID = communityIDs[0]
			}
			results = append(results, result)
		}
	}

	gojiplus.ServeJSON(rw, req, &api.BulkGetChannelCommunitiesResponse{
		Results: results,
	})
}

func validateBulkGetChannelCommunitiesParams(params api.BulkGetChannelCommunitiesRequest) error {
	for _, login := range params.ChannelLogins {
		if login == "" {
			return errors.New("login must not be empty")
		}
	}
	return nil
}
