package app

import (
	"net/http"
	"testing"

	c "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	"code.justin.tv/chat/golibs/testutils"
	"code.justin.tv/chat/zuma/app/config"
	"code.justin.tv/chat/zuma/backend"
)

func TestBulkGetChannelCommunities(t *testing.T) {
	c.Convey("test TopCommunities handler", t, func() {

		var (
			logins   = []string{"test", "asdf", "lol"}
			usersMap = map[string]backend.SiteUser{
				"test": backend.SiteUser{
					UserID: "111",
					Login:  "test",
				},
				"asdf": backend.SiteUser{
					UserID: "222",
					Login:  "asdf",
				},
				"lol": backend.SiteUser{
					UserID: "333",
					Login:  "lol",
				},
			}
			communities = map[string][]string{
				"111": []string{"asdfasdf"},
				"222": []string{"blahblah"},
				"333": []string{"testest"},
			}
		)

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend: backenderMock,
			Conf: config.Config{
				Environment: "production",
			},
		})

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/channels/communities/bulk_get_by_login",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"channel_logins": logins,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("with successful response for BulkGetSiteUsersFromLogins", func() {
			backenderMock.On("BulkGetSiteUsersFromLogins", mock.Anything, logins).
				Return(usersMap, nil).Once()

			c.Convey("with successful response for bulk get channel communities", func() {
				backenderMock.On("BulkGetChannelCommunities", mock.Anything, mock.Anything).
					Return(communities, nil).Once()

				apitest.ExpectedStatus = http.StatusOK
				// TODO: Enforce deterministic ordering of these results so that they can be tested
				// apitest.Expected = map[string]interface{}{
				// 	"results": []interface{}{
				// 		map[string]interface{}{
				// 			"channel_id":    "111",
				// 			"channel_login": "test",
				// 			"community_id":  "asdfasdf",
				// 		},
				// 		map[string]interface{}{
				// 			"channel_id":    "222",
				// 			"channel_login": "asdf",
				// 			"community_id":  "blahblah",
				// 		},
				// 		map[string]interface{}{
				// 			"channel_id":    "333",
				// 			"channel_login": "lol",
				// 			"community_id":  "testest",
				// 		},
				// 	},
				// }
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})
	})
}
