package app

import (
	"errors"
	"net/http"
	"strconv"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/common/goauthorization"

	"golang.org/x/net/context"
)

const unsetChannelCommunityCapability = "-"

const maxCommunitiesPerChannel = 1

func (h *handlers) SetChannelCommunity(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.SetChannelCommunityRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateSetChannelCommunityParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"manage_channel_community": goauthorization.CapabilityClaim{
			"community_id": params.CommunityID,
			"channel_id":   params.ChannelID,
		},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}

	_, exists, err = h.Backend.GetCommunityBan(ctx, params.CommunityID, params.ChannelID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if exists {
		serveError(rw, req, api.ErrCodeChannelBannedFromCommunity, http.StatusForbidden)
		return
	}

	_, exists, err = h.Backend.GetCommunityTimeout(ctx, params.CommunityID, params.ChannelID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if exists {
		serveError(rw, req, api.ErrCodeChannelTimedOutFromCommunity, http.StatusForbidden)
		return
	}

	err = h.Backend.SetChannelCommunities(ctx, params.ChannelID, []string{params.CommunityID})
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	rw.WriteHeader(http.StatusOK)
}

func validateSetChannelCommunityParams(params api.SetChannelCommunityRequest) error {
	if channelID, err := strconv.ParseInt(params.ChannelID, 10, 64); err != nil {
		return errors.New("channel_id must be integer")
	} else if channelID <= 0 {
		return errors.New("channel_id must be positive")
	}
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	return nil
}

func (h *handlers) SetChannelCommunities(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.SetChannelCommunitiesRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateSetChannelCommunitiesParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	if len(params.CommunityIDs) > maxCommunitiesPerChannel {
		serveError(rw, req, api.ErrCodeChannelTooManyCommunities, http.StatusBadRequest)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"manage_channel_community": goauthorization.CapabilityClaim{
			"community_ids": params.CommunityIDs,
			"channel_id":    params.ChannelID,
		},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}

	for _, communityID := range params.CommunityIDs {
		community, exists, err := h.Backend.GetCommunity(ctx, communityID)
		if err != nil {
			gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
			return
		} else if !exists {
			serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
			return
		}

		if community.TOSBanned {
			serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
			return
		}

		_, exists, err = h.Backend.GetCommunityBan(ctx, communityID, params.ChannelID)
		if err != nil {
			gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
			return
		} else if exists {
			serveError(rw, req, api.ErrCodeChannelBannedFromCommunity, http.StatusForbidden)
			return
		}

		_, exists, err = h.Backend.GetCommunityTimeout(ctx, communityID, params.ChannelID)
		if err != nil {
			gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
			return
		} else if exists {
			serveError(rw, req, api.ErrCodeChannelTimedOutFromCommunity, http.StatusForbidden)
			return
		}
	}

	err = h.Backend.SetChannelCommunities(ctx, params.ChannelID, params.CommunityIDs)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	rw.WriteHeader(http.StatusOK)
}

func validateSetChannelCommunitiesParams(params api.SetChannelCommunitiesRequest) error {
	if channelID, err := strconv.ParseInt(params.ChannelID, 10, 64); err != nil {
		return errors.New("channel_id must be integer")
	} else if channelID <= 0 {
		return errors.New("channel_id must be positive")
	}
	return nil
}

func (h *handlers) UnsetChannelCommunity(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.UnsetChannelCommunityRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateUnsetChannelCommunityParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"manage_channel_community": goauthorization.CapabilityClaim{
			"community_id": unsetChannelCommunityCapability,
			"channel_id":   params.ChannelID,
		},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}

	err = h.Backend.SetChannelCommunities(ctx, params.ChannelID, []string{})
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	rw.WriteHeader(http.StatusOK)
}

func validateUnsetChannelCommunityParams(params api.UnsetChannelCommunityRequest) error {
	if channelID, err := strconv.ParseInt(params.ChannelID, 10, 64); err != nil {
		return errors.New("channel_id must be integer")
	} else if channelID <= 0 {
		return errors.New("channel_id must be positive")
	}
	return nil
}

func (h *handlers) GetChannelCommunity(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.GetChannelCommunityRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateGetChannelCommunityParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	communityIDs, err := h.Backend.GetChannelCommunities(ctx, params.ChannelID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	resp := api.GetChannelCommunityResponse{
		CommunityID:  "",
		CommunityIDs: communityIDs,
	}
	if len(communityIDs) >= 1 {
		resp.CommunityID = communityIDs[0]
	}

	gojiplus.ServeJSON(rw, req, resp)
}

func validateGetChannelCommunityParams(params api.GetChannelCommunityRequest) error {
	if channelID, err := strconv.ParseInt(params.ChannelID, 10, 64); err != nil {
		return errors.New("channel_id must be integer")
	} else if channelID <= 0 {
		return errors.New("channel_id must be positive")
	}
	return nil
}

func (h *handlers) ReportChannelCommunity(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.ReportChannelCommunityRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateReportChannelCommunityRequest(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	// Temporary: until Visage starts sending CommunityID as a param
	var communityID string
	if params.CommunityID == "" {
		communityIDs, err := h.Backend.GetChannelCommunities(ctx, params.ChannelID)
		if err != nil {
			gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
			return
		} else if len(communityIDs) == 0 {
			serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
			return
		}
		communityID = communityIDs[0]
	} else {
		communityID = params.CommunityID
	}

	community, exists, err := h.Backend.GetCommunity(ctx, communityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	var userID string
	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		// No auth is required to file a report.
		userID = ""
	} else {
		userID = token.GetSubject()
	}

	err = h.Backend.PublishChannelCommunityReport(ctx, params.ChannelID, communityID, userID, params.Description)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	rw.WriteHeader(http.StatusOK)
}

func validateReportChannelCommunityRequest(params api.ReportChannelCommunityRequest) error {
	if channelID, err := strconv.ParseInt(params.ChannelID, 10, 64); err != nil {
		return errors.New("channel_id must be integer")
	} else if channelID <= 0 {
		return errors.New("channel_id must be positive")
	}
	return nil
}

// Utility functions for dealing with string slices
func contains(strs []string, str string) bool {
	for _, s := range strs {
		if s == str {
			return true
		}
	}
	return false
}

func remove(strs []string, str string) []string {
	res := []string{}
	for _, s := range strs {
		if s != str {
			res = append(res, s)
		}
	}
	return res
}
