package app

import (
	"errors"
	"net/http"
	"testing"

	c "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	"code.justin.tv/chat/golibs/testutils"
	"code.justin.tv/chat/zuma/app/api"
	mockauth "code.justin.tv/chat/zuma/app/mocks/auth"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/common/goauthorization"
	"code.justin.tv/common/jwt/claim"
)

func TestSetChannelCommunity(t *testing.T) {
	c.Convey("test SetChannelCommunity handler", t, func() {

		authMock := &mockauth.IDecoder{}
		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		channelID := "555"
		communityID := "asdfasdf"
		requester := "123"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/channels/communities/set",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"channel_id":   channelID,
				"community_id": communityID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if community exists", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, true, nil).Once()

			c.Convey("if auth token is valid", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(&goauthorization.AuthorizationToken{
						Claims: goauthorization.TokenClaims{
							Sub: claim.Sub{
								Sub: requester,
							},
						},
					}, nil)

				c.Convey("if the user has edit channel community permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"manage_channel_community": goauthorization.CapabilityClaim{
							"community_id": communityID,
							"channel_id":   channelID,
						},
					}).Return(nil).Once()

					c.Convey("if the community is not banned", func() {
						backenderMock.On("GetCommunityBan", mock.Anything, communityID, channelID).
							Return(backend.CommunityBan{}, false, nil).Once()

						c.Convey("if the community is not timed out", func() {
							backenderMock.On("GetCommunityTimeout", mock.Anything, communityID, channelID).
								Return(backend.CommunityTimeout{}, false, nil).Once()

							c.Convey("if the set channel community call succeeds", func() {
								backenderMock.On("SetChannelCommunities", mock.Anything, channelID, []string{communityID}).
									Return(nil).Once()

								apitest.ExpectedStatus = http.StatusOK
								testutils.RunTest(t, s, apitest)
								backenderMock.AssertExpectations(t)
							})

							c.Convey("if the set channel community call errors", func() {
								backenderMock.On("SetChannelCommunities", mock.Anything, channelID, []string{communityID}).
									Return(errors.New("error")).Once()

								apitest.ExpectedStatus = http.StatusInternalServerError
								testutils.RunTest(t, s, apitest)
								backenderMock.AssertExpectations(t)
							})
						})

						c.Convey("if the community is timed out", func() {
							backenderMock.On("GetCommunityTimeout", mock.Anything, communityID, channelID).
								Return(backend.CommunityTimeout{}, true, nil).Once()

							apitest.ExpectedStatus = http.StatusForbidden
							apitest.Expected = map[string]interface{}{
								"status": float64(http.StatusForbidden),
								"error":  api.ErrCodeChannelTimedOutFromCommunity,
							}
							testutils.RunTest(t, s, apitest)
							backenderMock.AssertExpectations(t)
						})

						c.Convey("if the call to check timeout errors", func() {
							backenderMock.On("GetCommunityTimeout", mock.Anything, communityID, channelID).
								Return(backend.CommunityTimeout{}, false, errors.New("error")).Once()

							apitest.ExpectedStatus = http.StatusInternalServerError
							testutils.RunTest(t, s, apitest)
							backenderMock.AssertExpectations(t)
						})
					})

					c.Convey("if the community is banned", func() {
						backenderMock.On("GetCommunityBan", mock.Anything, communityID, channelID).
							Return(backend.CommunityBan{}, true, nil).Once()

						apitest.ExpectedStatus = http.StatusForbidden
						apitest.Expected = map[string]interface{}{
							"status": float64(http.StatusForbidden),
							"error":  api.ErrCodeChannelBannedFromCommunity,
						}
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})

					c.Convey("if the call to check ban errors", func() {
						backenderMock.On("GetCommunityBan", mock.Anything, communityID, channelID).
							Return(backend.CommunityBan{}, false, errors.New("error")).Once()

						apitest.ExpectedStatus = http.StatusInternalServerError
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})
				})

				c.Convey("if the user does not have edit channel community permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"manage_channel_community": goauthorization.CapabilityClaim{
							"community_id": communityID,
							"channel_id":   channelID,
						},
					}).Return(errors.New("error")).Once()

					apitest.ExpectedStatus = http.StatusForbidden
					apitest.Expected = map[string]interface{}{
						"status": float64(http.StatusForbidden),
						"error":  api.ErrCodeRequestingUserNotPermitted,
					}
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})
			})

			c.Convey("if auth token is not valid", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(nil, errors.New("error"))

				apitest.ExpectedStatus = http.StatusUnauthorized
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusUnauthorized),
					"error":  api.ErrCodeRequestingUserNotPermitted,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if community does not exist", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if community is tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{TOSBanned: true}, true, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityTOSBanned,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with error response for get community", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestSetChannelCommunities(t *testing.T) {
	c.Convey("test SetChannelCommunities handler", t, func() {

		authMock := &mockauth.IDecoder{}
		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		communityID1 := "aaa"
		// communityID2 := "bbb"
		// communityID3 := "ccc"
		channelID := "555"
		// TODO: change this back once we start allowing multiple communities
		//communityIDs := []string{communityID1, communityID2, communityID3}
		communityIDs := []string{communityID1}
		requester := "123"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/channels/communities/set_multiple",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"channel_id":    channelID,
				"community_ids": communityIDs,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if there are too many communities listed", func() {
			apitest.BodyJSON["community_ids"] = []string{"asdf", "123"}
			apitest.ExpectedStatus = http.StatusBadRequest
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if auth token is valid", func() {
			authMock.On("ParseToken", mock.Anything).
				Return(&goauthorization.AuthorizationToken{
					Claims: goauthorization.TokenClaims{
						Sub: claim.Sub{
							Sub: requester,
						},
					},
				}, nil)

			c.Convey("if the user has edit channel community permissions", func() {
				authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
					"manage_channel_community": goauthorization.CapabilityClaim{
						"community_ids": communityIDs,
						"channel_id":    channelID,
					},
				}).Return(nil).Once()

				c.Convey("if all communities exist, and the channel isn't banned or timed out", func() {
					backenderMock.On("GetCommunity", mock.Anything, communityID1).
						Return(backend.Community{}, true, nil).Once()
					backenderMock.On("GetCommunityBan", mock.Anything, communityID1, channelID).
						Return(backend.CommunityBan{}, false, nil).Once()
					backenderMock.On("GetCommunityTimeout", mock.Anything, communityID1, channelID).
						Return(backend.CommunityTimeout{}, false, nil).Once()

					// TODO: change this back once we start allowing multiple communities
					// backenderMock.On("GetCommunity", mock.Anything, communityID2).
					// 	Return(backend.Community{}, true, nil).Once()
					// backenderMock.On("GetCommunityBan", mock.Anything, communityID2, channelID).
					// 	Return(backend.CommunityBan{}, false, nil).Once()
					// backenderMock.On("GetCommunityTimeout", mock.Anything, communityID2, channelID).
					// 	Return(backend.CommunityTimeout{}, false, nil).Once()
					//
					// backenderMock.On("GetCommunity", mock.Anything, communityID3).
					// 	Return(backend.Community{}, true, nil).Once()
					// backenderMock.On("GetCommunityBan", mock.Anything, communityID3, channelID).
					// 	Return(backend.CommunityBan{}, false, nil).Once()
					// backenderMock.On("GetCommunityTimeout", mock.Anything, communityID3, channelID).
					// 	Return(backend.CommunityTimeout{}, false, nil).Once()

					c.Convey("if the call to set channel communities succeeds", func() {
						backenderMock.On("SetChannelCommunities", mock.Anything, channelID, communityIDs).
							Return(nil).Once()

						apitest.ExpectedStatus = http.StatusOK
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})

					c.Convey("if the set channel community call errors", func() {
						backenderMock.On("SetChannelCommunities", mock.Anything, channelID, communityIDs).
							Return(errors.New("error")).Once()

						apitest.ExpectedStatus = http.StatusInternalServerError
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})
				})

				c.Convey("if the user is banned in a community", func() {

					// backenderMock.On("GetCommunity", mock.Anything, communityID1).
					// 	Return(backend.Community{}, true, nil).Once()
					// backenderMock.On("GetCommunityBan", mock.Anything, communityID1, channelID).
					// 	Return(backend.CommunityBan{}, false, nil).Once()
					// backenderMock.On("GetCommunityTimeout", mock.Anything, communityID1, channelID).
					// 	Return(backend.CommunityTimeout{}, false, nil).Once()
					//
					// backenderMock.On("GetCommunity", mock.Anything, communityID2).
					// 	Return(backend.Community{}, true, nil).Once()
					// backenderMock.On("GetCommunityBan", mock.Anything, communityID2, channelID).
					// 	Return(backend.CommunityBan{}, true, nil).Once()

					// TODO: replace below .On()s with above .On()s
					backenderMock.On("GetCommunity", mock.Anything, communityID1).
						Return(backend.Community{}, true, nil).Once()
					backenderMock.On("GetCommunityBan", mock.Anything, communityID1, channelID).
						Return(backend.CommunityBan{}, true, nil).Once()

					apitest.ExpectedStatus = http.StatusForbidden
					apitest.Expected = map[string]interface{}{
						"status": float64(http.StatusForbidden),
						"error":  api.ErrCodeChannelBannedFromCommunity,
					}
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})

				c.Convey("if the user is timed out in a community", func() {
					// backenderMock.On("GetCommunity", mock.Anything, communityID1).
					// 	Return(backend.Community{}, true, nil).Once()
					// backenderMock.On("GetCommunityBan", mock.Anything, communityID1, channelID).
					// 	Return(backend.CommunityBan{}, false, nil).Once()
					// backenderMock.On("GetCommunityTimeout", mock.Anything, communityID1, channelID).
					// 	Return(backend.CommunityTimeout{}, false, nil).Once()
					//
					// backenderMock.On("GetCommunity", mock.Anything, communityID2).
					// 	Return(backend.Community{}, true, nil).Once()
					// backenderMock.On("GetCommunityBan", mock.Anything, communityID2, channelID).
					// 	Return(backend.CommunityBan{}, false, nil).Once()
					// backenderMock.On("GetCommunityTimeout", mock.Anything, communityID2, channelID).
					// 	Return(backend.CommunityTimeout{}, true, nil).Once()

					// TODO: replace below .On()s with above .On()s
					backenderMock.On("GetCommunity", mock.Anything, communityID1).
						Return(backend.Community{}, true, nil).Once()
					backenderMock.On("GetCommunityBan", mock.Anything, communityID1, channelID).
						Return(backend.CommunityBan{}, false, nil).Once()
					backenderMock.On("GetCommunityTimeout", mock.Anything, communityID1, channelID).
						Return(backend.CommunityTimeout{}, true, nil).Once()

					apitest.ExpectedStatus = http.StatusForbidden
					apitest.Expected = map[string]interface{}{
						"status": float64(http.StatusForbidden),
						"error":  api.ErrCodeChannelTimedOutFromCommunity,
					}
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})
			})

			c.Convey("if the user does not have edit channel community permissions", func() {
				authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
					"manage_channel_community": goauthorization.CapabilityClaim{
						"community_ids": communityIDs,
						"channel_id":    channelID,
					},
				}).Return(errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusForbidden
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusForbidden),
					"error":  api.ErrCodeRequestingUserNotPermitted,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if auth token is not valid", func() {
			authMock.On("ParseToken", mock.Anything).
				Return(nil, errors.New("error"))

			apitest.ExpectedStatus = http.StatusUnauthorized
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusUnauthorized),
				"error":  api.ErrCodeRequestingUserNotPermitted,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestUnsetChannelCommunity(t *testing.T) {
	c.Convey("test UnsetChannelCommunity handler", t, func() {

		authMock := &mockauth.IDecoder{}
		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		channelID := "555"
		requester := "123"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/channels/communities/unset",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"channel_id": channelID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if auth token is valid", func() {
			authMock.On("ParseToken", mock.Anything).
				Return(&goauthorization.AuthorizationToken{
					Claims: goauthorization.TokenClaims{
						Sub: claim.Sub{
							Sub: requester,
						},
					},
				}, nil)

			c.Convey("if the user has edit channel community permissions", func() {
				authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
					"manage_channel_community": goauthorization.CapabilityClaim{
						"community_id": unsetChannelCommunityCapability,
						"channel_id":   channelID,
					},
				}).Return(nil).Once()

				c.Convey("if the unset channel community call succeeds", func() {
					backenderMock.On("SetChannelCommunities", mock.Anything, channelID, []string{}).
						Return(nil).Once()

					apitest.ExpectedStatus = http.StatusOK
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})

				c.Convey("if the unset channel community call errors", func() {
					backenderMock.On("SetChannelCommunities", mock.Anything, channelID, []string{}).
						Return(errors.New("error")).Once()

					apitest.ExpectedStatus = http.StatusInternalServerError
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})
			})

			c.Convey("if the user does not have edit channel community permissions", func() {
				authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
					"manage_channel_community": goauthorization.CapabilityClaim{
						"community_id": unsetChannelCommunityCapability,
						"channel_id":   channelID,
					},
				}).Return(errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusForbidden
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusForbidden),
					"error":  api.ErrCodeRequestingUserNotPermitted,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if auth token is not valid", func() {
			authMock.On("ParseToken", mock.Anything).
				Return(nil, errors.New("error"))

			apitest.ExpectedStatus = http.StatusUnauthorized
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusUnauthorized),
				"error":  api.ErrCodeRequestingUserNotPermitted,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestGetChannelCommunity(t *testing.T) {
	c.Convey("test GetChannelCommunity handler", t, func() {

		authMock := &mockauth.IDecoder{}
		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		communityID := "asdfasdf"
		channelID := "555"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/channels/communities/get",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"channel_id": channelID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if the get channel community call succeeds", func() {
			backenderMock.On("GetChannelCommunities", mock.Anything, channelID).
				Return([]string{communityID}, nil).Once()

			apitest.ExpectedStatus = http.StatusOK
			apitest.Expected = map[string]interface{}{
				"community_id":  communityID,
				"community_ids": []interface{}{communityID},
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the get channel community call succeeds but there is no community", func() {
			backenderMock.On("GetChannelCommunities", mock.Anything, channelID).
				Return([]string{}, nil).Once()

			apitest.ExpectedStatus = http.StatusOK
			apitest.Expected = map[string]interface{}{
				"community_id":  "",
				"community_ids": []interface{}{},
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the get channel community call errors", func() {
			backenderMock.On("GetChannelCommunities", mock.Anything, channelID).
				Return(nil, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestReportChannelCommunity(t *testing.T) {
	c.Convey("test ReportChannelCommunity handler", t, func() {

		authMock := &mockauth.IDecoder{}
		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		communityID := "asdfasdf"
		channelID := "555"
		requester := "123"
		description := "blah blah"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/channels/communities/report",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"channel_id":  channelID,
				"description": description,
			},
			ExpectedStatus: http.StatusOK,
		}
		c.Convey("if channel is in a community", func() {
			backenderMock.On("GetChannelCommunities", mock.Anything, channelID).
				Return([]string{communityID}, nil).Once()

			c.Convey("if community exists", func() {
				backenderMock.On("GetCommunity", mock.Anything, communityID).
					Return(backend.Community{}, true, nil).Once()

				c.Convey("if auth token is valid", func() {
					authMock.On("ParseToken", mock.Anything).
						Return(&goauthorization.AuthorizationToken{
							Claims: goauthorization.TokenClaims{
								Sub: claim.Sub{
									Sub: requester,
								},
							},
						}, nil)

					c.Convey("with successful response for publish report", func() {
						backenderMock.On("PublishChannelCommunityReport", mock.Anything, channelID, communityID, requester, description).
							Return(nil).Once()

						apitest.ExpectedStatus = http.StatusOK
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})

					c.Convey("with error response for publish report", func() {
						backenderMock.On("PublishChannelCommunityReport", mock.Anything, channelID, communityID, requester, description).
							Return(errors.New("error")).Once()

						apitest.ExpectedStatus = http.StatusInternalServerError
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})
				})

				c.Convey("if auth token is not valid", func() {
					authMock.On("ParseToken", mock.Anything).
						Return(nil, errors.New("error"))

					c.Convey("with successful response for publish report", func() {
						backenderMock.On("PublishChannelCommunityReport", mock.Anything, channelID, communityID, "", description).
							Return(nil).Once()

						apitest.ExpectedStatus = http.StatusOK
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})
				})
			})

			c.Convey("if community does not exist", func() {
				backenderMock.On("GetCommunity", mock.Anything, communityID).
					Return(backend.Community{}, false, nil).Once()

				apitest.ExpectedStatus = http.StatusNotFound
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusNotFound),
					"error":  api.ErrCodeCommunityIDNotFound,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})

			c.Convey("with error response for get community", func() {
				backenderMock.On("GetCommunity", mock.Anything, communityID).
					Return(backend.Community{}, false, errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusInternalServerError
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if channel is not in a community", func() {
			backenderMock.On("GetChannelCommunities", mock.Anything, channelID).
				Return([]string{}, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}
