package app

import (
	"net/http"
	"sort"
	"strconv"
	"strings"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"

	"golang.org/x/net/context"
)

func (h *handlers) CheckEmoticonsAccess(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	var params api.CheckEmoticonsAccessRequest
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	if params.UserID == "" {
		err := errx.New("user ID is empty")
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	// TODO: update GetEmoteSets to not check for staff bool since Mako doesn't require it anymore.
	emoticonSetIntIDs, err := h.Backend.GetEmoteSets(ctx, params.UserID, true)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	// doesNotExist tracks emoticons that don't exist.
	doesNotExist := []string{}
	// noAccess tracks emoticons the user has no access to.
	noAccess := []string{}

	// Convert to map for quick lookups for an emoticon set ID.
	emoticonSetIDsMap := map[string]struct{}{}
	for _, s := range emoticonSetIntIDs {
		emoticonSetIDsMap[strconv.Itoa(s)] = struct{}{}
	}

	for _, emoticonID := range params.EmoticonIDs {
		setID, ok := h.Backend.GetEmoticonSet(emoticonID)
		if !ok {
			doesNotExist = append(doesNotExist, emoticonID)
			continue
		}

		if _, hasAccess := emoticonSetIDsMap[setID]; !hasAccess {
			noAccess = append(noAccess, emoticonID)
		}
	}

	// If there were invalid emoticons, return a 422 Unprocessable Entity.
	if len(doesNotExist) > 0 {
		err := errx.New("invalid emoticon(s): " + strings.Join(doesNotExist, ", "))
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusUnprocessableEntity)
		return
	}

	emoticonSetIDs := []string{}
	for s := range emoticonSetIDsMap {
		emoticonSetIDs = append(emoticonSetIDs, s)
	}
	sort.Strings(emoticonSetIDs)
	gojiplus.ServeJSON(rw, req, api.CheckEmoticonsAccessResponse{
		UserEmoticonSetIDs: emoticonSetIDs,
		NoAccessIDs:        noAccess,
	})
}
