package app

import (
	"bytes"
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend/backendfakes"
	"github.com/stretchr/testify/assert"
)

func TestCheckEmoticonAccess(t *testing.T) {
	fakeBackender := &backendfakes.FakeBackender{}
	fakeBackender.GetEmoteSetsReturns([]int{2, 3}, nil)
	fakeBackender.GetEmoticonSetReturnsOnCall(0, "2", true)
	fakeBackender.GetEmoticonSetReturnsOnCall(1, "3", true)
	fakeBackender.GetEmoticonSetReturnsOnCall(2, "4", true)

	s, _ := New(Params{Backend: fakeBackender})
	ts := httptest.NewServer(s)
	defer ts.Close()

	b, _ := json.Marshal(api.CheckEmoticonsAccessRequest{
		UserID:      "111111",
		EmoticonIDs: []string{"10", "20", "30"},
	})
	req, _ := http.NewRequest("POST", ts.URL+"/v1/users/emoticons/check_access", bytes.NewReader(b))
	resp, _ := http.DefaultClient.Do(req)

	var actual api.CheckEmoticonsAccessResponse
	err := json.NewDecoder(resp.Body).Decode(&actual)
	assert.Nil(t, err)

	assert.Equal(t, http.StatusOK, resp.StatusCode)

	expected := api.CheckEmoticonsAccessResponse{
		UserEmoticonSetIDs: []string{"2", "3"},
		NoAccessIDs:        []string{"30"},
	}
	assert.Equal(t, expected, actual)
}

func TestCheckEmoticonAccessInvalidEmoticon(t *testing.T) {
	fakeBackender := &backendfakes.FakeBackender{}
	fakeBackender.GetEmoteSetsReturns([]int{2, 3}, nil)
	fakeBackender.GetEmoticonSetReturnsOnCall(0, "2", true)
	fakeBackender.GetEmoticonSetReturnsOnCall(1, "", false)
	fakeBackender.GetEmoticonSetReturnsOnCall(2, "", false)

	s, _ := New(Params{Backend: fakeBackender})
	ts := httptest.NewServer(s)
	defer ts.Close()

	b, _ := json.Marshal(api.CheckEmoticonsAccessRequest{
		UserID:      "111111",
		EmoticonIDs: []string{"10", "20", "30"},
	})
	req, _ := http.NewRequest("POST", ts.URL+"/v1/users/emoticons/check_access", bytes.NewReader(b))
	resp, _ := http.DefaultClient.Do(req)

	var actual struct {
		Message string `json:"message"`
	}
	err := json.NewDecoder(resp.Body).Decode(&actual)
	assert.Nil(t, err)

	assert.Equal(t, http.StatusUnprocessableEntity, resp.StatusCode)
	assert.Equal(t, "invalid emoticon(s): 20, 30", actual.Message)
}
