package app

import (
	"errors"
	"net/http"
	"regexp"
	"strings"
	"time"
	"unicode/utf8"

	"github.com/pborman/uuid"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/chat/zuma/markdown"
	"code.justin.tv/common/goauthorization"

	"golang.org/x/net/context"
)

const (
	defaultLanguage = "EN"
	noneString      = "none"

	minAccountAge = 90 * 24 * time.Hour // 90 days

	maxCommunityNameLen    = 25
	minCommunityNameLen    = 3
	maxDisplayNameLen      = 25
	minDisplayNameLen      = 3
	maxShortDescriptionLen = 160
	maxLongDescriptionLen  = 1572864 // 1.5 MB
	maxRulesLen            = 1572864
	maxCommunitiesPerUser  = 5

	minCombiningCharacters         = 13
	maxCombiningCharactersPercent  = 35
	maxCombiningCharactersGroups   = 5
	maxThaiNonspaceCharacterGroups = 1
)

var (
	validName                            = regexp.MustCompile(`^[a-zA-Z0-9]+[a-zA-Z0-9\.\-_~]*$`)
	combiningDiacraticalMarksRegex       = regexp.MustCompile("[\u0300-\u036F]|[\u1AB0-\u1AFF]|[\u1DC0-\u1DFF]|[\u20D0-\u20FF]|[\uFE20-\uFE2F]")
	combiningDiacraticalMarksGroupsRegex = regexp.MustCompile("([\u0300-\u036F]|[\u1AB0-\u1AFF]|[\u1DC0-\u1DFF]|[\u20D0-\u20FF]|[\uFE20-\uFE2F]){2,}")
	thaiNonSpaceCharactersGroupsRegex    = regexp.MustCompile("([\u0E4E]){3,}")
)

func (h *handlers) CreateCommunity(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.CreateCommunityRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateCreateCommunityParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"create_community_account_age": goauthorization.CapabilityClaim{},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeAccountTooYoung, http.StatusForbidden)
		return
	}

	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"create_community_email_verified": goauthorization.CapabilityClaim{},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeUnverifiedEmail, http.StatusForbidden)
		return
	}

	userID := token.GetSubject()

	lowercaseName := strings.ToLower(params.Name)

	// TEMPORARY: While visage doesn't send display name, just copy the name.
	if params.DisplayName == "" {
		params.DisplayName = params.Name
	}

	if !validName.MatchString(params.Name) || len(params.Name) > maxCommunityNameLen || len(params.Name) < minCommunityNameLen {
		serveError(rw, req, api.ErrCodeCommunityNameInvalid, http.StatusBadRequest)
		return
	}
	if len(params.DisplayName) > maxDisplayNameLen || len(params.DisplayName) < minDisplayNameLen {
		serveError(rw, req, api.ErrCodeDisplayNameInvalid, http.StatusBadRequest)
		return
	}
	if len(params.LongDescription) > maxLongDescriptionLen {
		serveError(rw, req, api.ErrCodeLongDescriptionTooLong, http.StatusBadRequest)
		return
	}
	if len(params.ShortDescription) > maxShortDescriptionLen {
		serveError(rw, req, api.ErrCodeShortDescriptionTooLong, http.StatusBadRequest)
		return
	}
	if len(params.Rules) > maxRulesLen {
		serveError(rw, req, api.ErrCodeRulesTooLong, http.StatusBadRequest)
		return
	}
	if params.Language != nil && !isValidLanguage(*params.Language) {
		serveError(rw, req, api.ErrCodeInvalidLanguage, http.StatusBadRequest)
		return
	}

	user, exists, err := h.Backend.GetSiteUser(ctx, userID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		logx.Error(ctx, "user not found", logx.Fields{
			"user_id": userID,
		})
		serveError(rw, req, api.ErrCodeRequestingUserNotFound, http.StatusBadRequest)
		return
	}

	// TODO: Move this check to a Cartman capability
	twoFA, err := h.Backend.TwoFactorEnabled(ctx, user.UserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !twoFA {
		// Two factor authentication isn't enabled
		serveError(rw, req, api.ErrCodeNoTwoFactor, http.StatusForbidden)
		return
	}

	isReserved, err := h.isReservedName(ctx, lowercaseName)
	isPartnerClaimingOwn := false
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if isReserved {
		if isPartner, err := h.Backend.IsPartner(ctx, user.UserID); err != nil {
			gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
			return
		} else if !(isPartner && lowercaseName == strings.ToLower(user.Login)) {
			// if the user is a not a partner claiming their own name
			serveError(rw, req, api.ErrCodeCommunityNameReserved, http.StatusConflict)
			return
		} else {
			isPartnerClaimingOwn = true
		}
	}

	_, exists, err = h.Backend.GetCommunityIDByName(ctx, lowercaseName)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if exists {
		serveError(rw, req, api.ErrCodeCommunityNameExists, http.StatusConflict)
		return
	}

	// error if owner already has 5 communities and isn't staff
	// TODO: Move this to a Cartman capability
	communityIDs, err := h.Backend.GetCommunityIDsByOwner(ctx, user.UserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if len(communityIDs) >= maxCommunitiesPerUser && !user.IsStaff {
		serveError(rw, req, api.ErrCodeOwnTooManyCommunities, http.StatusForbidden)
		return
	}

	longDescriptionHTML := markdown.ConvertMarkdown(params.LongDescription)
	rulesHTML := markdown.ConvertMarkdown(params.Rules)
	language := defaultLanguage
	if params.Language != nil {
		language = strings.ToUpper(*params.Language)
	}

	communityID := uuid.New()
	community := backend.Community{
		CommunityID:         communityID,
		Name:                lowercaseName,
		DisplayName:         params.DisplayName,
		LongDescription:     params.LongDescription,
		LongDescriptionHTML: longDescriptionHTML,
		ShortDescription:    params.ShortDescription,
		Rules:               params.Rules,
		RulesHTML:           rulesHTML,
		OwnerUserID:         user.UserID,
		Email:               user.Email,
		Language:            language,
		BannerImageName:     noneImageName,
		AvatarImageName:     noneImageName,
	}
	err = h.Backend.PutCommunity(ctx, community)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	// update community search index
	err = h.indexCommunity(ctx, community)
	if err != nil {
		logx.Error(ctx, err)
	}

	// If a partner has claimed their own community, remove it from the reserved
	// names list
	if isPartnerClaimingOwn {
		err = h.Backend.DeleteCommunityReservedName(ctx, community.Name)
		if err != nil {
			logx.Error(ctx, err)
		}
	}

	// track community_create_complete event to spade
	go func() {
		err := h.Backend.TrackEvent(context.Background(), "community_create_complete", map[string]interface{}{
			"community_id":   community.CommunityID,
			"community_name": community.Name,
			"login":          user.Login,
			"is_subadmin":    user.IsAdmin,
			"is_admin":       user.IsStaff,
		})
		if err != nil {
			logx.Error(ctx, err)
		}
	}()

	gojiplus.ServeJSON(rw, req, &api.CreateCommunityResponse{
		CommunityID: communityID,
	})
}

func validateCreateCommunityParams(params api.CreateCommunityRequest) error {
	if params.Name == "" {
		return errors.New("name must not be empty")
	}
	// TEMPORARY: only validate the display name if it is included in the request
	if params.DisplayName != "" {
		if err := validateCommunityDisplayName(params.DisplayName); err != nil {
			return err
		}
	}
	if params.ShortDescription == "" {
		return errors.New("short_description must not be empty")
	}
	if params.LongDescription == "" {
		return errors.New("long_description must not be empty")
	}
	if params.Rules == "" {
		return errors.New("rules must not be empty")
	}
	return nil
}

func (h *handlers) GetCommunityByName(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.GetCommunityByNameRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateGetCommunityByNameParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	communityID, exists, err := h.Backend.GetCommunityIDByName(ctx, params.Name)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityNameNotFound, http.StatusNotFound)
		return
	}

	gojiplus.ServeJSON(rw, req, &api.GetCommunityByNameResponse{
		CommunityID: communityID,
	})
}

func validateGetCommunityByNameParams(params api.GetCommunityByNameRequest) error {
	if params.Name == "" {
		return errors.New("name must not be empty")
	}
	return nil
}

func validateCommunityDisplayName(name string) error {
	if name == "" {
		return errors.New("display_name must not be empty")
	}

	// Zalgo detection stolen from Clue
	// https://git-aws.internal.justin.tv/chat/tmi/blob/c457be8/clue/logic/privmsg_enforce.go#L959-L993
	if utf8.RuneCountInString(name) >= minCombiningCharacters {
		return nil
	}

	numCombiningCharacters := len(combiningDiacraticalMarksRegex.FindAllString(name, -1))
	percentCombiningCharacters := float64(numCombiningCharacters) / float64(utf8.RuneCountInString(name)) * 100

	numCombiningCharactersGroups := len(combiningDiacraticalMarksGroupsRegex.FindAllString(name, -1))

	if numCombiningCharacters < minCombiningCharacters {
		return nil
	}

	if (percentCombiningCharacters > maxCombiningCharactersPercent) || (numCombiningCharactersGroups > maxCombiningCharactersGroups) {
		return errors.New("display_name has invalid characters")
	}
	return nil
}

func (h *handlers) indexCommunity(ctx context.Context, community backend.Community) error {
	return h.Backend.IndexCommunity(ctx, backend.IndexedCommunity{
		CommunityID:      community.CommunityID,
		Name:             community.Name,
		ShortDescription: community.ShortDescription,
		LongDescription:  community.LongDescription,
		Rules:            community.Rules,
		RulesHTML:        community.RulesHTML,
		AvatarImageURL:   h.communityImageURL(community.CommunityID, community.AvatarImageName, api.ImageTypeAvatar),
	})
}
