package app

import (
	"errors"
	"net/http"
	"testing"
	"time"

	c "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	"code.justin.tv/chat/golibs/testutils"
	"code.justin.tv/chat/zuma/app/api"
	mockauth "code.justin.tv/chat/zuma/app/mocks/auth"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/common/goauthorization"
	"code.justin.tv/common/jwt/claim"
)

func TestCreateCommunity(t *testing.T) {
	c.Convey("test CreateCommunity handler", t, func() {

		authMock := &mockauth.IDecoder{}

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})
		requester := "12345"
		name := "testasdf"
		displayName := "Test ASDF"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/create",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"name":              name,
				"display_name":      displayName,
				"long_description":  "blah blah",
				"short_description": "blah",
				"rules":             "heh",
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if the auth token parses", func() {
			authMock.On("ParseToken", mock.Anything).
				Return(&goauthorization.AuthorizationToken{
					Claims: goauthorization.TokenClaims{
						Sub: claim.Sub{
							Sub: requester,
						},
					},
				}, nil)

			c.Convey("if the user has account age permissions", func() {
				authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
					"create_community_account_age": goauthorization.CapabilityClaim{},
				}).Return(nil).Once()

				c.Convey("if the user has verified email permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"create_community_email_verified": goauthorization.CapabilityClaim{},
					}).Return(nil).Once()

					c.Convey("if all of the params are valid", func() {
						c.Convey("if the requester exists", func() {
							backenderMock.On("GetSiteUser", mock.Anything, requester).
								Return(backend.SiteUser{UserID: requester}, true, nil).Once()

							c.Convey("if the requester has 2FA enabled", func() {
								backenderMock.On("TwoFactorEnabled", mock.Anything, requester).
									Return(true, nil).Once()

								c.Convey("if the name is not reserved", func() {
									backenderMock.On("ListCommunityReservedNames", mock.Anything).
										Return([]string{}, nil).Once()

									c.Convey("if the name doesn't already exist", func() {
										backenderMock.On("GetCommunityIDByName", mock.Anything, name).
											Return("", false, nil).Once()

										c.Convey("if the user doesn't have 5 communities already", func() {
											backenderMock.On("GetCommunityIDsByOwner", mock.Anything, requester).
												Return([]string{"111", "222"}, nil).Once()

											c.Convey("if PutCommunity succeeds", func() {
												backenderMock.On("PutCommunity", mock.Anything, mock.Anything).
													Return(nil).Once()

												c.Convey("if IndexCommunity succeeds", func() {
													backenderMock.On("IndexCommunity", mock.Anything, mock.Anything).
														Return(nil).Once()

													backenderMock.On("TrackEvent", mock.Anything, "community_create_complete", mock.Anything).
														Return(nil).Once()

													apitest.ExpectedStatus = http.StatusOK
													testutils.RunTest(t, s, apitest)
													time.Sleep(50 * time.Millisecond)
													backenderMock.AssertExpectations(t)
												})
											})
										})
									})
								})
							})
						})

						c.Convey("if the requester is making their own name", func() {
							backenderMock.On("GetSiteUser", mock.Anything, requester).
								Return(backend.SiteUser{UserID: requester, Login: name}, true, nil).Once()

							c.Convey("if the requester has 2FA enabled", func() {
								backenderMock.On("TwoFactorEnabled", mock.Anything, requester).
									Return(true, nil).Once()

								c.Convey("if the name is reserved", func() {
									backenderMock.On("ListCommunityReservedNames", mock.Anything).
										Return([]string{name}, nil).Once()

									c.Convey("if the user is a partner claiming their own name", func() {
										backenderMock.On("IsPartner", mock.Anything, requester).
											Return(true, nil).Once()

										c.Convey("if the name doesn't already exist", func() {
											backenderMock.On("GetCommunityIDByName", mock.Anything, name).
												Return("", false, nil).Once()

											c.Convey("if the user doesn't have 5 communities already", func() {
												backenderMock.On("GetCommunityIDsByOwner", mock.Anything, requester).
													Return([]string{"111", "222"}, nil).Once()

												c.Convey("if PutCommunity succeeds", func() {
													backenderMock.On("PutCommunity", mock.Anything, mock.Anything).
														Return(nil).Once()

													c.Convey("if IndexCommunity succeeds", func() {
														backenderMock.On("IndexCommunity", mock.Anything, mock.Anything).
															Return(nil).Once()

														backenderMock.On("TrackEvent", mock.Anything, "community_create_complete", mock.Anything).
															Return(nil).Once()
														backenderMock.On("DeleteCommunityReservedName", mock.Anything, name).
															Return(nil).Once()

														apitest.ExpectedStatus = http.StatusOK
														testutils.RunTest(t, s, apitest)
														time.Sleep(50 * time.Millisecond)
														backenderMock.AssertExpectations(t)
													})
												})
											})
										})
									})
								})
							})
						})
					})

				})
			})
		})

		c.Convey("if the token does not parse", func() {
			authMock.On("ParseToken", mock.Anything).
				Return(nil, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusUnauthorized
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusUnauthorized),
				"error":  api.ErrCodeRequestingUserNotPermitted,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestGetCommunityByName(t *testing.T) {
	c.Convey("test GetCommunityByName handler", t, func() {

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend: backenderMock,
		})
		name := "testasdf"
		id := "123123"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/get_by_name",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"name": name,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if the name exists", func() {
			backenderMock.On("GetCommunityIDByName", mock.Anything, name).
				Return(id, true, nil).Once()

			apitest.Expected = map[string]interface{}{
				"community_id": id,
			}
			apitest.ExpectedStatus = http.StatusOK
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the name doesn't exist", func() {
			backenderMock.On("GetCommunityIDByName", mock.Anything, name).
				Return("", false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityNameNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if there's an error", func() {
			backenderMock.On("GetCommunityIDByName", mock.Anything, name).
				Return("", false, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}
