package app

import (
	"errors"
	"net/http"
	"strconv"
	"time"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/common/goauthorization"

	"golang.org/x/net/context"
)

const defaultBanReason = "default"
const maxLimit = 100
const defaultLimit = 10

func (h *handlers) GetCommunityBan(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.GetCommunityBanRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateGetCommunityBanParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"view_community_moderation": goauthorization.CapabilityClaim{
			"community_id": params.CommunityID,
		},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}

	ban, exists, err := h.Backend.GetCommunityBan(ctx, params.CommunityID, params.UserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	gojiplus.ServeJSON(rw, req, &api.GetCommunityBanResponse{
		IsBanned:  exists,
		Start:     time.Unix(0, ban.StartTSUnixNano).UTC(),
		ModUserID: ban.ModUserID,
		Reason:    ban.Reason,
	})
}

func validateGetCommunityBanParams(params api.GetCommunityBanRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	if userID, err := strconv.ParseInt(params.UserID, 10, 64); err != nil {
		return errors.New("user_id must be integer")
	} else if userID <= 0 {
		return errors.New("user_id must be positive")
	}
	return nil
}

func (h *handlers) ListCommunityBans(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.ListCommunityBansRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateListCommunityBansParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"view_community_moderation": goauthorization.CapabilityClaim{
			"community_id": params.CommunityID,
		},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}

	// specify cursor
	var cursor int64
	if params.Cursor == "" {
		cursor = 0
	} else {
		parsed, err := strconv.ParseInt(params.Cursor, 10, 64)
		if err != nil {
			gojiplus.ServeError(ctx, rw, req, errors.New("invalid cursor"), http.StatusBadRequest)
			return
		}
		cursor = parsed
	}

	// specify limit
	limit := params.Limit
	if limit <= 0 {
		limit = defaultLimit
	} else if limit > maxLimit {
		limit = maxLimit
	}

	bans, err := h.Backend.ListCommunityBans(ctx, params.CommunityID, cursor, limit)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	resp := &api.ListCommunityBansResponse{
		Bans: []api.CommunityBan{},
	}
	for _, ban := range bans {
		resp.Bans = append(resp.Bans, api.CommunityBan{
			UserID:    ban.UserID,
			Start:     time.Unix(0, ban.StartTSUnixNano).UTC(),
			ModUserID: ban.ModUserID,
			Reason:    ban.Reason,
		})
		resp.Cursor = strconv.FormatInt(ban.StartTSUnixNano, 10)
	}

	gojiplus.ServeJSON(rw, req, resp)
}

func validateListCommunityBansParams(params api.ListCommunityBansRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	if params.Cursor != "" {
		if cursor, err := strconv.ParseInt(params.Cursor, 10, 64); err != nil || cursor < 0 {
			return errors.New("invalid cursor")
		}
	}

	return nil
}

func (h *handlers) AddCommunityBan(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.AddCommunityBanRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateAddCommunityBanParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	reason := params.Reason
	if params.Reason == "" {
		reason = defaultBanReason
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"moderate_community": goauthorization.CapabilityClaim{
			"community_id": params.CommunityID,
			"target_id":    params.TargetUserID,
		},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}

	userID := token.GetSubject()

	if userID == params.TargetUserID {
		serveError(rw, req, api.ErrCodeTargetUserSelf, http.StatusBadRequest)
		return
	}

	if params.TargetUserID == community.OwnerUserID {
		serveError(rw, req, api.ErrCodeTargetUserOwner, http.StatusForbidden)
		return
	}

	// business logic forbidding adding bans
	targetUser, exists, err := h.Backend.GetSiteUser(ctx, params.TargetUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeTargetUserNotFound, http.StatusBadRequest)
		return
	}
	if targetUser.IsStaff {
		serveError(rw, req, api.ErrCodeTargetUserStaff, http.StatusForbidden)
		return
	}

	targetIsMod, err := h.Backend.IsCommunityMod(ctx, params.CommunityID, params.TargetUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if targetIsMod {
		serveError(rw, req, api.ErrCodeTargetUserMod, http.StatusForbidden)
		return
	}

	// noop if target is already banned in channel
	_, exists, err = h.Backend.GetCommunityBan(ctx, params.CommunityID, params.TargetUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if exists {
		// noop
		rw.WriteHeader(http.StatusOK)
		return
	}

	// if user is timed out, replace with a full ban
	_, exists, err = h.Backend.GetCommunityTimeout(ctx, params.CommunityID, params.TargetUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if exists {
		err = h.Backend.RemoveCommunityTimeout(ctx, params.CommunityID, params.TargetUserID)
		if err != nil {
			gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
			return
		}
	}

	startTime := time.Now()
	err = h.Backend.AddCommunityBan(ctx, backend.CommunityModerationAction{
		CommunityID:  params.CommunityID,
		ModUserID:    userID,
		TargetUserID: params.TargetUserID,
		Reason:       reason,
		Type:         "ban",
		StartTime:    startTime,
	})
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	currentCommunities, err := h.Backend.GetChannelCommunities(ctx, params.TargetUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if contains(currentCommunities, params.CommunityID) {
		// currently broadcasting to this community - unset them
		err = h.Backend.SetChannelCommunities(ctx, params.TargetUserID, remove(currentCommunities, params.CommunityID))
		if err != nil {
			gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
			return
		}
	}

	if err := h.Backend.AddCommunityModerationLog(ctx, backend.CommunityModerationAction{
		CommunityID:  community.CommunityID,
		ModUserID:    userID,
		TargetUserID: params.TargetUserID,
		Type:         "ban",
		StartTime:    startTime,
		Reason:       params.Reason,
	}); err != nil {
		logx.Error(ctx, err)
	}

	recentlyBroadcast, err := h.Backend.ChannelRecentlyBroadcastToCommunity(ctx, params.TargetUserID, params.CommunityID)
	if err != nil {
		logx.Error(ctx, err)
	} else if recentlyBroadcast {
		err = h.Backend.PublishCommunityModerationAction(ctx, params.CommunityID, userID, params.TargetUserID, api.ModActionTypeBan, reason)
		if err != nil {
			logx.Error(ctx, err)
		}
	}

	rw.WriteHeader(http.StatusOK)
}

func validateAddCommunityBanParams(params api.AddCommunityBanRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	if targetUserID, err := strconv.ParseInt(params.TargetUserID, 10, 64); err != nil {
		return errors.New("target_user_id must be integer")
	} else if targetUserID <= 0 {
		return errors.New("target_user_id must be positive")
	}
	return nil
}

func (h *handlers) RemoveCommunityBan(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.RemoveCommunityBanRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateRemoveCommunityBanParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"moderate_community": goauthorization.CapabilityClaim{
			"community_id": params.CommunityID,
			"target_id":    params.TargetUserID,
		},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}

	// noop if target is not already banned
	_, exists, err = h.Backend.GetCommunityBan(ctx, params.CommunityID, params.TargetUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		// noop
		rw.WriteHeader(http.StatusOK)
		return
	}

	err = h.Backend.RemoveCommunityBan(ctx, params.CommunityID, params.TargetUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	startTime := time.Now()
	userID := token.GetSubject()
	if err := h.Backend.AddCommunityModerationLog(ctx, backend.CommunityModerationAction{
		CommunityID:  community.CommunityID,
		ModUserID:    userID,
		TargetUserID: params.TargetUserID,
		Type:         "unban",
		StartTime:    startTime,
	}); err != nil {
		logx.Error(ctx, err)
	}

	rw.WriteHeader(http.StatusOK)
}

func validateRemoveCommunityBanParams(params api.RemoveCommunityBanRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	if targetUserID, err := strconv.ParseInt(params.TargetUserID, 10, 64); err != nil {
		return errors.New("target_user_id must be integer")
	} else if targetUserID <= 0 {
		return errors.New("target_user_id must be positive")
	}
	return nil
}
